/*
 * This file is part of the Liquid project.
 * It realizes a UI Module of project. It provides all
 * GUI elements of application (panels, workspace, menu and etc.).
 *
 * Copyright (C) 2009 Kirpichonock K.N. <kirpiche@cs.karelia.ru>
 * Copyright (C) 2009 Volkov A.A. <volkov@cs.karelia.ru>
 * Copyright (C) 2009 Dmitriev V.V. <vdmitrie@cs.karelia.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "error_handler/errors.h"

#include <math.h>
#include <QColor>
#include <QColorDialog>
#include <QFileDialog>
#include <QDialog>
#include <QMessageBox>
#include <QTouchEvent>

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);    

    errorHandler = new ErrorHandler(this);
    painter = new Painter(this, errorHandler);

    ui->horizontalLayout->addWidget(painter);       
    m_fileName = QString::null;

    createActions();
    createMenus();

    this->setStyleSheet("QToolBar {background: black; spacing: 1px}");
    ui->toolBar->addAction(brushAct);
    ui->toolBar->addAction(colorToolAct);
    ui->toolBar->addAction(eraserAct);
    ui->toolBar->addAction(iceAct);    
    ui->toolBar->addAction(fullScreenAct);    

    connect(errorHandler, SIGNAL(accFatalError()), SLOT(disableAccelerometer()));
    connect(painter, SIGNAL(disableAccelerometer()), SLOT(disableAccelerometer()));
    connect(painter, SIGNAL(activateAccelerometer()), SLOT(activateAccelerometer()));
}

MainWindow::~MainWindow()
{
    delete ui;
}

// Creates all actions for Menus and Tools panel
void MainWindow::createActions() {
    newAct = new QAction(tr("&New"), this);
    newAct->setShortcut(QKeySequence::New);
    connect(newAct, SIGNAL(triggered()), this, SLOT(newImage()));

    openAct = new QAction(tr("&Open..."), this);
    openAct->setShortcuts(QKeySequence::Open);
    connect(openAct, SIGNAL(triggered()), this, SLOT(open()));

    saveAct = new QAction(tr("Save"), this);
    saveAct->setShortcuts(QKeySequence::Save);
    connect(saveAct, SIGNAL(triggered()), this, SLOT(save()));

    saveAsAct = new QAction(tr("&Save As..."), this);
    saveAsAct->setShortcuts(QKeySequence::SaveAs);
    connect(saveAsAct, SIGNAL(triggered()), this, SLOT(saveAs()));

    exitAct = new QAction(tr("E&xit"), this);
    exitAct->setShortcuts(QKeySequence::Close);
    connect(exitAct, SIGNAL(triggered()), this, SLOT(close()));

    brushAct = new QAction(tr("Brush"), this);
    brushAct->setIcon(QIcon(":/brush.png"));
    connect(brushAct, SIGNAL(triggered()), SLOT(setBrush()));

    colorToolAct = new QAction(tr("Color Tool"), this);
    colorToolAct->setIcon(QIcon(":/palette.png"));
    connect(colorToolAct, SIGNAL(triggered()), SLOT(showColorDialog()));

    eraserAct = new QAction(tr("Eraser"), this);
    eraserAct->setIcon(QIcon(":/drop.png"));
    connect(eraserAct, SIGNAL(triggered()), SLOT(setEraser()));

    iceAct = new QAction(tr("Ice"), this);
    iceAct->setIcon(QIcon(":/ice.png"));
    connect(iceAct, SIGNAL(triggered()), SLOT(setIce()));

    fullScreenAct = new QAction(tr("Full Screen"), this);
    fullScreenAct->setIcon(QIcon(":/fullscreen.png"));
    fullScreenAct->setCheckable(true);
    connect(fullScreenAct, SIGNAL(triggered(bool)), SLOT(fullScreen(bool)));

    toggleAcc = new QAction(tr("Accelerometer"), this);
    toggleAcc->setCheckable(true);
    toggleAcc->setChecked(true);
    connect(toggleAcc, SIGNAL(toggled(bool)), SLOT(toggleAccelerometer(bool)));

    aboutAct = new QAction(tr("About program"), this);
    connect(aboutAct, SIGNAL(triggered()), SLOT(showAbout()));
}

// Creates menus of application
void MainWindow::createMenus() {
    fileMenu = ui->menubar->addMenu(tr("&File"));
    fileMenu->addAction(newAct);
    fileMenu->addAction(openAct);
    fileMenu->addAction(saveAct);
    fileMenu->addAction(saveAsAct);
    fileMenu->addSeparator();
    fileMenu->addAction(exitAct);

    modeMenu = ui->menubar->addMenu(tr("&Mode"));
    modeMenu->addAction(toggleAcc);    

    aboutMenu = ui->menubar->addMenu(tr("&About"));
    aboutMenu->addAction(aboutAct);
}

// Shows dilog with information about application and developers
void MainWindow::showAbout() {
        QMessageBox::about(this, tr("About program"), tr(
                "<div style='y-overflow:scroll;'>"
                "<strong>Version: %1</strong><br>"
                "<strong>License: GPLv2</strong><br>"
                "<strong>Homepage:</strong> <a href='http://liquid-graphic.garage.maemo.org'>link</a>
                "<p>The <strong>%2</strong> is a graphic editor for Nokia's tablet. <br>"
                "In our application user can controll a <br>"
                "Brush, using accelerometer or touchscreen of a tablet. <br><br>"
                "<strong>Developers: </strong>"
                "<table cellpadding=2 cellspacing=1>"
                "<tr><td>Volkov A.A.</td> <td> (volkov@cs.karelia.ru)</td></tr>"
                "<tr><td>Dmitriev V.V.</td> <td> (vdmitrie@cs.karelia.ru)</td></tr>"
                "<tr><td>Kirpichonock K.N.</td><td> (kirpiche@cs.karelia.ru)</td></tr>"
                "</table></p></div>"
                ).arg(QCoreApplication::applicationVersion(), QCoreApplication::applicationName()));
}

// Creates new image
void MainWindow::newImage() {
    m_fileName = QString::null;
    painter->createImage();
}

// Opens local graphic file
void MainWindow::open()
{
    const QString fileName = QFileDialog::getOpenFileName(this,
                                                          tr("Open File"),
                                                          QDir::currentPath(),
                                                          "Images: (*.png *.jpeg *.jpg)");
    if (!fileName.isEmpty()) {
        if (!painter->openImage(fileName)) {
            QMessageBox::information(this, tr("Pearl"),
                                     tr("Cannot load %1.").arg(fileName));
            return;
        }
        painter->adjustSize();
    }
    m_fileName = fileName;
}

// Saves file with current name or shows Save As Dialog
void MainWindow::save() {
    if(m_fileName.isEmpty())
        saveAs();
    else
        if(!painter->saveImage(m_fileName, "png")) {
            errorHandler->returnError(UI_SAVEFILE_ERROR);
        }
}

// Shows Save As Dialog
bool MainWindow::saveAs() {
    const QString initialPath = QDir::currentPath() + "/untitled.png";
    const QString fileName = QFileDialog::getSaveFileName(this, tr("Save As"),
                                                          initialPath);
    m_fileName = fileName;
    if (fileName.isEmpty()) {
        return false;
    } else {
        return painter->saveImage(fileName, "png");
    }
}

// Shows exit dialog (Save, Discard, Canchel), when user close application
bool MainWindow::showExitDialog() {
     QMessageBox msgBox;
     msgBox.setText("The document has been modified.");
     msgBox.setInformativeText("Do you want to save your changes?");
     msgBox.setStandardButtons(QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel);
     msgBox.setDefaultButton(QMessageBox::Save);

     int ret = msgBox.exec();

     switch (ret) {
        case QMessageBox::Save:
            if(!saveAs())
                return false;
            return true;            
        case QMessageBox::Discard:
            return true;
            break;
        case QMessageBox::Cancel:
            return false;
            break;
        default:            
            break;
     }
     return false;
}

// Handler for close signal, shows exit dialog
void MainWindow::closeEvent(QCloseEvent *e){    
    if(m_fileName.isEmpty() || painter->isModified) {
        if(showExitDialog())
            e->accept();
        else
            e->ignore();
    } else
        e->accept();
}

// Activate or disable full screen mode
void MainWindow::fullScreen(bool makeFullScreen) {
    if(makeFullScreen) {
        painter->setFullHeight(true);
        showFullScreen();
    } else {
        painter->setFullHeight(false);
        showNormal();
    }
}

// Disables accelerometer mode
void MainWindow::disableAccelerometer() {
    toggleAcc->setChecked(false);
}

// Activates Accelerometer mode
void MainWindow::activateAccelerometer() {
    toggleAcc->setChecked(true);
}

// Handler for Menu Mode->Accelerometer entry
void MainWindow::toggleAccelerometer(bool isChecked) {
    if(isChecked) {
        if(!painter->setAccMode(true))
            errorHandler->returnError(ACC_TURNON_ERROR);
    } else {
        if(!painter->setAccMode(false))
            errorHandler->returnError(ACC_TURNOFF_ERROR);
    }
}

// Handler for Brush entry in tools panel
void MainWindow::setBrush() {
    painter->setTool(BRUSH);    
}

// Handler for Drop entry in tools panel
void MainWindow::setEraser() {
    painter->setTool(DROP);    
}

// Handler for Ice entry in tools panel
void MainWindow::setIce() {
    painter->setTool(ICE);    
}

// Shows palette
void MainWindow::showColorDialog() {
    QColor color = QColorDialog::getColor(Qt::green, this);
    if (color.isValid()) {
         painter->setColor(color);         
    }
}

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}
