/*
 * This file is part of the Liquid project.
 * It reads and processes data from accelerometer. After data
 * is processed, it generates a signal, which Accelerometer class handles.
 *
 * Copyright (C) 2009 Kirpichonock K.N. <kirpiche@cs.karelia.ru>
 * Copyright (C) 2009 Volkov A.A. <volkov@cs.karelia.ru>
 * Copyright (C) 2009 Dmitriev V.V. <vdmitrie@cs.karelia.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include "accelerometer/devorientation.h"
#include <math.h>
#include <error_handler/errors.h>
#include <QDebug>
#include <QTimer>
#include <QFile>
#include <QTextStream>
#include <QStringList>

// Constructor
devOrientation::devOrientation(ErrorHandler *errorHandler)
{
    eh = new ErrorHandler;
    eh = errorHandler;
    m_timer = new QTimer;
    m_counter = 0;    
    connect(m_timer, SIGNAL(timeout()), SLOT(getXYZ()));
    m_timer->start(10);
}

// Gets acceleration from accelerometer
void devOrientation::getXYZ() {
    QFile file("/sys/class/i2c-adapter/i2c-3/3-001d/coord");

    if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
        eh->returnError(ACC_OPEN_ERROR);
        return;
    }
    QTextStream in(&file);

    data = in.readAll();

    process_data();
}

// Disables reading data from accelerometer
bool devOrientation::stopTimer(bool stop) {
    if(stop)
        return disconnect(m_timer, SIGNAL(timeout()), this, SLOT(getXYZ()));
    else
        return connect(m_timer, SIGNAL(timeout()), SLOT(getXYZ()));
}

// Processes data, which we get from sensor
void devOrientation::process_data() {
    QStringList data_splited = data.split(" ");   

    x = data_splited[0];
    y = data_splited[1];
    z = data_splited[2];

    float fx = x.toFloat() / 100;
    float fy = y.toFloat() / 100;
    float fz = z.toFloat() / 100;

    QVector3D accVector3 = QVector3D(-fx, -fy, -fz);
    emit dataProcessed(accVector3);
}


// Stub for sensor, it generates acceleration for testing
QString devOrientation::generateCoords() {
    QString coord;

    if (m_counter < 200) {
        coord = "0 100 -1000";
        m_counter++;
        return coord;
    }

    if (m_counter >= 200 && m_counter < 600 ) {
        coord = "0 -100 -1000";
        m_counter++;
        return coord;
    }

    if (m_counter >= 600 && m_counter < 1000) {
        coord = "0 100 -1000";
        m_counter++;
        return coord;
    }

    if (m_counter >= 1000 && m_counter < 1400 ) {
        coord = "0 -100 -1000";
        m_counter++;
        return coord;
    } else {
        m_counter = 0;
        return "0 0 -1000";
    }
}
