/*
 * Licensed under BSD license.  See LICENCE.TXT  
 *
 * Produced by:	Jeff Lait
 *
 *      	7DRL Development
 *
 * NAME:        ai.cpp ( Save Scummer Library, C++ )
 *
 * COMMENTS:
 */

#include "mob.h"

#include "map.h"
#include "msg.h"
#include "text.h"
#include "speed.h"
#include "item.h"

bool
MOB::aiAvoidDirection(int dx, int dy) const
{
    POS	qp = pos().delta(dx, dy);
    ITEMLIST	items;

    if (qp.allItems(items))
    {
	for (int i = 0; i < items.entries(); i++)
	{
	    if (items(i)->getDefinition() == ITEM_CORPSE)
		return true;
	}
    }
    return false;
}

bool
MOB::aiForcedAction()
{
    // Check for the phase...
    PHASE_NAMES		phase;

    phase = spd_getphase();

    if (!alive())
	return false;

    myYellHystersis--;
    if (myYellHystersis < 0)
	myYellHystersis = 0;

    myBoredom++;
    if (myBoredom > 1000)
	myBoredom = 1000;

    // On normal & slow phases, items get heartbeat.
    if (phase == PHASE_NORMAL || phase == PHASE_SLOW)
    {
	// We don't want magic rings in other people's pockets
	// to expire.
	if (getDefinition() == MOB_AVATAR)
	{
	    int i;
	    for (i = myInventory.entries(); i --> 0;)
	    {
		if (myInventory(i)->runHeartbeat())
		{
		    ITEM		*item = myInventory(i);
		    removeItem(item);
		    delete item;
		}
	    }
	}

	// Regenerate if we are capable of it
	if (defn().isregenerate || hasItem(ITEM_REGENRING))
	{
	    gainHP(1);
	}

	// Apply damage if inside.
	if (isSwallowed())
	{
	    msg_format("%S <be> digested!", this);

	    // Make it a force action if it kills us.
	    if (applyDamage(0, rand_range(1, 4)))
		return true;
	}
    }

    if (isMeditating())
    {
	// People meditating move very fast.
	return false;
    }

    switch (phase)
    {
	case PHASE_FAST:
	    if (hasItem(ITEM_SPEEDBOOTS))
		break;
	    
	    // Not fast, no phase
	    if (defn().isfast)
		break;
	    return true;

	case PHASE_QUICK:
	    // Not quick, no phase!
	    if (hasItem(ITEM_QUICKBOOST))
		break;
	    return true;

	case PHASE_SLOW:
	    if (defn().isslow)
		return true;
	    if (hasItem(ITEM_SLOW))
		return true;
	    break;

	case PHASE_NORMAL:
	    break;
    }
    if (mySkipNextTurn)
    {
	mySkipNextTurn = false;
	return true;
    }
    return false;
}

bool
MOB::aiDoAI()
{
    return aiDoAIType(getAI());
}

bool
MOB::aiDoAIType(AI_NAMES aitype)
{
    // Rebuild our home location.
    if (!myHome.valid())
    {
	myHome = pos();
    }

    if (myFleeCount > 0)
    {
	myFleeCount--;
	if (!isAvatar() && aiFleeFromAvatar())
	    return true;
    }
    switch (aitype)
    {
	case AI_NONE:
	    // Just stay put!
	    return true;

	case AI_STAYHOME:
	    // If we are at home stay put.
	    if (pos() == myHome)
		return true;

	    // Otherwise, go home.
	    // FALL THROUGH

	case AI_HOME:
	    // Move towards our home square.
	    int		dist;

	    dist = pos().dist(myHome);

	    // If we can shoot the avatar or charge him, do so.
	    if (aiRangeAttack())
		return true;
	    if (aiCharge(getAvatar(), AI_CHARGE))
		return true;

	    // A good chance to just stay put.
	    if (rand_chance(70))
	    {
		return actionBump(0, 0);
	    }

	    if (rand_choice(dist))
	    {
		// Try to home.
		if (aiMoveTo(myHome))
		    return true;
	    }
	    // Default to wandering
	    return aiRandomWalk();

	case AI_ORTHO:
	    if (aiRangeAttack())
		return true;
	    if (aiCharge(getAvatar(), AI_CHARGE, true))
		return true;
	    // Default to wandering
	    return aiRandomWalk(true);

	case AI_CHARGE:
	    if (aiRangeAttack())
		return true;
	    if (aiCharge(getAvatar(), AI_CHARGE))
		return true;
	    // Default to wandering
	    return aiRandomWalk();

	case AI_PATHTO:
	    if (aiKillPathTo(getAvatar()))
		return true;
		    
	    // Default to wandering
	    return aiRandomWalk();

	case AI_FLANK:
	    if (aiRangeAttack())
		return true;
	    if (aiCharge(getAvatar(), AI_FLANK))
		return true;
	    // Default to wandering
	    return aiRandomWalk();

	case AI_RANGECOWARD:
	    // If we can see avatar, shoot at them!
	    // We keep ourselves at range if possible, unless
	    // retreat is blocked, in which case we will melee.
	    if (pos().isFOV())
	    {
		int		 dist, dx, dy;

		if (aiAcquireAvatar())
		{
		    pos().dirTo(myTarget, dx, dy);
		    dist = pos().dist(myTarget);

		    if (dist == 1)
		    {
			// We are in melee range.  Try to flee.
			if (aiFleeFrom(myTarget))
			    return true;

			// Failed to flee.  Kill!
			return actionMelee(dx, dy);
		    }

		    // Try ranged attack.
		    if (aiRangeAttack())
			return true;

		    // Failed range attack.  If outside of range, charge.
		    if (dist > getRangedRange())
		    {
			if (aiCharge(getAvatar(), AI_FLANK))
			    return true;
		    }

		    // Otherwise, wander within the range hoping to line up.
		    // Trying to double think the human is likely pointless
		    // as they'll be trying to avoid lining up to.
		    return aiRandomWalk();
		}
	    }
		
	    if (aiCharge(getAvatar(), AI_FLANK))
		return true;

	    // Default to wandering
	    return aiRandomWalk();

	    
	case AI_COWARD:
	    if (aiRangeAttack())
		return true;

	    if (aiFleeFromAvatar())
		return true;
	    // Failed to flee, cornered, so charge!
	    if (aiCharge(getAvatar(), AI_CHARGE))
		return true;

	    return aiRandomWalk();

	case AI_MOUSE:
	    return aiDoMouse();

	case AI_STRAIGHTLINE:
	    return aiStraightLine();

	case AI_SMARTKOBOLD:
	    return aiStrategy();
    }

    return false;
}

bool
MOB::aiBattlePrep()
{
    for (int i = 0; i < myInventory.entries(); i++)
    {
	ITEM		*item = myInventory(i);

	if (item->isPotion())
	{
	    return actionQuaff(item);
	}
    }
    return false;
}

bool
MOB::aiDestroySomethingGood(MOB *denyee)
{
    ITEM		*wand = lookupWand();
    ITEM		*enemywand = denyee->lookupWand();

    if (wand)
    {
	// Only bother wasting our turn destroying it if the
	// avatar hasn't yet acquired one.
	if (aiLikeMoreWand(wand, enemywand) == wand)
	{
	    return actionBreak(wand);
	}
    }
    // Deal with any left over potions
    return aiBattlePrep();
}

bool
MOB::aiTwitch(MOB *avatarvis)
{
    // Did we see a murder?
    if (mySawMurder)
    {
	mySawMurder = false;
	if (myStrategy == STRATEGY_SAMEROOM)
	{
	    // End of this strategy!
	    myStrategy = STRATEGY_FETCHITEMS;
	    return actionYell(YELL_MURDERER);
	}
    }

    if (mySawMeanMurder)
    {
	mySawMeanMurder = false;
	myStrategy = STRATEGY_KILLCHASE;
	return actionYell(YELL_KILLCHASE);
    }

    // Process any yells we received.
    YELL_NAMES		yell;

    FOREACH_YELL(yell)
    {
	if (myHeardYell[yell])
	{
	    // Processed it.
	    myHeardYell[yell] = false;
	    bool		sameroom = myHeardYellSameRoom[yell];
	    myHeardYellSameRoom[yell] = false;

	    switch (yell)
	    {
		case YELL_KEEPOUT:
		    // Whatever.
		    break;

		case YELL_MURDERER:
		case YELL_INVADER:
		    // If we already are alerted, ignore.
		    // otherwise we upgrade our strategy and yell.
		    // We always want to shout this even if someone
		    // else already shouted it.
		    if (myStrategy == STRATEGY_SAMEROOM)
		    {
			myStrategy = STRATEGY_FETCHITEMS;
			return actionYell(YELL_INVADER);
		    }
		    break;

		case YELL_RANGED:
		    if (!myAvatarHasRanged)
		    {
			myAvatarHasRanged = true;
			if (!sameroom || !rand_chance(10))
			    return actionYell(YELL_RANGED);
		    }
		    break;

		case YELL_KILL:
		    // End of the trap type..
		    if (myStrategy != STRATEGY_KILL && myStrategy != STRATEGY_KILLCHASE)
		    {
			myStrategy = STRATEGY_KILL;
			if (!sameroom || !rand_chance(10))
			    return actionYell(YELL_KILL);
		    }
		    break;

		case YELL_KILLCHASE:
		    // Kobold rage.
		    if (myStrategy != STRATEGY_KILLCHASE)
		    {
			myStrategy = STRATEGY_KILLCHASE;
			if (!sameroom || !rand_chance(10))
			    return actionYell(YELL_KILLCHASE);
		    }
		    break;


		case YELL_VICTORY:
		    // Time to party!
		    if (myStrategy != STRATEGY_PARTY)
		    {
			myStrategy = STRATEGY_PARTY;
			if (!sameroom || rand_chance(10))
			    return actionYell(YELL_VICTORY);
		    }
		    break;

		case YELL_LOCATION:
		case YELL_HEARDLOCATION:
		    if (myYellHystersis <= 0)
		    {
			// If we already see the avatar, we don't care.
			// If the shout came from our room, we don't
			// need to continue it.
			if (!avatarvis && !sameroom)
			{
			    // We heard through the grapevine.  Technically
			    // this is too recent a location, but close enough
			    // for this purpose.
			    if (getAvatar())
				myTarget = getAvatar()->pos();

			    // Because we will hear an echo from adjoining
			    // rooms we don't want to keep reechoing
			    // forever, stalling all movement!
			    // Most other yells force a state transition
			    // so avoid this.
			    myYellHystersis = 15;

			    return actionYell(YELL_HEARDLOCATION);
			}
		    }
		    break;
	    }
	}
    }

    // Run this after the yells so everyone who sees doesn't spam.
    // Did we see our victory?
    if (mySawVictory)
    {
	mySawVictory = false;
	if (myStrategy != STRATEGY_PARTY)
	{
	    myStrategy = STRATEGY_PARTY;
	    return actionYell(YELL_VICTORY);
	}
    }

    // If we can see the avatar, we may need to trigger some yelling.
    if (avatarvis)
    {
	// Same room strategy is still at peace... unless!
	if (myStrategy == STRATEGY_SAMEROOM)
	{
	    // If the avatar is in the foyer, cautiously warn them.
	    if (avatarvis->pos().inFoyerRoom())
	    {
		if (!rand_choice(20))
		{
		    return actionYell(YELL_KEEPOUT);
		}
		// Otherwise we keep away, but stay same room.
		if (aiFleeFrom(avatarvis->pos(), true))
		    return true;
	    }
	    else
	    {
		// Crap, the avatar has invaded!
		myStrategy = STRATEGY_FETCHITEMS;
		return actionYell(YELL_INVADER);
	    }
	}


	// If we aren't in kill mode, spring the trap if the avatar
	// has entered the trap room, storage, or ambush rooms.
	if (myStrategy != STRATEGY_KILL && myStrategy != STRATEGY_KILLCHASE)
	{
	    if (avatarvis->pos().inStorageRoom() ||
		avatarvis->pos().inTrapRoom() ||
		avatarvis->pos().inAmbushRoom())
	    {
		myStrategy = STRATEGY_KILL;
		return actionYell(YELL_KILL);
	    }
	}

	// If the avatar has a ranged weapon, let people know
	if (myStrategy != STRATEGY_SAMEROOM && 
	    !myAvatarHasRanged && avatarvis->lookupWand())
	{
	    return actionYell(YELL_RANGED);
	}
    }

    return false;
}


bool
MOB::aiTactics(MOB *avatarvis)
{
    int			dx, dy;

    if (getDefinition() == MOB_KOBOLDBABY)
    {
	if (avatarvis)
	{
	    int		adist = avatarvis->pos().dist(pos());
	    bool	aisrange = avatarvis->lookupWand() ? true : false;
	    // We flee at distance three so we never accidentally
	    // wander next to the avatar.
	    if (adist <= 3)
	    {
		if (aiFleeFromSafe(avatarvis->pos(), aisrange))
		    return true;		// fight another day!
	    }
	}

	return false;
    }

    if (avatarvis && myStrategy != STRATEGY_SAMEROOM
		  && myStrategy != STRATEGY_KILL
		  && myStrategy != STRATEGY_KILLCHASE)
    {
	// Protect ourselves from the avatar.
	int		adist = avatarvis->pos().dist(pos());
	bool		aisrange = avatarvis->lookupWand() ? true : false;

	// We need to start destroying our surplus early if
	// the avatar gets within range.
	// If we are at distance 2, and have one surplus ranged
	// weapon, we should destroy it now or else later path
	// finding might get us to distance 1 without enough time!
	if (adist > 1 && adist <= numSurplusRange()+1)
	{
	    if (aiDestroySomethingGood(avatarvis))
		return true;
	}

	// We want to quaff potions if the avatar gets too close
	// Otherwise we wait for the final battle.
	if (adist > 1 && adist < 3 && aiBattlePrep())
	    return true;

	if (adist <= 1)
	{
	    // crap crap crap!
	    if (aiFleeFromSafe(avatarvis->pos(), aisrange))
		return true;		// fight another day!

	    // Couldn't run.  Deny anything good!
	    if (aiDestroySomethingGood(avatarvis))
		return true;

	    // Go out like a brave kobold!
	    pos().dirTo(avatarvis->pos(), dx, dy);
	    return actionBump(dx, dy);
	}

	// Carry on!
    }

    if (avatarvis && 
	    (myStrategy == STRATEGY_KILL ||
	     myStrategy == STRATEGY_KILLCHASE))
    {
	// Protect ourselves from the avatar.
	int		adist = avatarvis->pos().dist(pos());
	bool		aisrange = avatarvis->lookupWand() ? true : false;

	// Battle prep, consumes potions.
	if (adist > 1 && aiBattlePrep())
	    return true;

	// Always the ranged attacks with kobolds
	// This refuses to do point blank range attacks.
	if (aiRangeAttack())
	    return true;

	int		hostiles = avatarvis->numberMeleeHostiles();

	if (adist <= 1)
	{
	    // crap crap crap!
	    if (hostiles <= 1 || lookupWand())
	    {
		// It's just me!  Not time yet!
		if (aiFleeFromSafe(avatarvis->pos(), aisrange))
		    return true;		// fight another day!
	    }

	    // Couldn't run.  Deny anything good!
	    if (aiDestroySomethingGood(avatarvis))
		return true;

	    // Go out like a brave kobold!
	    pos().dirTo(avatarvis->pos(), dx, dy);
	    return actionBump(dx, dy);
	}

	// Carry on!
	// If we are adist 2, and the avatar lacks a ranged attack
	// and is not yet surrounded by two kobolds,  we wait.
	// Otherwise we charge.
	// But, if we have a ranged attack, we don't want to charge if
	// we are too close.
	if (adist == 2 && ((lookupWand()) || ((!aisrange) && (hostiles < 2))))
	{
	    if (!rand_choice(5))
		return actionYell(YELL_TAUNT);
	    return actionBump(0, 0);
	}

	// Check if the avatar is in the same range, don't chase
	// out of the ambush room.
	if (myStrategy == STRATEGY_KILLCHASE ||
	    (avatarvis->pos().inTrapRoom() || avatarvis->pos().inAmbushRoom() ||
	     avatarvis->pos().inStorageRoom()))
	{
	    return aiCharge(avatarvis, AI_FLANK, false);
	}
	else
	{
	    // Avatar retreats, so we retreat.
	    if (aiFleeFromSafe(avatarvis->pos(), aisrange))
		return true;		// fight another day!
	}
    }

    return false;
}

bool
MOB::aiWantsAnyMoreItems() const
{
    // Is there anything we want for christmas?
    if (!lookupWeapon())
	return true;

    if (!lookupWand())
	return true;

    return false;
}

bool
MOB::aiWantsItem(ITEM *item) const
{
    if (!item)
	return false;

    if (item->isMelee() && !lookupWeapon())
	return true;

    if (item->isRanged() && !lookupWand())
	return true;

    // All else is fodder.
    // "Everything is fodder" was Megatron's favorite saying according
    // to the summary comic books I have.
    return false;
}

bool
MOB::aiStrategy()
{
    MOB		*avatarvis;
    MOBLIST	 enemies;

    // Before strategy comes tactics!
    avatarvis = 0;
    getVisibleEnemies(enemies);
    if (enemies.entries())
	avatarvis = enemies(0);

    // First test if there are any twitch things to do.
    if (aiTwitch(avatarvis))
	return true;


    if (aiTactics(avatarvis))
	return true;

    if (getDefinition() == MOB_KOBOLDBABY)
    {
	switch (myStrategy)
	{
	    case STRATEGY_SAMEROOM:
	    {
		// Keep to the same room
		return aiRandomWalk(false, true);
	    }

	    case STRATEGY_FETCHITEMS:
	    case STRATEGY_AMBUSH:
	    case STRATEGY_KILL:
	    case STRATEGY_KILLCHASE:
	    {
		// Run straight to storage and stay there
		if (!pos().inStorageRoom())
		{
		    if (!myTarget.inStorageRoom())
			myTarget = pos().map()->findRoomOfType(ROOMTYPE_STORAGE);
		    if (aiPathFindToAvoid(myTarget, avatarvis))
			return true;
		    // Stuck.
		    return aiRandomWalk();
		}

		// Otherwise, random walk, stick to the storage room
		return aiRandomWalk(false, true);
	    }

	    case STRATEGY_PARTY:
	    {
		if (!rand_choice(200))
		{
		    return actionYell(YELL_VICTORY);
		}

		if (pos().roomType() == ROOMTYPE_NURSERY)
		{
		    return aiRandomWalk(false, true);
		}

		if (myTarget.roomType() != ROOMTYPE_NURSERY)
		    myTarget = pos().map()->findRoomOfType(ROOMTYPE_NURSERY);

		if (aiPathFindTo(myTarget))
		    return true;
		return aiRandomWalk();
	    }
	}
    }

    switch (myStrategy)
    {
	case STRATEGY_SAMEROOM:
	{
	    // If we see the avatar..
	    // Keep to the same room
	    return aiRandomWalk(false, true);
	}

	case STRATEGY_FETCHITEMS:
	{
	    // Pick up only items we expressed an interest in.
	    if (pos().item() && pos().item()->getInterestedUID() == getUID())
	    {
		return actionPickup();
	    }

	    // Find a local item to pickup.
	    ITEMLIST		items;
	    MOBLIST		fellowmobs;
	    pos().getAllRoomItems(items);
	    pos().getAllRoomMobs(fellowmobs);

	    if (items.entries())
	    {
		// Search all items.  If one is flagged for us,
		// use it.  Otherwise pick the closest item
		// that isn't reserved by someone already in this room.
		int			mindist = 0, dist;
		ITEM			*bestitem = 0;
		ITEM			*item;
		int			itemuid;
		bool			instorage = pos().inStorageRoom();

		// Quick check if we already claimed something.
		for (int i = 0; i < items.entries(); i++)
		{
		    item = items(i);
		    if (item->getDefinition() == ITEM_CORPSE)
			continue;

		    itemuid = item->getInterestedUID();
		    if (itemuid != INVALID_UID)
		    {
			// See if it is already claimed...
			if (itemuid == getUID())
			{
			    // By me!
			    bestitem = item;
			    break;
			}
		    }
		}

		if (!bestitem)
		{
		    for (int i = 0; i < items.entries(); i++)
		    {
			item = items(i);

			// Only try to equip in storage room.
			if (instorage && !aiWantsItem(item))
			    continue;

			itemuid = item->getInterestedUID();
			if (itemuid != INVALID_UID)
			{
			    // See if any mob has it.
			    bool	uidvalid = false;
			    for (int j = 0; j < fellowmobs.entries(); j++)
			    {
				if (fellowmobs(j)->getUID() == itemuid)
				{
				    uidvalid = true;
				    break;
				}
			    }
			    if (!uidvalid)
			    {
				item->setInterestedUID(INVALID_UID);
			    }
			    else
			    {
				// Valid uid, so leave it with that mob
				continue;
			    }
			}

			// Check the dist to this item.
			dist = pos().dist(item->pos());
			if (!bestitem || dist < mindist)
			{
			    mindist = dist;
			    bestitem = item;
			}
		    }
		}

		if (bestitem)
		{
		    // Our new best friend!
		    bestitem->setInterestedUID(getUID());
		    myTarget = bestitem->pos();

		    if (myTarget == pos())
		    {
			// Woohoo!  We are already there.
			return actionPickup(bestitem);
		    }

		    if (aiPathFindToAvoid(myTarget, avatarvis))
			return true;

		    // Stuck.
		    return aiRandomWalk();
		}
	    }

	    // If no items to pick up, see if we have extra.  Then
	    // we have to go to storage to drop them.  If we lack something
	    // cool, we want to go to the storage room.  But if we are
	    // already there, we must have not found anything.
	    if (hasSurplusItems() || 
		(!pos().inStorageRoom() && aiWantsAnyMoreItems()))
	    {
		// If we are in the storage room, drop!
		if (pos().inStorageRoom() && hasSurplusItems())
		{
		    // In case hasSurplus and DropSurplus are out
		    // of sync we comp the two.
		    if (!actionDropSurplus())
		    {
			myStrategy = STRATEGY_AMBUSH;
		    }
		}
		// Otherwise we head to the storage room.
		if (!myTarget.inStorageRoom())
		    myTarget = pos().map()->findRoomOfType(ROOMTYPE_STORAGE);
		if (aiPathFindToAvoid(myTarget, avatarvis))
		    return true;
		// Stuck.
		return aiRandomWalk();
	    }

	    // Path to the trap room.
	    // We don't go straight to Ambush as we may stumble along
	    // items on the way.
	    if (!pos().inStorageRoom() && !pos().inAmbushRoom() && !pos().inTrapRoom())
	    {
		if (!myTarget.inTrapRoom())
		    myTarget = pos().map()->findRoomOfType(ROOMTYPE_TRAP);
		if (aiPathFindToAvoid(myTarget, avatarvis))
		    return true;
		// Stuck.
		return aiRandomWalk();
	    }

	    // Otherwise, head for the ambush.
	    myStrategy = STRATEGY_AMBUSH;
	    // FALLTHROUGH
	}

	case STRATEGY_AMBUSH:
	{
	    if (pos().roomId() == myTarget.roomId() && pos().inAmbushRoom())
	    {
		// Same room wander.
		// But only if we get to the Right ambush room.
		return aiRandomWalk(false, true);
	    }

	    if (!myTarget.inAmbushRoom())
	    {
		// Find the canonical target.
		myTarget = pos().map()->findRoomOfType(ROOMTYPE_AMBUSH);
	    }
	    // And go there
	    if (aiPathFindToAvoid(myTarget, avatarvis))
		return true;
	    // Can't get there yet, wander
	    return aiRandomWalk();
	}

	case STRATEGY_KILL:
	case STRATEGY_KILLCHASE:
	{
	    if (avatarvis)
	    {
		// Small chance of alerting to where the avatar is.
		if (!rand_choice(100))
		{
		    return actionYell(YELL_LOCATION);
		}
		if (aiKillPathTo(avatarvis))
		    return true;
	    }
	    if (myTarget.valid())
	    {
		// If we get there and find nothing, abort.
		if (myTarget == pos())
		    myTarget = POS();
		else
		{
		    if (aiPathFindTo(myTarget))
			return true;
		}
	    }
	    // Include room changes.
	    return aiRandomWalk();
	}

	case STRATEGY_PARTY:
	{
	    if (!rand_choice(200))
	    {
		return actionYell(YELL_VICTORY);
	    }
	    return aiDoAIType(AI_HOME);
	}
    }

    return aiRandomWalk();
}


double
avatar_weight_weapon(ITEM *item)
{
    int		ip, ia, ic;
    double	weight;
	
    item->getWeaponStats(ip, ia, ic);

    // Estimate average damage for our weight.
    weight = ip * 0.5;
    weight += ic;
    weight *= ia / 100.;
	
    return weight;
}

double
avatar_weight_wand(ITEM *item)
{
    int		ip, ia, ic, ir;
    double	weight;
	
    item->getRangeStats(ir, ip, ic, ia);

    // Estimate average damage for our weight.
    weight = ip * 0.5;
    weight += ic;
    // Area technically improves with the square, but in practice
    // you don't have everyone bunched, so make it just
    // the linear
    weight *= ia;

    // Each two points of range gives us another attack on a charging
    // foe.
    weight *= ir * 0.5;

    return weight;
}

ITEM *
MOB::aiLikeMoreWeapon(ITEM *a, ITEM *b) const
{
    // Trivial case
    if (!a) return b;
    if (!b) return a;
    
    if (avatar_weight_weapon(b) > avatar_weight_weapon(a))
	return b;
    // Default to a.
    return a;
}

ITEM *
MOB::aiLikeMoreWand(ITEM *a, ITEM *b) const
{
    // Trivial case
    if (!a) return b;
    if (!b) return a;
    
    if (avatar_weight_wand(b) > avatar_weight_wand(a))
	return b;
    // Default to a.
    return a;
}

bool
MOB::aiDoMouse()
{
    int		lastdir, dir, newdir;
    int		dx, dy;
    bool	found = false;
    int		i;

    // Least three bits track our last direction.
    lastdir = myAIState & 7;

    // We want to try and track a wall.  We want this wall to
    // be on our right hand side.

    // Take a look at right hand square.
    rand_angletodir((lastdir + 2) & 7, dx, dy);
    
    if (!canMoveDir(dx, dy, true))
    {
	// Got a wall to the right.  Try first available direction.
	// Because we want to hug using diagonals, we try the forward
	// and right first.
	dir = (lastdir+1) & 7;
    }
    else
    {
	// No wall on right.  Try and go straight forward by default.
	dir = lastdir;
    }

    // If everyway is blocked, bump straight forward.
    newdir = lastdir;
    for (i = 0; i < 8; i++)
    {
	rand_angletodir(dir, dx, dy);
	if (canMoveDir(dx, dy, true))
	{
	    newdir = dir;
	    break;
	}
	// Keeping wall on right means turning left first!
	dir--;
	dir &= 7;
    }

    if (newdir == -1)
	newdir = lastdir;
    else
	found = true;

    // Store our last direction.
    myAIState &= ~7;
    myAIState |= newdir;

    if (found)
    {
	return actionBump(dx, dy);
    }

    // Mouse is cornered!  Return to normal AI.
    return false;
}

bool
MOB::aiStraightLine()
{
    int		lastdir;
    int		dx, dy;
    MOB		*target;
    POS		t;

    // Least three bits track our last direction.
    lastdir = myAIState & 7;

    // Bump & go.

    // Take a look at right hand square.
    rand_angletodir(lastdir, dx, dy);

    t = pos().delta(dx, dy);
    
    // If target is avatar, run into him!
    target = t.mob();
    if (target && !isFriends(target))
    {
	return actionBump(dx, dy);
    }
    
    if (!canMove(t, true))
    {
	// Can't go there!  Pick new direction.
	// Do not want same direction again.
	lastdir += rand_choice(7) + 1;
	lastdir &= 7;
    }
    myAIState = lastdir;
    rand_angletodir(lastdir, dx, dy);
    t = pos().delta(dx, dy);
    
    // If target is avatar, run into him!
    target = t.mob();
    if (target && !isFriends(target))
    {
	return actionBump(dx, dy);
    }

    // Bump failed.
    if (!canMove(t, true))
	return false;

    // Move forward!
    return actionBump(dx, dy);
}

bool
MOB::aiFleeFromAvatar()
{
    if (aiAcquireAvatar())
    {
	if (aiFleeFrom(myTarget))
	    return true;
    }

    return false;
}

bool
MOB::aiFleeFrom(POS goal, bool sameroom)
{
    int		dx, dy;
    int		angle, resultangle = 0, i;
    int		choice = 0;

    pos().dirTo(goal, dx, dy);
    dx = -dx;
    dy = -dy;
    
    angle = rand_dirtoangle(dx, dy);

    // 3 ones in same direction get equal chance.
    for (i = angle-1; i <= angle+1; i++)
    {
	rand_angletodir(i, dx, dy);
	if (sameroom)
	{
	    POS	goal;
	    goal = pos().delta(dx, dy);
	    if (goal.roomId() != pos().roomId())
		continue;
	}
	if (canMoveDir(dx, dy))
	{
	    choice++;
	    if (!rand_choice(choice))
		resultangle = i;
	}
    }

    if (!choice)
    {
	// Try a bit more desperately...
	for (i = angle-2; i <= angle+2; i += 4)
	{
	    rand_angletodir(i, dx, dy);
	    if (sameroom)
	    {
		POS	goal;
		goal = pos().delta(dx, dy);
		if (goal.roomId() != pos().roomId())
		    continue;
	    }
	    if (canMoveDir(dx, dy))
	    {
		choice++;
		if (!rand_choice(choice))
		    resultangle = i;
	    }
	}
    }

    if (choice)
    {
	// Move in the direction
	rand_angletodir(resultangle, dx, dy);
	return actionBump(dx, dy);
    }

    // Failed
    return false;
}

bool
MOB::aiFleeFromSafe(POS goal, bool avoidrange)
{
    if (aiFleeFromSafe(goal, avoidrange, true))
	return true;
    return aiFleeFromSafe(goal, avoidrange, false);
}

bool
MOB::aiFleeFromSafe(POS goal, bool avoidrange, bool avoidmob)
{
    int		dx, dy;
    int		angle, resultangle = 0, i;
    int		choice = 0;

    pos().dirTo(goal, dx, dy);
    dx = -dx;
    dy = -dy;
    
    angle = rand_dirtoangle(dx, dy);

    // 3 ones in same direction get equal chance.
    for (i = angle-1; i <= angle+1; i++)
    {
	rand_angletodir(i, dx, dy);

	// Ignore if it gets as beside!
	POS	g;
	g = pos().delta(dx, dy);
	if (g.dist(goal) <= 1)
	    continue;
	if (avoidrange && goal.mob())
	{
	    if (goal.mob()->canTargetAtRange(g))
		continue;
	}
	if (canMoveDir(dx, dy, avoidmob))
	{
	    choice++;
	    if (!rand_choice(choice))
		resultangle = i;
	}
    }

    if (!choice)
    {
	// Try a bit more desperately...
	for (i = angle-2; i <= angle+2; i += 4)
	{
	    rand_angletodir(i, dx, dy);
	    // Ignore if it gets as beside!
	    POS	g;
	    g = pos().delta(dx, dy);
	    if (g.dist(goal) <= 1)
		continue;
	    if (avoidrange && goal.mob())
	    {
		if (goal.mob()->canTargetAtRange(g))
		    continue;
	    }
	    if (canMoveDir(dx, dy, avoidmob))
	    {
		choice++;
		if (!rand_choice(choice))
		    resultangle = i;
	    }
	}
    }

    if (choice)
    {
	// Move in the direction
	rand_angletodir(resultangle, dx, dy);
	return actionBump(dx, dy);
    }

    // Failed
    return false;
}

bool
MOB::aiAcquireAvatar()
{
    MOB		*a;
    a = getAvatar();

    if (a && !a->alive())
	return false;

    return aiAcquireTarget(a);
}

bool
MOB::aiAcquireTarget(MOB *a)
{
    if (a && a->isSwallowed())
    {
	myTarget = POS();
	return false;
    }
    // If we can see avatar, charge at them!  Otherwise, move
    // to our target if set.  Otherwise random walk.
    if (a && 
	a->pos().isFOV() &&
	pos().isFOV())
    {
	int		dist;
	// Determine distance
	dist = pos().dist(a->pos());
	if (dist <= defn().sightrange)
	{
	    myTarget = a->pos();
	}
    }
    else
    {
	// Random chance of forgetting avatar location.
	// Also, if current location is avatar location, we forget it.
	if (pos() == myTarget)
	    myTarget = POS();
	if (!rand_choice(10))
	    myTarget = POS();
    }

    return myTarget.valid();
}

bool
MOB::aiRandomWalk(bool orthoonly, bool sameroom)
{
    int		dx, dy;
    int		angle, resultangle = 0;
    int		choice = 0;

    for (angle = 0; angle < 8; angle++)
    {
	if (orthoonly && (angle & 1))
	    continue;

	rand_angletodir(angle, dx, dy);
	if (sameroom)
	{
	    POS	goal;
	    goal = pos().delta(dx, dy);
	    if (goal.roomId() != pos().roomId())
		continue;
	    if (goal.tile() == TILE_DOOR)
	    {
		// Obviously going through a door is changing the room!
		continue;
	    }
	}
	if (aiAvoidDirection(dx, dy))
	    continue;
	if (canMoveDir(dx, dy))
	{
	    choice++;
	    if (!rand_choice(choice))
		resultangle = angle;
	}
    }

    if (choice)
    {
	// Move in the direction
	rand_angletodir(resultangle, dx, dy);
	return actionWalk(dx, dy);
    }

    // Failed
    return false;
}

bool
MOB::aiCharge(MOB *foe, AI_NAMES aitype, bool orthoonly)
{
    if (aiAcquireTarget(foe))
    {
	if (aitype == AI_FLANK)
	{
	    if (aiFlankTo(myTarget))
		return true;
	}
	else
	{
	    if (aiMoveTo(myTarget, orthoonly))
		return true;
	}
    }

    return false;
}

bool
MOB::aiRangeAttack(MOB *target)
{
    // See if we have a ranged attack at all!
    if (!defn().range_valid && !lookupWand())
	return false;

    int		 dx, dy;
    MOB		*a = getAvatar(), *v;

    // Override the target from the avatar...
    if (target)
	a = target;

    if (!a)
	return false;

    if (!a->alive())
	return false;

    pos().dirTo(a->pos(), dx, dy);

    // If we are in melee range, don't use our ranged attack!
    if (pos().dist(a->pos()) <= 1)
    {
	return false;
    }

    // Try ranged attack.
    v = pos().traceBullet(getRangedRange(), dx, dy);
    if (v == a)
    {
	// Potential range attack.
	return actionFire(dx, dy);
    }

    return false;
}

bool
MOB::canTargetAtRange(POS goal) const
{
    int		dx, dy;

    pos().dirTo(goal, dx, dy);

    // If we are in melee range, don't use our ranged attack!
    if (pos().dist(goal) <= 1)
    {
	return false;
    }

    // Try ranged attack.
    int		range = getRangedRange();

    POS		next = pos();

    while (range > 0)
    {
	range--;
	next = next.delta(dx, dy);
	if (next == goal)
	    return true;
	if (next.mob())
	{
	    return false;
	}

	// Stop at a wall.
	if (!next.defn().ispassable)
	    return false;
    }
    return false;
}

bool
MOB::aiMoveTo(POS t, bool orthoonly)
{
    int		dx, dy, dist;
    int		angle, resultangle = 0, i;
    int		choice = 0;

    pos().dirTo(t, dx, dy);

    if (orthoonly && dx && dy)
    {
	// Ensure source is orthogonal.
	if (rand_choice(2))
	    dx = 0;
	else
	    dy = 0;
    }

    dist = pos().dist(t);

    if (dist == 1)
    {
	// Attack!
	if (canMoveDir(dx, dy, false))
	    return actionBump(dx, dy);
	return false;
    }

    // Move in general direction, preferring a straight line.
    angle = rand_dirtoangle(dx, dy);

    for (i = 0; i <= 2; i++)
    {
	if (orthoonly && (i & 1))
	    continue;

	rand_angletodir(angle-i, dx, dy);
	if (canMoveDir(dx, dy))
	{
	    choice++;
	    if (!rand_choice(choice))
		resultangle = angle-i;
	}
	
	rand_angletodir(angle+i, dx, dy);
	if (canMoveDir(dx, dy))
	{
	    choice++;
	    if (!rand_choice(choice))
		resultangle = angle+i;
	}

	if (choice)
	{
	    rand_angletodir(resultangle, dx, dy);

	    // If wew can leap, maybe leap?
	    if (defn().canleap)
	    {
		if (dist > 2 &&
		    canMoveDir(2*dx, 2*dy))
		{
		    formatAndReport("%S <leap>.");
		    dx *= 2;
		    dy *= 2;
		}
	    }
	    
	    return actionWalk(dx, dy);
	}
    }

    return false;
}

bool
MOB::aiKillPathTo(MOB *target)
{
    if (!target)
	return false;

    myTarget = target->pos();
    
    return aiPathFindTo(target->pos());
}

bool
MOB::aiPathFindTo(POS goal)
{
    if (aiPathFindTo(goal, true))
	return true;
    return aiPathFindTo(goal, false);
}

bool
MOB::aiPathFindTo(POS goal, bool avoidmob)
{
    int		dx, dy, ndx, ndy;
    int		curdist, dist;
    int		distmap;

    bool	found;

    // see if already there.
    if (pos() == goal)
	return false;

    // Really, really, inefficient.
    distmap = pos().map()->buildDistMap(goal);

    curdist = pos().getDistance(distmap);
    
    // We don't care if this square is unreachable, as if it is the
    // square with the mob, almost by definition it is unreachable
    
    found = false;
    int			nmatch = 0;
    FORALL_8DIR(dx, dy)
    {
	dist = pos().delta(dx, dy).getDistance(distmap);

	if (avoidmob)
	    if (pos().delta(dx, dy).mob())
		continue;
	// If there is a corpse there, keep away.
	if (aiAvoidDirection(dx, dy))
	    continue;
	if (dist >= 0 && dist <= curdist)
	{
	    found = true;

	    if (dist < curdist)
	    {
		// Net new smallest
		ndx = dx;
		ndy = dy;
		curdist = dist;
		nmatch = 1;
	    }
	    else
	    {
		// Replace if chance.
		nmatch++;
		if (!rand_choice(nmatch))
		{
		    ndx = dx;
		    ndy = dy;
		}
	    }
	}
    }

    // If we didn't find a direction, abort
    if (!found)
    {
	if (isAvatar())
	    formatAndReport("%S cannot find a path there.");
	return false;
    }

    // Otherwise, act.
    return actionWalk(ndx, ndy);
}

bool
MOB::aiPathFindToAvoid(POS goal, MOB *avoid)
{
    if (aiPathFindToAvoid(goal, avoid, true))
	return true;
    return aiPathFindToAvoid(goal, avoid, false);
}

double
aiWeightedDist(double dist, double avoid)
{
    // Max effect is 10
    avoid = 10 - avoid;
    if (avoid < 0)
	return dist;

    // Go non-linear!
    avoid *= avoid;
    avoid /= 10;

    return dist + avoid;
}

bool
MOB::aiPathFindToAvoid(POS goal, MOB *avoid, bool avoidmob)
{
    int		dx, dy, ndx, ndy;
    double	curdist, dist;
    int		distmap, avoidmap;

    bool	found;

    if (!avoid)
	return aiPathFindTo(goal, avoidmob);

    // see if already there.
    if (pos() == goal)
	return false;

    // Really, really, inefficient.
    distmap = pos().map()->buildDistMap(goal);
    avoidmap = pos().map()->buildDistMap(avoid->pos());

    // We bias towards path finding so that one will flow around the
    // target rather than stop hard.
    curdist = aiWeightedDist(pos().getDistance(distmap),
			     pos().getDistance(avoidmap));
    
    // We don't care if this square is unreachable, as if it is the
    // square with the mob, almost by definition it is unreachable
    
    found = false;
    int			nmatch = 0;
    FORALL_8DIR(dx, dy)
    {
	dist = pos().delta(dx, dy).getDistance(distmap);

	if (dist < 0)
	    continue;

	if (avoidmob)
	    if (pos().delta(dx, dy).mob())
		continue;

	// Refuse to step beside the avoid mob.
	if (avoid->pos().dist(pos().delta(dx, dy)) <= 1)
	    continue;

	// If the avoid mob has a ranged attack, don't be stupid.
	if (avoid->lookupWand())
	{
	    if (avoid->canTargetAtRange(pos().delta(dx, dy)))
		continue;
	}
	
	dist = aiWeightedDist(dist, pos().delta(dx, dy).getDistance(avoidmap));
	if (dist <= curdist)
	{
	    found = true;

	    if (dist < curdist)
	    {
		// Net new smallest
		ndx = dx;
		ndy = dy;
		curdist = dist;
		nmatch = 1;
	    }
	    else
	    {
		// Replace if chance.
		nmatch++;
		if (!rand_choice(nmatch))
		{
		    ndx = dx;
		    ndy = dy;
		}
	    }
	}
    }

    // If we didn't find a direction, abort
    if (!found)
    {
	if (isAvatar())
	    formatAndReport("%S cannot find a path there.");
	return false;
    }

    // Otherwise, act.
    return actionWalk(ndx, ndy);
}

bool
MOB::aiFlankTo(POS goal)
{
    int		dx, dy, dist;
    int		angle, resultangle = 0, i, j;
    int		choice = 0;

    MOB		*avatarvis = goal.mob();
    bool	aisrange = false;
    
    if (avatarvis)
	aisrange = avatarvis->lookupWand() ? true : false;

    pos().dirTo(goal, dx, dy);

    dist = pos().dist(goal);

    if (dist == 1)
    {
	// Attack!
	if (canMove(goal, false))
	    return actionBump(dx, dy);
	return false;
    }

    // Move in general direction, preferring a straight line.
    angle = rand_dirtoangle(dx, dy);

    for (j = 0; j <= 2; j++)
    {
	// To flank, we prefer the non-straight approach.
	switch (j)
	{
	    case 0:
		i = 1;
		break;
	    case 1:
		i = 0;
		break;
	    case 2:
		i = 2;
		break;
	}
	rand_angletodir(angle-i, dx, dy);
	if (canMoveDir(dx, dy) &&
	    (!aisrange || !avatarvis->canTargetAtRange(pos().delta(dx, dy))))
	{
	    choice++;
	    if (!rand_choice(choice))
		resultangle = angle-i;
	}
	
	rand_angletodir(angle+i, dx, dy);
	if (canMoveDir(dx, dy) &&
	    (!aisrange || !avatarvis->canTargetAtRange(pos().delta(dx, dy))))
	{
	    choice++;
	    if (!rand_choice(choice))
		resultangle = angle+i;
	}

	if (choice)
	{
	    rand_angletodir(resultangle, dx, dy);
	    return actionWalk(dx, dy);
	}
    }

    return false;
}

