/*
 * Licensed under BSD license.  See LICENCE.TXT  
 *
 * Produced by:	Jeff Lait
 *
 *      	Jacob's Matrix Development
 *
 * NAME:        engine.cpp ( Jacob's Matrix, C++ )
 *
 * COMMENTS:
 *	Our game engine.  Grabs commands from its command
 *	queue and processes them as fast as possible.  Provides
 *	a mutexed method to get a recent map from other threads.
 *	Note that this will run on its own thread!
 */

#include "engine.h"

#include "msg.h"
#include "rand.h"
#include "map.h"
#include "mob.h"
#include "speed.h"
#include "item.h"
#include <time.h>

static void *
threadstarter(void *vdata)
{
    ENGINE	*engine = (ENGINE *) vdata;

    engine->mainLoop();

    return 0;
}

ENGINE::ENGINE(DISPLAY *display)
{
    myMap = myOldMap = 0;
    myDisplay = display;

    myThread = THREAD::alloc();
    myThread->start(threadstarter, this);
}

ENGINE::~ENGINE()
{
    if (myMap)
	myMap->decRef();
    if (myOldMap)
	myOldMap->decRef();
}

MAP *
ENGINE::copyMap()
{
    AUTOLOCK	a(myMapLock);

    if (myOldMap)
	myOldMap->incRef();

    return myOldMap;
}

void
ENGINE::updateMap()
{
    {
	AUTOLOCK	a(myMapLock);

	if (myOldMap)
	    myOldMap->decRef();

	myOldMap = myMap;
    }
    if (myOldMap)
    {
	myMap = new MAP(*myOldMap);
	myMap->incRef();
    }
}

#define VERIFY_ALIVE() \
    if (avatar && !avatar->alive()) \
    {			\
	msg_report("Dead people can't move.  ");	\
	break;			\
    }

void redrawWorld();

bool
ENGINE::awaitRebuild()
{
    int		value = 0;

    while (!myRebuildQueue.remove(value))
    {
	redrawWorld();
    }

    myRebuildQueue.clear();

    return value;
}

void
ENGINE::awaitSave()
{
    int		value;

    while (!mySaveQueue.remove(value))
    {
	redrawWorld();
    }

    mySaveQueue.clear();
}

void
ENGINE::mainLoop()
{
    rand_setseed((long) time(0));

    COMMAND		cmd;
    MOB			*avatar;
    bool		timeused = false;
    bool		doheartbeat = true;

    while (1)
    {
	avatar = 0;
	if (myMap)
	    avatar = myMap->avatar();

	timeused = false;
	if (doheartbeat && avatar && avatar->aiForcedAction())
	{
	    cmd = COMMAND(ACTION_NONE);
	    timeused = true;
	}
	else
	{
	    if (doheartbeat)
		msg_newturn();

	    // Allow the other thread a chance to redraw.
	    // Possible we might want a delay here?
	    updateMap();
	    avatar = 0;
	    if (myMap)
		avatar = myMap->avatar();
	    cmd = queue().waitAndRemove();

	    doheartbeat = false;
	}

	switch (cmd.action())
	{
	    case ACTION_WAIT:
		// The dead are allowed to wait!
		timeused = true;
		break;

	    case ACTION_RESTART:
	    {
		int		loaded = 1;
		if (myMap)
		    myMap->decRef();

		myMap = MAP::load();

		if (!myMap)
		{
		    loaded = 0;
		    avatar = MOB::create(MOB_AVATAR);
		    myMap = new MAP(cmd.dx(), avatar, myDisplay);
		}

#if 0
		// Force lots of loads
		while (1)
		{
		    MAP		*temp;
		    avatar = MOB::create(MOB_AVATAR);
		    temp = new MAP(cmd.dx(), avatar, myDisplay);
		    temp->incRef();
		    temp->decRef();
		}
#endif

		myMap->incRef();
		myMap->setDisplay(myDisplay);
		myMap->rebuildFOV();
		myMap->cacheItemPos();

		// Flag we've rebuilt.
		myRebuildQueue.append(loaded);
		break;
	    }

	    case ACTION_SAVE:
	    {
		// Only save good games
		if (myMap && avatar && avatar->alive())
		{
		    myMap->save();
		}
		mySaveQueue.append(0);
		break;
	    }

	    case ACTION_REBOOTAVATAR:
	    {
		if (avatar)
		    avatar->gainHP(avatar->defn().max_hp);
		break;
	    }

	    case ACTION_BUMP:
	    {
		VERIFY_ALIVE()
		if (avatar)
		    timeused = avatar->actionBump(cmd.dx(), cmd.dy());
		break;
	    }
	
	    case ACTION_DROP:
	    {
		VERIFY_ALIVE()
		if (avatar)
		    timeused = avatar->actionDrop(avatar->getItemFromNo(cmd.dx()));
		break;
	    }
	
	    case ACTION_BREAK:
	    {
		VERIFY_ALIVE()
		if (avatar)
		    timeused = avatar->actionBreak(avatar->getItemFromNo(cmd.dx()));
		break;
	    }
	
	    case ACTION_EAT:
	    {
		VERIFY_ALIVE()
		if (avatar)
		    timeused = avatar->actionEat(avatar->getItemFromNo(cmd.dx()));
		break;
	    }
	
	    case ACTION_MEDITATE:
	    {
		VERIFY_ALIVE()
		if (avatar)
		    timeused = avatar->actionMeditate();
		break;
	    }
	
	    case ACTION_SEARCH:
	    {
		VERIFY_ALIVE()
		if (avatar)
		    timeused = avatar->actionSearch();
		break;
	    }

	    case ACTION_CREATEITEM:
	    {
		VERIFY_ALIVE()
		if (avatar)
		{
		    ITEM	*item = ITEM::create((ITEM_NAMES) cmd.dx());
		    avatar->addItem(item);
		}
		break;
	    }
	
	    case ACTION_QUAFF:
	    {
		VERIFY_ALIVE()
		if (avatar)
		    timeused = avatar->actionQuaff(avatar->getItemFromNo(cmd.dx()));
		break;
	    }
	
	    case ACTION_ROTATE:
	    {
		if (avatar)
		    timeused = avatar->actionRotate(cmd.dx());
		break;
	    }

	    case ACTION_FIRE:
	    {
		VERIFY_ALIVE()
		if (avatar)
		{
		    timeused = avatar->actionFire(cmd.dx(), cmd.dy());
		}
		break;
	    }

	    case ACTION_SUICIDE:
	    {
		if (avatar)
		{
		    if (avatar->alive())
		    {
			msg_report("Your time has run out!  ");
			// We want the flame to die.
			avatar->gainHP(-avatar->getHP());
		    }
		}
		break;
	    }
	}

	if (myMap && timeused)
	{
	    if (cmd.action() != ACTION_SEARCH &&
		cmd.action() != ACTION_NONE)
	    {
		// Depower searching
		if (avatar)
		    avatar->setSearchPower(0);
	    }

	    // We need to build the FOV for the monsters as they
	    // rely on the updated FOV to track, etc.
	    // Rebuild the FOV map
	    // Don't do this if no avatar, or the avatar is dead,
	    // as we want the old fov.
	    if (avatar && avatar->alive())
		myMap->rebuildFOV();

	    // Update the world.
	    myMap->doMoveNPC();
	    spd_inctime();

	    // Rebuild the FOV map
	    myMap->rebuildFOV();

	    doheartbeat = true;
	}

	// Allow the other thread a chance to redraw.
	// Possible we might want a delay here?
	updateMap();
    }
}
