using Gtk;
using Gdk;
using Hildon;

public class Discovery
{
	private HashTable<string,string> connection;
	private Knots knots;
	private string[] servers;
	private bool cache;
	private string last_url;
	private int timeout;
	private int last_file_size;
	private string _username;
	private string _password;
	
	public Discovery(Knots knots)
	{
		this.knots = knots;
		this.cache = true;
		this.timeout = 30;
	}
	
	public int get_last_file_size()
	{
		return last_file_size;
	}
	
	public void set_url_cache(bool cache)
	{
		this.cache = cache;
	}
	
	public int server_count()
	{
		return servers.length;	
	}
	
	public void select_server(int selected)
	{
		if (selected < server_count())
			load_connection(servers[selected], true, false);
	}
	
	public string serverlist_xml()
	{
		string xml = "<?xml version=\"1.0\" encoding=\"UTF-8\" ?><root><items>";
		int index = 0;
		foreach(string server in servers)
		{
			string[] tokens = server.split("/");
			if (tokens[0].strip() == "")
				tokens[0] = tokens[3];
			xml += "<item><server>" + tokens[0] + "</server><id>" + index.to_string() + "</id><type>server</type></item>";
			index += 1;
		}
		return xml + "</items><pages><current>1</current><total>1</total></pages></root>";
	}
	
	public bool load_connection(string? address, bool save, bool test)
	{
		if (address != null && address.str("/") != null)
		{
			string[] tokens = address.split("/");
			string[] keys = {"name", "ssl", "port", "address"};
			if (tokens.length == 4)
			{
				connection = new HashTable<string,string>(str_hash, str_equal);
				for (int i = 0; i < 4; i++)
				{
					connection.insert(keys[i], tokens[i]);
				}
				set_timeout(10);
				if (!test || test_connection())
				{
					if (save)
						knots.get_settings().set_value("last_server", address);
					set_timeout(30);
					return true;
				}
				else
				{
					connection = null;
					set_timeout(30);
					return false;
				}
			}
		}
		return false;
	}
	
	public void add_server(string server)
	{
		if (server != null)
		{
			foreach(string s in servers)
			{
				if (s.split("/")[1] == server.split("/")[1] && s.split("/")[2] == server.split("/")[2] && s.split("/")[3] == server.split("/")[3])
					return;
			}
			servers += server;
		}
	}
	
	public void discover_server_address()
	{
		this.connection = null;
		servers = new string[0];
		string[] address = new string[0];
		string discover = null;
		try
		{
			GLib.Process.spawn_command_line_sync("/usr/share/knots/discover" + (knots.get_settings().get_value("discover_wait") != null ? " " + knots.get_settings().get_value("discover_wait") : ""), out discover);
		}
		catch (GLib.SpawnError se)
		{
			print("ERROR: %s\n", se.message);
		}
		if (discover != null)
		{
			string[] serverlist = discover.strip().split("\n");
			foreach(string server in serverlist)
			{
				if (server.str("/") != null)
				{
					address += server;
				}
			}
		}
		if (knots.get_settings().get_value("last_server") != null)
			address += knots.get_settings().get_value("last_server");
		foreach(string server in address)
		{
			if (load_connection(server, false, true))
			{
				add_server(server);
			}	
		}
		/*
		if (server_count() == 0)
		{
			string server = show_address_dialog();
			if (server != null && load_connection(server, false, true))
			{
				add_server(server);
			}
		}
		*/
		if (server_count() == 1)
		{
			knots.get_settings().set_value("last_server", servers[0]);
			load_connection(servers[0], true, false);
			knots.update_transcode_profiles();
			
		}
	}
	
	public string? show_address_dialog()
	{
		MessageDialog dialog = new MessageDialog.with_markup(null, 0, Gtk.MessageType.QUESTION, Gtk.ButtonsType.OK_CANCEL, "Server address:");
		dialog.get_image().hide();
		gtk_window_set_portrait_flags(dialog, Hildon.PortraitFlags.SUPPORT);
		Hildon.Entry server_address = new Hildon.Entry(SizeType.FINGER_HEIGHT);
		string last_address = "192.168.0.1:1978";
		bool use_ssl = false;
		if (knots.get_settings().get_value("last_server") != null)
		{
			string[] tokens = knots.get_settings().get_value("last_server").split("/");
			last_address = tokens[3] + ":" + tokens[2];
			if (tokens[1] == "1")
				use_ssl = true;
		}
		server_address.set_text(last_address);
		Hildon.CheckButton cb = new Hildon.CheckButton(SizeType.FINGER_HEIGHT);
		cb.set_active(use_ssl);
		cb.set_label("Server uses SSL");
		dialog.vbox.pack_start(server_address, false, true, 1);
		dialog.vbox.pack_start(cb, false, true, 1);
		server_address.show();
		cb.show();
		var response = dialog.run();
		if (response == ResponseType.OK)
		{
			string[] address_port = server_address.get_text().split(":");
			if (address_port.length == 2)
			{
				string address = address_port[0] + "/" + (cb.get_active() ? "1" : "0") + "/" + address_port[1] + "/" + address_port[0];
				dialog.destroy();
				return address;
			}
		}
		dialog.destroy();
		return null;
	}
	
	public bool test_connection()
	{
		if (connected())
		{
			string test = fetch("/external/test_connection");
			if (test != null && test.str("KNOTS_OK") != null)
			{
				return true;
			}
		}
		return false;
	}
	
	public string get_last_url()
	{
		return last_url;
	}
	
	public void disconnect()
	{
		this.connection = null;
		knots.disable_transcode_menu(true);
	}
	
	public string fetch(string url)
	{
		return fetch_from_address(url, address());
	}
	
	public void set_timeout(int timeout)
	{
		this.timeout = timeout;
	}
	
	public void authenticate()
	{
		if (username() == null || password() == null)
		{
			Hildon.LoginDialog login = new LoginDialog(knots.get_window());
			var response = login.run();
			if (response == ResponseType.OK)
			{
				_username = login.get_username();
				_password = login.get_password();
			}
			else
			{
				reset_credentials();
			}
			login.destroy();
		}
	}
	
	public string fetch_from_address(string? urli, string? server_address)
	{
		
		string url = urli.replace (" ", "%20").replace("<", "%3C").replace(">" , "%3E");
		string uri = (uses_ssl() ? "https" : "http") + "://" + server_address + url;
		last_url = uri;
		if (cache)
		{
			string content = knots.get_common().cached_url(uri);
			if (content != null && content != "")
			{
				return content;
			}
		}
		if (!Thread.supported())
		{
		    return "";
		}
		var session = new Soup.SessionSync ();
		session.timeout = timeout;
		var message = new Soup.Message ("GET", uri);
		session.authenticate +=  (sess, msg, auth, retrying) => {
		    if (retrying)
		    {
		    	return;
		    }
		    authenticate();
		    auth.authenticate(username(),password());
		};
		session.send_message (message);
		string body = message.response_body.data;
		last_file_size = (int)message.response_body.length;
		message=null;
		session=null;
		if (body != null)
		{
			body = xml_cleanup(body);
			if (cache && body.length < 10000)
			{
				knots.get_common().cache_url(uri, body);
			}
			else
				cache = true;
			return body;
		
		}
		cache = true;
		return "";
	}
	
	public void reset_credentials()
	{
		_username = null;
		_password = null;
	}
	
	public string xml_cleanup(string data)
	{
		if (data != null && data.str("<?xml") != null)
		{
			return data.split("</root>")[0] + "</root>";
		}
		else
		if (data != null && data.str("END_CONTENT") != null)
		{
			return data.split("END_CONTENT")[0];
		}
		return data;
	}
	
	public bool connected()
	{
		return connection != null;
	}
	
	public bool uses_ssl()
	{
		if (connection != null)
		{
			return ((string)connection.lookup("ssl")).to_int() == 1;
		}
		return false;
	}
	
	public string? username()
	{
		/*
		if (connected())
			return (string)connection.lookup("username");
		return null;
		*/
		return _username;
	}
	
	public string? password()
	{
		/*
		if (connected())
			return (string)connection.lookup("password");
		return null;
		*/
		return _password;
	}
	
	public string? ip()
	{
		if (connected())
			return (string)connection.lookup("address");
		return null;
	}
	
	public int port()
	{
		if (connected())
			return ((string)connection.lookup("port")).to_int();
		return -1;
	}
	
	public string? address()
	{
		if (connected())
			return ip() + ":" + port().to_string();
		return null;
	}
	
	public string? server_name()
	{
		if (connected())
			return (string)connection.lookup("name");
		return null;
	}
	
	
}
