/*
 * knights_callbacks.hpp
 *
 * Interface used by KnightsEngine to inform client code of events
 * that are taking place in the dungeon. This interface is also
 * re-used in KnightsClient.
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef KNIGHTS_CALLBACKS_HPP
#define KNIGHTS_CALLBACKS_HPP

#include <string>
#include <vector>

class DungeonView;
class MiniMap;
class Sound;
class StatusDisplay;
class UserControl;

class KnightsCallbacks {
public:
    virtual ~KnightsCallbacks() { }

    //
    // Most updates are sent through the DungeonView, MiniMap or
    // StatusDisplay interfaces.
    // 
    // The client code must create these objects separately, and
    // implement the following methods to allow KnightsEngine to
    // access them.
    //
    // NOTE: Currently the game only allows two players, so player_num
    // will be either 0 or 1.
    //
    virtual DungeonView & getDungeonView(int player_num) = 0;
    virtual MiniMap & getMiniMap(int player_num) = 0;
    virtual StatusDisplay & getStatusDisplay(int player_num) = 0;

    //
    // The remaining updates are sent directly through this interface;
    // see methods below.
    //
    
    // Sound
    virtual void playSound(int player_num, const Sound &sound, int frequency) = 0;

    // Tell us when someone has won/lost
    virtual void winGame(int player_num) = 0;
    virtual void loseGame(int player_num) = 0;

    // Controls
    // NOTE: Clients should return ONLY UserControls that they got from setAvailableControls, they should not make their
    // own UserControl objects. This is because we static_cast the UserControls down to Controls and this would crash if
    // the UserControl didn't have the right type!
    virtual void setAvailableControls(int player_num, const std::vector<std::pair<const UserControl*, bool> > &available_controls) = 0;
    virtual void setMenuHighlight(int player_num, const UserControl *highlight) = 0;

    // Flash Screen
    virtual void flashScreen(int player_num, int delay) = 0;
};

#endif
