/*
 * colour_change.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "colour_change.hpp"
#include "protocol.hpp"

#include <algorithm>
using namespace std;

void ColourChange::add(Colour old_col, Colour new_col)
{
    // NB would be faster to do some kind of incremental sort (eg an insertion sort),
    // but it's easier just to append the new value and then re-sort the whole lot :)
    mappings.push_back(make_pair(old_col, new_col));
    stable_sort(mappings.begin(), mappings.end());
}

struct CmpFirst {
    bool operator()(const pair<Colour,Colour> &lhs,
                    const pair<Colour,Colour> &rhs) const
    {
        return lhs.first < rhs.first;
    }
};

bool ColourChange::lookup(Colour old_col, Colour &new_col) const
{
    pair<Colour, Colour> x;
    x.first = old_col;

    CmpFirst cmp;
    vector<pair<Colour,Colour> >::const_iterator it
        = lower_bound(mappings.begin(), mappings.end(), x, cmp);
    if (it == mappings.end() || cmp(x, *it)) {
        return false;
    } else {
        new_col = it->second;
        return true;
    }
}


Colour::Colour(Coercri::InputByteBuf &buf)
{
    r = buf.readUbyte();
    g = buf.readUbyte();
    b = buf.readUbyte();
    a = buf.readUbyte();
}

void Colour::serialize(Coercri::OutputByteBuf &buf) const
{
    buf.writeUbyte(r);
    buf.writeUbyte(g);
    buf.writeUbyte(b);
    buf.writeUbyte(a);
}

ColourChange::ColourChange(Coercri::InputByteBuf &buf)
{
    const int num = buf.readVarInt();
    mappings.reserve(num);
    for (int i = 0; i < num; ++i) {
        Colour c1(buf);
        Colour c2(buf);
        mappings.push_back(std::make_pair(c1, c2));
    }
}

void ColourChange::serialize(Coercri::OutputByteBuf &buf) const
{
    buf.writeVarInt(mappings.size());
    for (vector<pair<Colour,Colour> >::const_iterator it = mappings.begin(); it != mappings.end(); ++it) {
        it->first.serialize(buf);
        it->second.serialize(buf);
    }
}
