/*
 * graphic.hpp
 *
 * Stores filename and other information needed to identify a Graphic.
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef GRAPHIC_HPP
#define GRAPHIC_HPP

#include "colour_change.hpp"
#include "kconfig_fwd.hpp"

#include "network/byte_buf.hpp" // coercri

#include <memory>
#include <string>

class Graphic {
public:
    explicit Graphic(const std::string &filename_,
                     int hx_ = 0, int hy_ = 0,
                     int r_ = -1, int g_ = -1, int b_ = -1)
        : filename(filename_),
          hx(hx_), hy(hy_), r(r_), g(g_), b(b_), id(0)
    { }

    // copy ctor. takes copy of the colour change if there is one.
    Graphic(const Graphic &rhs);

    const std::string &getFilename() const { return filename; }
    int getHX() const { return hx; }
    int getHY() const { return hy; }

    // returns (-1,-1,-1) if transparency not used.
    int getR() const { return r; }
    int getG() const { return g; }
    int getB() const { return b; }

    void setColourChange(ColourChange cc) { colour_change.reset(new ColourChange(cc)); }
    const ColourChange * getColourChange() const { return colour_change.get(); }
    
    int getID() const { return id; }    
    void setID(int i) { id = i; }

    // serialization
    explicit Graphic(int id_, Coercri::InputByteBuf &buf);
    void serialize(Coercri::OutputByteBuf &buf) const;
    
private:
    std::string filename;
    int hx, hy;
    int r, g, b;
    int id;
    std::auto_ptr<ColourChange> colour_change;  // most tiles don't use this. only used for the dead knight tiles currently.
};

std::auto_ptr<Graphic> PopGraphic(KConfig::KFile &kf);
    
#endif
