/*
 * colour_change.hpp
 *
 * Defines a mapping from old colours to new colours. Used for
 * dynamically changing colours of Graphics, eg for Knight House
 * Colours, or for changing the colour of your potion bottle when you
 * have strength, super etc.
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef COLOUR_CHANGE_HPP
#define COLOUR_CHANGE_HPP

#include "network/byte_buf.hpp" // coercri

#include <vector>
using namespace std;

struct Colour {
    unsigned char r, g, b, a;
    Colour() { }
    Colour(unsigned char rr, unsigned char gg, unsigned char bb, unsigned char aa=255) : r(rr), g(gg), b(bb), a(aa) { }
    bool operator<(const Colour &other) const {
        return r < other.r ? true
            : r > other.r ? false
            : g < other.g ? true
            : g > other.g ? false
            : b < other.b ? true
            : b > other.b ? false
            : a < other.a;
    };
    bool operator==(const Colour &other) const {
        return r==other.r && g==other.g && b==other.b && a == other.a;
    }
    bool operator!=(const Colour &other) const {
        return !(*this == other);
    }

    // serialization
    Colour(Coercri::InputByteBuf &buf);
    void serialize(Coercri::OutputByteBuf &buf) const;
};

class ColourChange {
public:
    // Constructor: creates an 'empty' ColourChange object (where all colours are passed
    // through unmodified).
    ColourChange() { }
    
    // empty(): true if this is the 'identity' colour change
    bool empty() const { return mappings.empty(); }
    
    // add: adds a mapping from old_col to new_col
    void add(Colour old_col, Colour new_col);

    // lookup: returns false if the src colour is to be left unchanged
    // returns true (and sets new_col) if the src colour is to be
    // changed to new_col.
    bool lookup(Colour old_col, Colour &new_col) const;

    // comparison functions.
    bool operator<(const ColourChange &other) const { return mappings < other.mappings; }
    bool operator==(const ColourChange &other) const { return mappings == other.mappings; }

    // serialization
    explicit ColourChange(Coercri::InputByteBuf &buf);
    void serialize(Coercri::OutputByteBuf &buf) const;
    
private:
    // Vector of (old,new) pairs. This is kept sorted (on "old" values) at all times.
    vector<pair<Colour,Colour> > mappings;
};

#endif
