/*
 * knights_server.hpp
 *
 * Represents a Knights server. Can run multiple games (and will start
 * multiple threads to do so).
 *
 * Note that access to the KnightsServer object (through its public
 * interface) should still be single threaded.
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef KNIGHTS_SERVER_HPP
#define KNIGHTS_SERVER_HPP

#include "game_info.hpp"

#include "timer/timer.hpp"  // coercri

#include "boost/shared_ptr.hpp"
#include <vector>

class KnightsConfig;
class KnightsLog;
class KnightsServerImpl;
class ServerConnection;

class KnightsServer {
public:
    typedef unsigned char ubyte;

    explicit KnightsServer(boost::shared_ptr<const KnightsConfig> config,
                           boost::shared_ptr<Coercri::Timer> timer,
                           bool allow_split_screen,
                           const std::string &motd_filename,
                           const std::string &password);
    ~KnightsServer();
    
    //
    // Methods for dealing with client connections
    // 
    
    // Call this when a new client connects
    // NOTE: ServerConnections are owned by KnightsServer and will be deleted by ~KnightsServer.
    // ip_addr is optional; if set it will be displayed in the log.
    ServerConnection & newClientConnection(std::string ip_addr = std::string());

    // Call when data is received from a client
    void receiveInputData(ServerConnection &conn, const std::vector<ubyte> &data);

    // Get data waiting to be sent to a client
    // (this copies the internal output buffer to the given vector, then clears the buffer)
    // Caller is responsible for making sure this data is sent to the client.
    void getOutputData(ServerConnection &conn, std::vector<ubyte> &data);

    // Call this when a client disconnects
    void connectionClosed(ServerConnection &conn);

    
    //
    // Methods for creating / destroying games.
    //
    
    // Start a new game
    void startNewGame(const std::string &game_name);

    // Destroy an empty game. This will fail if anyone is currently connected to the game.
    // Returns true if successful.
    bool closeGame(const std::string &game_name);

    // Get the list of running games.
    std::vector<GameInfo> getRunningGames() const;

    //
    // Query server status.
    //

    int getNumberOfPlayers() const;


    //
    // Logging
    //

    // Caller is responsible for creating & destroying the KnightsLog object
    // NOTE: This should be called before any games are created, as changes are
    // not (currently) propagated to existing KnightsGames when this is called.
    void setKnightsLog(KnightsLog *);
    
    
private:
    std::auto_ptr<KnightsServerImpl> pimpl;
};

#endif
