/*
 * server_dungeon_view.cpp
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "anim.hpp"
#include "graphic.hpp"
#include "overlay.hpp"
#include "protocol.hpp"
#include "server_dungeon_view.hpp"

namespace {
    void WriteRoomCoord(Coercri::OutputByteBuf &buf, int x, int y)
    {
        buf.writeNibbles(x+1, y+1);
    }

    void WriteTileInfo(Coercri::OutputByteBuf &buf, int depth, bool cc)
    {
        int d = depth + 64;
        if (d < 0 || d > 127) throw ProtocolError("WriteTileInfo: depth out of range");
        buf.writeUbyte(cc ? 128+d : d);
    }
}

void ServerDungeonView::setCurrentRoom(int r, int width, int height)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_SET_CURRENT_ROOM);
    buf.writeVarInt(r);
    WriteRoomCoord(buf, width, height);
}

void ServerDungeonView::addEntity(unsigned short int id, int x, int y, MapHeight ht, MapDirection facing,
                                  const Anim *anim, const Overlay *ovr, int af, int atz_diff,
                                  bool ainvuln,
                                  int cur_ofs, MotionType motion_type, int motion_time_remaining)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_ADD_ENTITY);
    buf.writeVarInt(id);
    WriteRoomCoord(buf, x, y);
    buf.writeNibbles(int(ht), facing);
    buf.writeVarInt(anim ? anim->getID() : 0);
    buf.writeVarInt(ovr ? ovr->getID() : 0);
    buf.writeNibbles(af, (int(motion_type)<<1) + int(ainvuln));
    if (af != 0) buf.writeShort(atz_diff);
    buf.writeUshort(cur_ofs);
    if (motion_type != MT_NOT_MOVING) buf.writeUshort(motion_time_remaining);
}

void ServerDungeonView::rmEntity(unsigned short int id)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_RM_ENTITY);
    buf.writeVarInt(id);
}

void ServerDungeonView::repositionEntity(unsigned short int id, int new_x, int new_y)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_REPOSITION_ENTITY);
    buf.writeVarInt(id);
    WriteRoomCoord(buf, new_x, new_y);
}

void ServerDungeonView::moveEntity(unsigned short int id, MotionType motion_type,
                                   int motion_duration, bool missile_mode)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_MOVE_ENTITY);
    buf.writeVarInt(id);
    buf.writeNibbles(motion_type, missile_mode?1:0);
    buf.writeUshort(motion_duration);
}

void ServerDungeonView::flipEntityMotion(unsigned short int id, int initial_delay, int motion_duration)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_FLIP_ENTITY_MOTION);
    buf.writeVarInt(id);
    buf.writeUshort(initial_delay);
    buf.writeUshort(motion_duration);
}

void ServerDungeonView::setAnimData(unsigned short int id, const Anim *anim, const Overlay *ovr,
                                    int af, int atz_diff, bool ainvuln, bool currently_moving)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_SET_ANIM_DATA);
    buf.writeVarInt(id);
    buf.writeVarInt(anim ? anim->getID() : 0);
    buf.writeVarInt(ovr ? ovr->getID() : 0);
    buf.writeNibbles(af, int(ainvuln)*2 + int(currently_moving));
    buf.writeShort(atz_diff);
}

void ServerDungeonView::setFacing(unsigned short int id, MapDirection new_facing)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_SET_FACING);
    buf.writeVarInt(id);
    buf.writeUbyte(new_facing);
}

void ServerDungeonView::clearTiles(int x, int y)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_CLEAR_TILES);
    WriteRoomCoord(buf, x, y);
}

void ServerDungeonView::setTile(int x, int y, int depth, const Graphic *gfx, boost::shared_ptr<const ColourChange> cc)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_SET_TILE);
    WriteRoomCoord(buf, x, y);
    WriteTileInfo(buf, depth, bool(cc));
    buf.writeVarInt(gfx ? gfx->getID() : 0);
    if (cc) cc->serialize(buf);
}

void ServerDungeonView::setItem(int x, int y, const Graphic *gfx)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_SET_ITEM);
    WriteRoomCoord(buf, x, y);
    buf.writeVarInt(gfx ? gfx->getID() : 0);
}

void ServerDungeonView::placeIcon(int x, int y, const Graphic *gfx, int dur)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_PLACE_ICON);
    WriteRoomCoord(buf, x, y);
    buf.writeVarInt(gfx ? gfx->getID() : 0);
    buf.writeUshort(dur);
}

void ServerDungeonView::flashMessage(const std::string &msg, int ntimes)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_FLASH_MESSAGE);
    buf.writeString(msg);
    buf.writeUbyte(ntimes);
}

void ServerDungeonView::cancelContinuousMessages()
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_CANCEL_CONTINUOUS_MESSAGES);
}

void ServerDungeonView::addContinuousMessage(const std::string &msg)
{
    Coercri::OutputByteBuf buf(out);
    buf.writeUbyte(SERVER_ADD_CONTINUOUS_MESSAGE);
    buf.writeString(msg);
}

