/*
 * server_callbacks.cpp
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "protocol.hpp"
#include "server_callbacks.hpp"
#include "server_dungeon_view.hpp"
#include "server_mini_map.hpp"
#include "server_status_display.hpp"
#include "sound.hpp"
#include "user_control.hpp"

ServerCallbacks::ServerCallbacks()
    : game_over(false)
{
    for (int i = 0; i < 2; ++i) {
        dungeon_view[i].reset(new ServerDungeonView(pub[i]));
        mini_map[i].reset(new ServerMiniMap(pub[i]));
        status_display[i].reset(new ServerStatusDisplay(pub[i]));

        prev_menu_highlight[i] = 0;
    }
}

ServerCallbacks::~ServerCallbacks()
{
}

void ServerCallbacks::appendPlayerCmds(int plyr, std::vector<ubyte> &out) const
{
    std::copy(pub[plyr].begin(), pub[plyr].end(), std::back_inserter(out));
    std::copy(prv[plyr].begin(), prv[plyr].end(), std::back_inserter(out));
    std::copy(all.begin(), all.end(), std::back_inserter(out));
    mini_map[plyr]->appendMiniMapCmds(out);
}

void ServerCallbacks::appendObserverCmds(std::vector<ubyte> &out) const
{
    for (int i = 0; i < 2; ++i) {
        out.push_back(SERVER_SWITCH_PLAYER);
        out.push_back(ubyte(i));
        const size_t prev_size = out.size();
        appendPlayerCmds(i, out);
        if (out.size() == prev_size) {
            // remove the SWITCH_PLAYER cmd, it isn't needed if there
            // was no output for that player
            out.pop_back();
            out.pop_back();
        }
    }
}

void ServerCallbacks::clearCmds()
{
    for (int i = 0; i < 2; ++i) {
        pub[i].clear();
        prv[i].clear();
        mini_map[i]->clearMiniMapCmds();
    }
    all.clear();
}

DungeonView & ServerCallbacks::getDungeonView(int i)
{
    return *dungeon_view[i];
}

MiniMap & ServerCallbacks::getMiniMap(int i)
{
    return *mini_map[i];
}

StatusDisplay & ServerCallbacks::getStatusDisplay(int i)
{
    return *status_display[i];
}

void ServerCallbacks::playSound(int plyr, const Sound &sound, int frequency)
{
    Coercri::OutputByteBuf buf(pub[plyr]);
    buf.writeUbyte(SERVER_PLAY_SOUND);
    buf.writeVarInt(sound.getID());
    buf.writeVarInt(frequency);
}

void ServerCallbacks::winGame(int plyr)
{
    Coercri::OutputByteBuf buf(pub[plyr]);
    buf.writeUbyte(SERVER_WIN_GAME);
    game_over = true;
    winner_num = plyr;
}

void ServerCallbacks::loseGame(int plyr)
{
    Coercri::OutputByteBuf buf(pub[plyr]);
    buf.writeUbyte(SERVER_LOSE_GAME);
}

void ServerCallbacks::setAvailableControls(int plyr, const std::vector<std::pair<const UserControl*, bool> > &available_controls)
{
    Coercri::OutputByteBuf buf(prv[plyr]);
    buf.writeUbyte(SERVER_SET_AVAILABLE_CONTROLS);
    buf.writeUbyte(available_controls.size());
    for (std::vector<std::pair<const UserControl*, bool> >::const_iterator it = available_controls.begin();
    it != available_controls.end(); ++it) {
        int x = it->first->getID();
        if (x <= 0 || x >= 128) throw UnexpectedError("Control ID out of range");
        if (it->second) x += 128;
        buf.writeUbyte(x);
    }
}

void ServerCallbacks::setMenuHighlight(int plyr, const UserControl *highlight)
{
    if (prev_menu_highlight[plyr] == highlight) return;
    prev_menu_highlight[plyr] = highlight;

    Coercri::OutputByteBuf buf(prv[plyr]);
    buf.writeUbyte(SERVER_SET_MENU_HIGHLIGHT);
    buf.writeUbyte(highlight ? highlight->getID() : 0);
}

void ServerCallbacks::flashScreen(int plyr, int delay)
{
    Coercri::OutputByteBuf buf(pub[plyr]);
    buf.writeUbyte(SERVER_FLASH_SCREEN);
    buf.writeVarInt(delay);
}
