/*
 * knights_server.cpp
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "knights_config.hpp"
#include "knights_game.hpp"
#include "knights_log.hpp"
#include "knights_server.hpp"
#include "protocol.hpp"
#include "sh_ptr_eq.hpp"

#include "network/byte_buf.hpp"  // coercri

#include <fstream>
#include <sstream>
#include <map>

class ServerConnection {
public:
    explicit ServerConnection(const std::string &ip)
        : wait_until(0), game_conn(0), client_version(0), 
          version_string_received(false), connection_accepted(false),
          failed_password_attempts(0), ip_addr(ip) { }
    
    // output buffer
    std::vector<unsigned char> output_data;
    unsigned int wait_until;   // don't send output before this time. used for password checking. 0 = disabled.

    // player name
    std::string player_name;
    
    // connection to game
    // INVARIANT: either game and game_conn are both null, or they are both non-null.
    boost::shared_ptr<KnightsGame> game;
    GameConnection * game_conn;
    std::string game_name;

    // have we got version string yet?
    int client_version;
    bool version_string_received;
    
    // have we accepted their player name (and password if applicable) yet?
    bool connection_accepted;

    int failed_password_attempts;
    std::string ip_addr;
};

const int MAX_PASSWORD_ATTEMPTS = 5;

typedef std::vector<boost::shared_ptr<ServerConnection> > connection_vector;
typedef std::map<std::string, boost::shared_ptr<KnightsGame> > game_map;

class KnightsServerImpl {
public:
    boost::shared_ptr<const KnightsConfig> knights_config;
    boost::shared_ptr<Coercri::Timer> timer;
    bool allow_split_screen;
    
    game_map games;
    connection_vector connections;

    std::string motd_file;
    std::string password;

    KnightsLog *knights_log;
};

namespace {
    void ReadDataFromKnightsGame(ServerConnection &conn)
    {
        if (conn.game) {
            if (conn.output_data.empty()) {
                // no existing data so can just overwrite
                conn.game->getOutputData(*conn.game_conn, conn.output_data);
            } else {
                // append it to the existing data
                std::vector<unsigned char> buf;
                conn.game->getOutputData(*conn.game_conn, buf);
                conn.output_data.insert(conn.output_data.end(), buf.begin(), buf.end());
            }
        }
    }

    void SendJoinGameDenied(ServerConnection &conn, const std::string &reason)
    {
        ReadDataFromKnightsGame(conn);
        Coercri::OutputByteBuf buf(conn.output_data);
        buf.writeUbyte(SERVER_JOIN_GAME_DENIED);
        buf.writeString(reason);
    }

    void SendError(ServerConnection &conn, const std::string &error, KnightsServerImpl &impl)
    {
        Coercri::OutputByteBuf buf(conn.output_data);
        buf.writeUbyte(SERVER_ERROR);
        buf.writeString(error);

        if (impl.knights_log) {
            impl.knights_log->logMessage(conn.game_name + "\terror\tplayer=" + conn.player_name + ", error=" + error);
        }
    }

    bool IsNameAvailable(const connection_vector &connections, const std::string &name)
    {
        for (connection_vector::const_iterator it = connections.begin(); it != connections.end(); ++it) {
            if ((*it)->player_name == name) return false;
        }
        return true;
    }

    void SendStartupMessages(Coercri::OutputByteBuf &out, ServerConnection &conn, connection_vector &connections)
    {
        // send the list of connected players to that player
        out.writeUbyte(SERVER_INITIAL_PLAYER_LIST);
        int np = 0;
        for (connection_vector::iterator it = connections.begin(); it != connections.end(); ++it) {
            if ((*it)->connection_accepted || it->get() == &conn) ++np;
        }
        out.writeVarInt(np);
        for (connection_vector::iterator it = connections.begin(); it != connections.end(); ++it) {
            if ((*it)->connection_accepted || it->get() == &conn) {
                out.writeString((*it)->player_name);
                out.writeString((*it)->game_name);
            }
        }                    

        // send a new player notification to all other players.
        for (connection_vector::iterator it = connections.begin(); it != connections.end(); ++it) {
            if (it->get() != &conn) {
                Coercri::OutputByteBuf out_other((*it)->output_data);
                out_other.writeUbyte(SERVER_PLAYER_CONNECTED);
                out_other.writeString(conn.player_name);
            }
        }

        // the connection is now accepted.
        conn.connection_accepted = true;
    }
}

KnightsServer::KnightsServer(boost::shared_ptr<const KnightsConfig> config, boost::shared_ptr<Coercri::Timer> timer,
                             bool allow_split_screen, const std::string &motd_file, const std::string &password)
    : pimpl(new KnightsServerImpl)
{
    pimpl->knights_config = config;
    pimpl->timer = timer;
    pimpl->allow_split_screen = allow_split_screen;
    pimpl->motd_file = motd_file;
    pimpl->password = password;
    pimpl->knights_log = 0;
}

KnightsServer::~KnightsServer()
{
}

ServerConnection & KnightsServer::newClientConnection(std::string ip)
{
    boost::shared_ptr<ServerConnection> new_conn(new ServerConnection(ip));
    pimpl->connections.push_back(new_conn);

    // Send him the MOTD
    if (!pimpl->motd_file.empty()) {
        std::ifstream str(pimpl->motd_file.c_str());
        std::string motd;
        while (1) {
            std::string line;
            std::getline(str, line);
            if (!str) break;
            motd += line;
            motd += '\n';
        }
        
        Coercri::OutputByteBuf buf(new_conn->output_data);
        buf.writeUbyte(SERVER_ANNOUNCEMENT);
        buf.writeString(motd);
    }

    // log that we got a new connection (can't show player name yet).
    if (pimpl->knights_log) {
        pimpl->knights_log->logMessage("\tincoming connection\taddr=" + ip);
    }
    
    return *new_conn;
}

void KnightsServer::receiveInputData(ServerConnection &conn,
                                     const std::vector<ubyte> &data)
{
    // This is where we decode incoming messages from the client

    std::string error_msg;
    
    try {
    
        Coercri::InputByteBuf buf(data);
    
        while (!buf.eof()) {

            if (!conn.version_string_received) {
                // The first thing the client sends us is a version string
                // e.g. "Knights/009".
                const std::string client_version_string = buf.readString();
                const std::string expected = "Knights/";
                if (client_version_string.substr(0, expected.length()) != expected) {
                    throw ProtocolError("Invalid connection string");
                }
                
                // Parse the version number
                std::istringstream str(client_version_string);
                str.seekg(expected.length());
                int ver = 0;
                str >> ver;
                if (ver < 9) {  // version 009 was the earliest to support server connections.
                    throw ProtocolError("Invalid connection string");
                }
                
                conn.client_version = ver;
                conn.version_string_received = true;
                continue;
            }
        
            const ubyte msg = buf.readUbyte();

            // If the player name and password have not yet been accepted then the client can only send us two
            // messages: CLIENT_SET_PLAYER_NAME and CLIENT_SEND_PASSWORD.
            if (!conn.connection_accepted && msg != CLIENT_SET_PLAYER_NAME && msg != CLIENT_SEND_PASSWORD) {
                throw ProtocolError("Access denied");
            }

            switch (msg) {

            case CLIENT_SET_PLAYER_NAME:
                {
                    const std::string new_name = buf.readString();
                    Coercri::OutputByteBuf out(conn.output_data);
                    if (!conn.player_name.empty()) {
                        SendError(conn, "Player name already set", *pimpl);
                    } else if (new_name.empty()) {
                        SendError(conn, "Bad player name", *pimpl);
                    } else if (!IsNameAvailable(pimpl->connections, new_name)) {
                        SendError(conn, "A player with the name \"" + new_name + "\" is already connected.", *pimpl);
                    } else {
                        // set the player name
                        conn.player_name = new_name;

                        // write a log message
                        if (pimpl->knights_log) {
                            pimpl->knights_log->logMessage("\tplayer connected\taddr=" + conn.ip_addr + ", player=" + new_name);
                        }

                        // If the server has a password then request the password. Otherwise proceed as if the
                        // password has just been accepted.
                        if (!pimpl->password.empty()) {
                            out.writeUbyte(SERVER_REQUEST_PASSWORD);
                            out.writeUbyte(1);
                        } else {
                            SendStartupMessages(out, conn, pimpl->connections);
                        }
                    }
                }
                break;

            case CLIENT_SEND_PASSWORD:
                {
                    const std::string their_password = buf.readString();
                    Coercri::OutputByteBuf out(conn.output_data);
                    if (conn.player_name.empty()) {
                        throw ProtocolError("Must set player name before sending password");
                    }
                    if (conn.failed_password_attempts == MAX_PASSWORD_ATTEMPTS) {
                        throw ProtocolError("Too many failed password attempts");                    
                    } else if (their_password == pimpl->password) {
                        Coercri::OutputByteBuf(conn.output_data);
                        SendStartupMessages(out, conn, pimpl->connections);
                        if (pimpl->knights_log) {
                            pimpl->knights_log->logMessage("\tpassword accepted\tplayer=" + conn.player_name);
                        }
                    } else {
                        ++conn.failed_password_attempts;
                        conn.wait_until = pimpl->timer->getMsec() + 2000;  // make them wait a couple of seconds between password attempts
                        out.writeUbyte(SERVER_REQUEST_PASSWORD);
                        out.writeUbyte(0);
                        if (pimpl->knights_log) {
                            pimpl->knights_log->logMessage("\tpassword rejected\tplayer=" + conn.player_name);
                        }
                    }
                }
                break;
            
            case CLIENT_REQUEST_GAME_LIST:
                {
                    Coercri::OutputByteBuf out(conn.output_data);
                    out.writeUbyte(SERVER_GAME_LIST);

                    const std::vector<GameInfo> game_infos(getRunningGames());
                    out.writeVarInt(game_infos.size());
                    for (std::vector<GameInfo>::const_iterator it = game_infos.begin(); it != game_infos.end(); ++it) {
                        out.writeString(it->game_name);
                        out.writeString(it->player_name[0]);
                        out.writeString(it->player_name[1]);
                        out.writeVarInt(it->num_observers);
                        out.writeVarInt(int(it->status));
                    }
                }
                break;
            
            case CLIENT_JOIN_GAME:
            case CLIENT_JOIN_GAME_SPLIT_SCREEN:
                {
                    const bool split_screen = (msg == CLIENT_JOIN_GAME_SPLIT_SCREEN);
                    const std::string game_name = buf.readString();

                    game_map::iterator it = pimpl->games.find(game_name);
                    if (conn.game) {
                        SendJoinGameDenied(conn, "You are already connected to a game");
                    } else if (it == pimpl->games.end()) {
                        SendJoinGameDenied(conn, "The game \"" + game_name + "\" does not exist on this server.");
                    } else if (split_screen && !it->second->isSplitScreenAllowed()) {
                        SendJoinGameDenied(conn, "Split screen game not allowed");
                    } else if (split_screen && (it->second->getNumPlayers() > 0 || it->second->getNumObservers() > 0)) {
                        SendJoinGameDenied(conn, "Cannot join split-screen if other players are already connected");
                    } else {

                        std::string client_name, client_name_2;
                    
                        if (split_screen) {
                            // dummy player names for the split screen mode
                            client_name = "Player 1";
                            client_name_2 = "Player 2";
                        } else {
                            // name 1 comes from the connection object. name 2 is unset.
                            client_name = conn.player_name;
                        }

                        // the following will send the SERVER_JOIN_GAME_ACCEPTED message and
                        // any necessary SERVER_PLAYER_JOINED_THIS_GAME messages.
                        // NOTE: This should not throw since we have checked all possible error conditions above.
                        conn.game_conn = &it->second->newClientConnection(client_name, client_name_2,
                                                                          conn.client_version);
                        conn.game = it->second;
                        conn.game_name = game_name;

                        // Now send SERVER_PLAYER_JOINED_GAME messages to all connections.
                        for (connection_vector::iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
                            Coercri::OutputByteBuf out((*it)->output_data);
                            out.writeUbyte(SERVER_PLAYER_JOINED_GAME);
                            out.writeString(client_name);
                            out.writeString(game_name);
                            // NOTE: don't bother with supporting the split screen mode here, so no msg for client_name_2.
                        }
                    }
                }
                break;
            
            case CLIENT_LEAVE_GAME:
                if (conn.game) {

                    const std::string game_name = conn.game_name;
                
                    // read any pending data from the game before we get rid of his game connection...
                    ReadDataFromKnightsGame(conn);

                    // remove him from the game
                    conn.game->clientLeftGame(*conn.game_conn);
                    conn.game.reset();
                    conn.game_conn = 0;
                    conn.game_name = "";

                    // tell him that he has been booted out of the game
                    conn.output_data.push_back(SERVER_LEAVE_GAME);

                    // send SERVER_PLAYER_LEFT_GAME messages to all connections
                    for (connection_vector::iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
                        Coercri::OutputByteBuf out((*it)->output_data);
                        out.writeUbyte(SERVER_PLAYER_LEFT_GAME);
                        out.writeString(conn.player_name);
                        out.writeString(game_name);
                    }
                }
                break;
            
            case CLIENT_CHAT:
                {
                    const std::string msg = buf.readString();

                    if (conn.game) {
                        conn.game->sendChatMessage(*conn.game_conn, msg);
                    } else {
                        // send the message to everybody who is not in a game (including the originator)
                        for (connection_vector::const_iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
                            if (!(*it)->game) {
                                Coercri::OutputByteBuf out_other((*it)->output_data);
                                out_other.writeUbyte(SERVER_CHAT);
                                out_other.writeString(conn.player_name);
                                out_other.writeUbyte(0);
                                out_other.writeString(msg);
                            }
                        }
                    }

                    // log it
                    if (pimpl->knights_log) {
                        std::string log_msg = conn.game_name + "\tchat\t";
                        log_msg += conn.player_name + ": " + msg;
                        pimpl->knights_log->logMessage(log_msg);
                    }
                }
                break;
            
            case CLIENT_SET_READY:
                {
                    const bool ready = buf.readUbyte() != 0;
                    if (conn.game) {
                        conn.game->setReady(*conn.game_conn, ready);
                    }
                }
                break;

            case CLIENT_SET_HOUSE_COLOUR:
                {
                    const int x = buf.readUbyte();
                    if (conn.game) {
                        conn.game->setHouseColour(*conn.game_conn, x);
                    }
                }
                break;

            case CLIENT_SET_MENU_SELECTION:
                {
                    const std::string key = buf.readString();
                    const int val = buf.readVarInt();
                    if (conn.game) {
                        conn.game->setMenuSelection(*conn.game_conn, key, val);
                    }
                }
                break;
            
            case CLIENT_FINISHED_LOADING:
                if (conn.game) {
                    conn.game->finishedLoading(*conn.game_conn);
                }
                break;
            
            case CLIENT_SEND_CONTROL:
                {
                    int control_num = buf.readUbyte();
                    int plyr = 0;
                    if (control_num >= 128) {
                        plyr = 1;
                        control_num -= 128;
                    }
                    if (conn.game) {
                        conn.game->sendControl(*conn.game_conn, plyr, control_num);
                    }
                }
                break;

            case CLIENT_READY_TO_END:
                if (conn.game) {
                    conn.game->readyToEnd(*conn.game_conn);
                }
                break;

            case CLIENT_QUIT:
                if (conn.game) {
                    conn.game->requestQuit(*conn.game_conn);
                }
                break;

            case CLIENT_SET_PAUSE_MODE:
                {
                    const bool p = buf.readUbyte() != 0;
                    if (conn.game) {
                        conn.game->setPauseMode(p);
                    }
                }
                break;

            case CLIENT_SPECIAL_REQUEST:
                {
                    const int request_num = buf.readVarInt();

                    // No special requests are defined at the moment, so we always send back a 0 return code.
                    Coercri::OutputByteBuf out(conn.output_data);
                    out.writeVarInt(0);
                }
                break;

            case CLIENT_CHANGE_PLAYER_NUM:
                {
                    const int new_num = buf.readVarInt();
                    if (conn.game) {
                        conn.game->changePlayerNum(*conn.game_conn, new_num);
                    }
                }
                break;
                
            default:
                throw ProtocolError("Unknown message code from client");
            }
        }

        return;  // everything below here is error handling code.
        
    } catch (std::exception &e) {
        error_msg = e.what();
    } catch (...) {
        error_msg = "Unknown Error";
    }

    // Exception thrown. This is (probably) because the client sent us
    // some bad data. Send him an error message (the client will
    // usually then respond by closing the connection). Note that we
    // clear out any existing data already in his buffer, this
    // prevents any half-written messages from being sent out.
    conn.output_data.clear();
    SendError(conn, error_msg, *pimpl);
}

void KnightsServer::getOutputData(ServerConnection &conn,
                                  std::vector<ubyte> &data)
{
    if (conn.wait_until != 0 && pimpl->timer->getMsec() < conn.wait_until) {
        // don't release the output yet -- still waiting
        data.clear();

    } else {    
        conn.wait_until = 0;
        
        ReadDataFromKnightsGame(conn);
        
        data.swap(conn.output_data);
        conn.output_data.clear();
    }
}

void KnightsServer::connectionClosed(ServerConnection &conn)
{
    // save the player's name & ip.
    const std::string name = conn.player_name;
    const std::string ip = conn.ip_addr;
    const std::string game_name = conn.game_name;
    
    // inform the KnightsGame (if we were connected to a game)
    if (conn.game) {
        conn.game->clientLeftGame(*conn.game_conn);
        conn.game.reset();
        conn.game_conn = 0;
        conn.game_name = "";
    }
    
    // erase it from 'connections'
    connection_vector::iterator it = std::find_if(pimpl->connections.begin(), 
                                                  pimpl->connections.end(),
                                                  ShPtrEq<ServerConnection>(&conn));
    if (it != pimpl->connections.end()) {
        pimpl->connections.erase(it);
    }

    // tell other players
    if (conn.connection_accepted) {
        for (connection_vector::iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
            Coercri::OutputByteBuf buf((*it)->output_data);
            buf.writeUbyte(SERVER_PLAYER_DISCONNECTED);
            buf.writeString(name);
        }
    }

    // log a message
    if (pimpl->knights_log) {
        pimpl->knights_log->logMessage(game_name + "\tplayer disconnected\taddr=" + ip + ", player=" + name);
    }
}

void KnightsServer::startNewGame(const std::string &game_name)
{
    if (game_name.empty()) throw UnexpectedError("Game name not set");
    
    game_map::const_iterator it = pimpl->games.find(game_name);
    if (it != pimpl->games.end()) {
        throw UnexpectedError("A game with this name already exists");
    } else {
        // The new game starts "empty" ie with no players attached to it.
        boost::shared_ptr<KnightsGame> game(new KnightsGame(pimpl->knights_config, pimpl->timer,
                                                            pimpl->allow_split_screen, pimpl->knights_log,
                                                            game_name));
        pimpl->games.insert(std::make_pair(game_name, game));
    }
}

bool KnightsServer::closeGame(const std::string &game_name)
{
    game_map::iterator it = pimpl->games.find(game_name);
    if (it != pimpl->games.end()
    && it->second->getNumPlayers() == 0
    && it->second->getNumObservers() == 0) {
        pimpl->games.erase(it);
        return true;
    } else {
        return false;
    }
}

std::vector<GameInfo> KnightsServer::getRunningGames() const
{
    std::vector<GameInfo> result;
    result.reserve(pimpl->games.size());
    for (game_map::const_iterator it = pimpl->games.begin(); it != pimpl->games.end(); ++it) {
        GameInfo gi;
        gi.game_name = it->first;
        it->second->getPlayerNames(gi.player_name[0], gi.player_name[1]);
        gi.num_observers = it->second->getNumObservers();
        gi.status = it->second->getStatus();
        result.push_back(gi);
    }
    return result;
}

int KnightsServer::getNumberOfPlayers() const
{
    return int(pimpl->connections.size());
}

void KnightsServer::setKnightsLog(KnightsLog *klog)
{
    pimpl->knights_log = klog;
}
