/*
 * knights_game.hpp
 *
 * Class representing an actual game of Knights. (Each KnightsServer
 * can have multiple KnightsGames.)
 *
 * This class is responsible for creating/destroying the sub-thread
 * used to run the game. 
 * 
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 * 
 */

#ifndef KNIGHTS_GAME_HPP
#define KNIGHTS_GAME_HPP

class KnightsConfig;
class KnightsGameImpl;
class GameConnection;

#include "game_info.hpp"  // for GameStatus

#include "gfx/color.hpp" // coercri
#include "timer/timer.hpp"  // coercri

#include "boost/shared_ptr.hpp"
#include <string>
#include <vector>

class KnightsLog;

class KnightsGame {
public:
    explicit KnightsGame(boost::shared_ptr<const KnightsConfig> config, boost::shared_ptr<Coercri::Timer> tmr,
                         bool allow_split_screen, KnightsLog *knights_log, const std::string &game_name);
    ~KnightsGame();

    // get information
    int getNumPlayers() const;
    int getNumObservers() const;
    void getPlayerNames(std::string &p0, std::string &p1);  // returns "" for empty seats.
    GameStatus getStatus() const;
    bool isSplitScreenAllowed() const;
    
    // add players/observers.
    // will throw an exception if the same player is added twice.
    // (if client_name_2 is non-empty will create a split-screen 2-player connection, if allowed.)
    GameConnection & newClientConnection(const std::string &client_name, const std::string &client_name_2, int client_version);

    // remove a player/observer.
    // the GameConnection is invalid after this call.
    void clientLeftGame(GameConnection &conn);


    // incoming msgs should be decoded by KnightsServer and either
    // handled there, or forwarded to one of the following methods:
    void sendChatMessage(GameConnection &, const std::string &msg);
    void setReady(GameConnection &, bool ready);
    void setHouseColour(GameConnection &, int hse_col);
    void finishedLoading(GameConnection &);
    void readyToEnd(GameConnection &);
    void requestQuit(GameConnection &);
    void setPauseMode(bool p); 
    void setMenuSelection(GameConnection &, const std::string &key, int value);
    void sendControl(GameConnection &, int plyr, unsigned char control_num);  // plyr is usually 0; can be 1 in split screen mode
    void changePlayerNum(GameConnection &, int new_num);

    // Get any outgoing msgs that need to be sent to the client.
    // Any existing contents of "data" are replaced.
    void getOutputData(GameConnection &conn, std::vector<unsigned char> &data);
    
private:
    boost::shared_ptr<KnightsGameImpl> pimpl;
};

#endif
