/*
 * knights_game.cpp
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 * 
 */

#include "misc.hpp"

#include "anim.hpp"
#include "graphic.hpp"
#include "knights_config.hpp"
#include "knights_engine.hpp"
#include "knights_game.hpp"
#include "knights_log.hpp"
#include "menu.hpp"
#include "menu_constraints.hpp"
#include "menu_selections.hpp"
#include "overlay.hpp"
#include "protocol.hpp"
#include "server_callbacks.hpp"
#include "sh_ptr_eq.hpp"
#include "sound.hpp"
#include "user_control.hpp"

#include "boost/scoped_ptr.hpp"
#include "boost/thread/locks.hpp"
#include "boost/thread/thread.hpp"

#include <algorithm>

#ifdef min
#undef min
#endif
#ifdef max
#undef max
#endif

class GameConnection {
public:
    GameConnection(const std::string &n, const std::string &n2, int pnum, int ver)
        : name(n), name2(n2),
          is_ready(false), finished_loading(false), ready_to_end(false), 
          player_num(pnum), house_colour(0),
          client_version(ver)
    {
        // set the house colour to the default for this player-num.
        if (pnum >= 0 && pnum <= 1) {
            house_colour = pnum;
        }
    }
    
    std::string name, name2;  // if name2.empty() then normal connection, else split screen connection.
    
    bool is_ready;    // true=ready to start game, false=want to stay in lobby
    bool finished_loading;   // true=ready to play, false=still loading
    bool ready_to_end;   // true=clicked mouse on winner/loser screen, false=still waiting.
    int player_num; // -1 if observer, 0 or 1 if player.
    int house_colour;  // Must fit in a ubyte. Set to zero for observers.

    int client_version;
    
    std::vector<unsigned char> output_data;    

    std::vector<const UserControl*> control_queue[2];
};

typedef std::vector<boost::shared_ptr<GameConnection> > game_conn_vector;

class KnightsGameImpl {
public:
    boost::shared_ptr<const KnightsConfig> knights_config;
    boost::shared_ptr<Coercri::Timer> timer;
    bool allow_split_screen;

    std::vector<const UserControl*> controls;
    
    MenuSelections menu_selections;
    std::string quest_description;
    game_conn_vector connections;

    bool game_over;
    bool pause_mode;
    
    // methods of KnightsGame should (usually) lock this mutex.
    // the update thread will also lock this mutex when it is making changes to the KnightsGameImpl or GameConnection structures.
    boost::mutex my_mutex;

    boost::thread update_thread;

    KnightsLog *knights_log;
    std::string game_name;

    std::string winner_name;
};

namespace {

    int GetSelection(const MenuSelections &msel, const std::string &key)
    {
        std::map<std::string, MenuSelections::Sel>::const_iterator it = msel.selections.find(key);
        if (it == msel.selections.end()) return -1;
        else return it->second.value;
    }
    
    std::string GenerateQuestDescription(const MenuSelections &msel, const KnightsConfig &knights_config)
    {
        const int quest = GetSelection(msel, "quest");
        if (quest < 0) {
            return "";
        } else if (quest > 0) {
            return knights_config.getQuestDescription(quest);
        } else {
            std::string result = "Custom Quest\n\n";

            const int mission = GetSelection(msel, "mission");
            const int exit = GetSelection(msel, "exit");
            const int num_gems = GetSelection(msel, "num_gems");
            const int gems_needed = GetSelection(msel, "gems_needed");

            if (mission == 0) {
                result += "You must secure all entry points to prevent your opponent entering the dungeon";
            } else if (mission == 4) {
                result += "You must strike the book with the wand in the special pentagram";
            } else {
                if (mission == 1) {
                    result += "Your mission is to ";
                } else {
                    result += "You must retrieve the ";
                    if (mission == 2) result += "book ";
                    else result += "wand ";
                    result += "and ";
                }
                result += "escape via ";
                switch (exit) {
                case 1:
                    result += "your entry point";
                    break;
                case 2:
                    result += "your opponent's entry point";
                    break;
                case 4:
                    result += "the guarded exit";
                    break;
                default:
                    result += "an unknown exit point";
                    break;
                }
            }
            if (gems_needed > 0) {
                result += " with ";
                result += char('0' + gems_needed);
                result += " out of ";
                result += char('0' + num_gems);
                result += " gems";
            }
            result += ".";

            return result;
        }
    }

    void SendMenuSelections(Coercri::OutputByteBuf &buf, const MenuSelections &msel_old, const MenuSelections &msel_new)
    {
        for (std::map<std::string, MenuSelections::Sel>::const_iterator it = msel_new.selections.begin(); 
        it != msel_new.selections.end(); ++it) {
            std::map<std::string, MenuSelections::Sel>::const_iterator it_old 
                = msel_old.selections.find(it->first);
            const bool needs_update = (it_old == msel_old.selections.end() 
                                     || it_old->second.value != it->second.value
                                     || it_old->second.allowed_values != it->second.allowed_values);
            if (needs_update) {
                buf.writeUbyte(SERVER_SET_MENU_SELECTION);
                buf.writeString(it->first);
                buf.writeVarInt(it->second.value);
                buf.writeVarInt(it->second.allowed_values.size());
                for (std::vector<int>::const_iterator av = it->second.allowed_values.begin(); 
                av != it->second.allowed_values.end(); ++av) {
                    buf.writeVarInt(*av);
                }
            }
        }
    }

    void SendQuestDescription(Coercri::OutputByteBuf &buf, const std::string &descr_old, const std::string &descr_new)
    {
        if (descr_new != descr_old) {
            buf.writeUbyte(SERVER_SET_QUEST_DESCRIPTION);
            buf.writeString(descr_new);
        }
    }
    
    void SendJoinGameAccepted(KnightsGameImpl &impl, Coercri::OutputByteBuf &buf,
                              int my_player_num)
    {
        // Send JOIN_GAME_ACCEPTED
        buf.writeUbyte(SERVER_JOIN_GAME_ACCEPTED);
        {
            std::vector<const Graphic*> graphics;
            impl.knights_config->getGraphics(graphics);
            buf.writeVarInt(graphics.size());
            for (int i = 0; i < graphics.size(); ++i) {
                ASSERT(graphics[i]->getID() == i+1);
                graphics[i]->serialize(buf);
            }
        }
        {
            std::vector<const Anim*> anims;
            impl.knights_config->getAnims(anims);
            buf.writeVarInt(anims.size());
            for (int i = 0; i < anims.size(); ++i) {
                ASSERT(anims[i]->getID() == i+1);
                anims[i]->serialize(buf);
            }
        }
        {
            std::vector<const Overlay*> overlays;
            impl.knights_config->getOverlays(overlays);
            buf.writeVarInt(overlays.size());
            for (int i = 0; i < overlays.size(); ++i) {
                ASSERT(overlays[i]->getID() == i+1);
                overlays[i]->serialize(buf);
            }
        }
        {
            std::vector<const Sound*> sounds;
            impl.knights_config->getSounds(sounds);
            buf.writeVarInt(sounds.size());
            for (int i = 0; i < sounds.size(); ++i) {
                ASSERT(sounds[i]->getID() == i+1);
                sounds[i]->serialize(buf);
            }
        }
        {
            std::vector<const UserControl*> controls;
            impl.knights_config->getStandardControls(controls);
            const int num_std_ctrls = controls.size();
            buf.writeVarInt(num_std_ctrls);
            for (int i = 0; i < num_std_ctrls; ++i) {
                ASSERT(controls[i]->getID() == i+1);
                controls[i]->serialize(buf);
            }
            impl.knights_config->getOtherControls(controls);
            buf.writeVarInt(controls.size());
            for (int i = 0; i < controls.size(); ++i) {
                ASSERT(controls[i]->getID() == i+1+num_std_ctrls);
                controls[i]->serialize(buf);
            }
        }
        impl.knights_config->getMenu().serialize(buf);
        buf.writeVarInt(impl.knights_config->getApproachOffset());

        // tell him what player num he has been assigned
        buf.writeVarInt(my_player_num);

        // tell him who players 0 and 1 are
        int np = 0;
        for (int pnum = 0; pnum < 2; ++pnum) {
            std::string name;
            bool ready = false;

            int house_colour = 0;
            
            for (game_conn_vector::const_iterator it = impl.connections.begin(); it != impl.connections.end(); ++it) {
                if ((*it)->player_num == pnum) {
                    name = (*it)->name;
                    ready = (*it)->is_ready;
                    house_colour = (*it)->house_colour;
                    ++np;
                    break;
                } else if (!(*it)->name2.empty() && (*it)->player_num + 1 == pnum) {
                    name = (*it)->name2;
                    ready = (*it)->is_ready;
                    ++np;
                    break;
                }
            }
            buf.writeString(name);
            buf.writeUbyte(ready ? 1 : 0);
            buf.writeUbyte(house_colour);
        }

        // tell him who the observers are
        // std::max necessary because connections.size()==1 but np==2 in split screen mode.
        buf.writeVarInt(std::max(0, int(impl.connections.size()) - np));
        for (game_conn_vector::const_iterator it = impl.connections.begin(); it != impl.connections.end(); ++it) {
            if ((*it)->player_num == -1) {
                ASSERT((*it)->name2.empty());
                buf.writeString((*it)->name);
            }
        }
        
        // Send the current menu selections
        SendMenuSelections(buf, MenuSelections(), impl.menu_selections);
        SendQuestDescription(buf, "", impl.quest_description);

        // Send the available knight house colours
        std::vector<Coercri::Color> hse_cols;
        impl.knights_config->getHouseColours(hse_cols);
        buf.writeUbyte(SERVER_SET_AVAILABLE_HOUSE_COLOURS);
        buf.writeUbyte(hse_cols.size());
        for (int i = 0; i < hse_cols.size(); ++i) {
            buf.writeUbyte(hse_cols[i].r);
            buf.writeUbyte(hse_cols[i].g);
            buf.writeUbyte(hse_cols[i].b);
        }
    }
    
    void SetMenuSelection(KnightsGameImpl &impl, const std::string &key, int value)
    {
        // If it's an invalid key then ignore it
        if (impl.menu_selections.selections.find(key) == impl.menu_selections.selections.end()) return;
        
        // Save the old settings
        MenuSelections msel_old = impl.menu_selections;
        std::string quest_descr_old = impl.quest_description;

        // If they've changed a non-quest item then change the quest to CUSTOM
        if (key != "quest" && impl.menu_selections.getValue(key) != value) {
            impl.menu_selections.setValue("quest", 0);
        }
        
        // Set the new setting
        impl.menu_selections.setValue(key, value);
        
        // Apply menu constraints
        impl.knights_config->getMenuConstraints().apply(impl.knights_config->getMenu(), impl.menu_selections);

        // Update the quest description
        impl.quest_description = GenerateQuestDescription(impl.menu_selections, *impl.knights_config);
        
        // Broadcast new menu selections to all clients.
        for (game_conn_vector::iterator it = impl.connections.begin(); it != impl.connections.end(); ++it) {
            Coercri::OutputByteBuf buf((*it)->output_data);
            SendMenuSelections(buf, msel_old, impl.menu_selections);
            SendQuestDescription(buf, quest_descr_old, impl.quest_description);
        }
    }

    void ResetMenuSelections(KnightsGameImpl &impl)
    {
        // set all menu selections to zero
        for (int i = 0; i < impl.knights_config->getMenu().getNumItems(); ++i) {
            const std::string & key = impl.knights_config->getMenu().getItem(i).getKey();
            impl.menu_selections.setValue(key, 0);
        }

        // now explicitly set "quest" to 1, this will also ensure that menu constraints get set correctly.
        SetMenuSelection(impl, "quest", 1);

        // Update the quest description also
        impl.quest_description = GenerateQuestDescription(impl.menu_selections, *impl.knights_config);
    }
    
    class UpdateThread {
    public:
        explicit UpdateThread(KnightsGameImpl &kg_, boost::shared_ptr<Coercri::Timer> timer_)
            : kg(kg_),
              timer(timer_),
              game_over_sent(false)
        { }

        void operator()()
        {
            try {
            
                {
                    // Initialize the game. (Lock kg while we do this, so we can read the knights_config & menu safely.)
                    boost::lock_guard<boost::mutex> lock(kg.my_mutex);

                    int hc0 = 0, hc1 = 1;
                    for (game_conn_vector::const_iterator it = kg.connections.begin(); it != kg.connections.end(); ++it) {
                        if ((*it)->player_num == 0) hc0 = (*it)->house_colour;
                        if ((*it)->player_num == 1) hc1 = (*it)->house_colour;
                    }

                    engine.reset(new KnightsEngine(kg.knights_config, kg.menu_selections, hc0, hc1));
                    callbacks.reset(new ServerCallbacks);
                }
                
                // Wait until all players have set the 'finished_loading' flag
                while (1) {
                    {
                        boost::lock_guard<boost::mutex> lock(kg.my_mutex);
                        bool all_loaded = true;
                        for (game_conn_vector::const_iterator it = kg.connections.begin(); it != kg.connections.end(); ++it) {
                            if ((*it)->player_num != -1 && !(*it)->finished_loading) {
                                all_loaded = false;
                                break;
                            }
                        }
                        if (all_loaded) break;
                    }
                    boost::this_thread::sleep(boost::posix_time::milliseconds(100));
                }
                
                // Go into game loop. NOTE: This will run forever until the main thread interrupts us
                // (or until an exception occurs).
                int last_time = timer->getMsec();
                while (1) {
                    // work out how long since the last update
                    const int new_time = timer->getMsec();
                    const int time_delta = new_time - last_time;

                    // only run an update if delta is min_time_delta or more.
                    // if > max_time_delta then only run an update of max_time_delta, and 'drop'
                    // the rest of the time (ie run more slowly than real-time).
                    const int min_time_delta = 50;
                    const int max_time_delta = 200;
                    if (time_delta > min_time_delta) {
                        update(std::min(time_delta, max_time_delta));
                        last_time += time_delta;
                    }

                    // work out how long until the next update (taking
                    // into account how long the update itself took).
                    const int time_since_update = timer->getMsec() - last_time;
                    const int time_to_wait = std::max(0, min_time_delta - time_since_update) + 1;
                    boost::this_thread::sleep(boost::posix_time::milliseconds(time_to_wait));
                }
                
            } catch (boost::thread_interrupted &) {
                // Allow this to go through. The code that interrupted us knows what it's doing...
                
            } catch (std::exception &e) {
                sendError(e.what());
            } catch (...) {
                sendError("Unknown error in update thread");
            }

            // Before we go, delete the KnightsEngine. This will make sure that the Mediator
            // is still around while the KnightsEngine destructor runs.
            try {
                engine.reset();
            } catch (...) {
                sendError("Error during KnightsEngine shutdown");
            }
        }

        void sendError(const char * msg) throw()
        {
            try {
                // send error to all players connected to this game.
                boost::lock_guard<boost::mutex> lock(kg.my_mutex);
                for (game_conn_vector::const_iterator it = kg.connections.begin(); it != kg.connections.end(); ++it) {
                    Coercri::OutputByteBuf buf((*it)->output_data);
                    buf.writeUbyte(SERVER_ERROR);
                    buf.writeString(msg);
                }
                // log the error as well.
                if (kg.knights_log) {
                    kg.knights_log->logMessage(kg.game_name + "\tError in update thread\t" + msg);
                }
            } catch (...) {
                // Disregard any exceptions here, as we don't want them to propagate up into the
                // boost::thread library internals (this would terminate the entire server if it happened).
            }
        }
        
        void update(int time_delta)
        {
            // read controls (and see if paused)
            {
                boost::lock_guard<boost::mutex> lock(kg.my_mutex);

                // we assume that allow_split_screen means this is a local game
                // (and therefore pause should be allowed)
                if (kg.pause_mode && kg.allow_split_screen) {
                    return;  // ignore the update
                }
                
                for (game_conn_vector::const_iterator it = kg.connections.begin(); it != kg.connections.end(); ++it) {
                    if ((*it)->player_num != -1) {
                        for (int p = 0; p < ((*it)->name2.empty() ? 1 : 2); ++p) {

                            const UserControl *final_ctrl = 0;

                            for (std::vector<const UserControl*>::const_iterator c = (*it)->control_queue[p].begin();
                            c != (*it)->control_queue[p].end(); ++c) {
                                final_ctrl = *c;
                                engine->setControl((*it)->player_num + p, *c);
                            }
                            (*it)->control_queue[p].clear();

                            // if the final control is continuous then re-insert it into the queue for next time.
                            if (final_ctrl && final_ctrl->isContinuous()) {
                                (*it)->control_queue[p].push_back(final_ctrl);
                            }
                        }
                    }
                }
            }
            
            // run the update
            engine->update(time_delta, *callbacks);

            // copy the results back to the GameConnection buffers
            {
                boost::lock_guard<boost::mutex> lock(kg.my_mutex);
                for (game_conn_vector::iterator it = kg.connections.begin(); it != kg.connections.end(); ++it) {
                    if ((*it)->player_num == -1) {
                        // TODO : The following line should execute iff allow_observers is true.
                        // (Currently we are not allowing observation at all so it is commented out.)
                        //callbacks->appendObserverCmds((*it)->output_data);
                    } else {
                        const bool split_screen = !(*it)->name2.empty();
                        if (split_screen) {
                            // Select player 0 for split screen mode
                            (*it)->output_data.push_back(SERVER_SWITCH_PLAYER);
                            (*it)->output_data.push_back(0);
                        }

                        const int buf_size_before_p0 = (*it)->output_data.size();
                        callbacks->appendPlayerCmds((*it)->player_num, (*it)->output_data);

                        if (split_screen) {
                            if ((*it)->output_data.size() == buf_size_before_p0) {
                                // can remove the SWITCH_PLAYER 0 cmd
                                (*it)->output_data.pop_back();
                                (*it)->output_data.pop_back();
                            }

                            // Select player 1 for split screen mode
                            (*it)->output_data.push_back(SERVER_SWITCH_PLAYER);
                            (*it)->output_data.push_back(1);

                            // Send player 1's data
                            const int buf_size_before_p1 = (*it)->output_data.size();
                            callbacks->appendPlayerCmds((*it)->player_num + 1, (*it)->output_data);

                            if ((*it)->output_data.size() == buf_size_before_p1) {
                                // can remove the SWITCH_PLAYER 1 cmd
                                (*it)->output_data.pop_back();
                                (*it)->output_data.pop_back();
                            }
                        }
                    }
                }
                callbacks->clearCmds();
            }

            // poll to see if the game is over.
            if (!game_over_sent && callbacks->isGameOver()) {
                boost::lock_guard<boost::mutex> lock(kg.my_mutex);
                kg.game_over = true;
                game_over_sent = true;

                // find names of all players, and the winner
                std::vector<std::string> loser_names;
                std::string winner_name;
                for (game_conn_vector::const_iterator it = kg.connections.begin(); it != kg.connections.end(); ++it) {
                    const int pnum = (*it)->player_num;
                    if (pnum != -1) {
                        if (pnum == callbacks->getWinnerNum()) {
                            winner_name = (*it)->name;
                        } else {
                            loser_names.push_back((*it)->name);
                        }
                        // note: name2 not supported here currently.
                    }
                }

                kg.winner_name = winner_name;
                
                if (kg.knights_log) {
                    std::string msg;
                    msg = kg.game_name + "\tgame won\t";
                    // the first name printed is the winner, then we list the losers (of which there will be
                    // only one at the moment).
                    msg += "winner=" + winner_name;
                    for (std::vector<string>::const_iterator it = loser_names.begin(); it != loser_names.end(); ++it) {
                        msg += std::string(", loser=") + *it;
                    }
                    kg.knights_log->logMessage(msg);
                }
            }
        }

    private:
        KnightsGameImpl &kg;
        boost::shared_ptr<Coercri::Timer> timer;
        boost::shared_ptr<KnightsEngine> engine;
        boost::shared_ptr<ServerCallbacks> callbacks;
        bool game_over_sent;
    };
    
    void StartGameIfReady(KnightsGameImpl &kg)
    {
        // mutex is locked at this point

        std::vector<std::string> names;            
        
        // count how many players are ready to leave the lobby
        int nready = 0;
        for (game_conn_vector::const_iterator it = kg.connections.begin(); it != kg.connections.end(); ++it) {
            if ((*it)->player_num != -1 && (*it)->is_ready) {
                ++nready;
                names.push_back((*it)->name);
                if (!(*it)->name2.empty()) {
                    ++nready;
                    names.push_back((*it)->name2);
                }
            }
        }

        // if 2 players then we can start the game
        if (nready == 2) {
            // Clear 'finished_loading' for each player; assign player nums; send startGame msg
            for (game_conn_vector::iterator it = kg.connections.begin(); it != kg.connections.end(); ++it) {
                (*it)->finished_loading = false;
                (*it)->ready_to_end = false;

                const unsigned char num_displays =
                    ( !(*it)->name2.empty() || (*it)->player_num == -1 ) ? 2 : 1;

                // For now, don't allow observers to watch the game
                const bool allow_observers = false;
                if ((*it)->player_num != -1 || allow_observers) {
                    (*it)->output_data.push_back(SERVER_START_GAME);
                    (*it)->output_data.push_back(num_displays);
                } else {
                    // for players who can't watch the game, at least tell them that it has started.
                    Coercri::OutputByteBuf buf((*it)->output_data);
                    buf.writeUbyte(SERVER_ANNOUNCEMENT);
                    buf.writeString("Game is now running.");
                }

                // clear all ready flags when the game starts.
                (*it)->is_ready = false;
            }

            // Start the update thread.
            UpdateThread thr(kg, kg.timer);
            boost::thread new_thread(thr);
            kg.update_thread.swap(new_thread);  // start the sub-thread -- game is now running.

            // Log a message if required.
            if (kg.knights_log) {
                std::ostringstream str;
                str << kg.game_name << "\tgame started\t";
                for (std::vector<std::string>::const_iterator it = names.begin(); it != names.end(); ++it) {
                    if (it != names.begin()) str << ", ";
                    str << *it;
                }

                for (std::map<std::string, MenuSelections::Sel>::const_iterator it = kg.menu_selections.selections.begin();
                it != kg.menu_selections.selections.end(); ++it) {
                    // TODO: could try to convert this to a summary message rather than dumping all settings?
                    str << ", " << it->first << "=" << it->second.value;
                }
                
                kg.knights_log->logMessage(str.str());
            }
        }
    }

    void StopGameAndReturnToMenu(KnightsGameImpl &kg)
    {
        // Interrupt the thread and tell all players to return to menu.
        // Called at game over (after all players have clicked mouse to continue)
        // or when somebody presses escape.

        // NOTE: Mutex should be UNLOCKED when calling this routine, otherwise we will get a 
        // deadlock if the update thread is currently waiting for the mutex. (Apparently
        // a thread can't be interrupted while it is waiting for a mutex.)
        
        kg.update_thread.interrupt();
        kg.update_thread.join();
        for (game_conn_vector::iterator it = kg.connections.begin(); it != kg.connections.end(); ++it) {
            (*it)->output_data.push_back(SERVER_GOTO_MENU);
        }

        // Find the winner's name if applicable, and send it to all players
        if (!kg.winner_name.empty()) {
            const std::string msg = "Game ended. " + kg.winner_name + " was the winner.";
            for (game_conn_vector::iterator it = kg.connections.begin(); it != kg.connections.end(); ++it) {
                Coercri::OutputByteBuf buf((*it)->output_data);
                buf.writeUbyte(SERVER_ANNOUNCEMENT);
                buf.writeString(msg);
            }
        }

        kg.winner_name = "";
        
        // Log end of game.
        if (kg.knights_log) {
            kg.knights_log->logMessage(kg.game_name + "\tgame ended");
        }
    }
}

KnightsGame::KnightsGame(boost::shared_ptr<const KnightsConfig> config,
                         boost::shared_ptr<Coercri::Timer> tmr,
                         bool allow_split_screen,
                         KnightsLog *knights_log,
                         const std::string &game_name)
    : pimpl(new KnightsGameImpl)
{
    pimpl->knights_config = config;
    pimpl->timer = tmr;
    pimpl->allow_split_screen = allow_split_screen;
    pimpl->game_over = false;
    pimpl->pause_mode = false;
    
    // initialize all menu settings to 0
    ResetMenuSelections(*pimpl);

    // set up our own controls vector.
    config->getStandardControls(pimpl->controls);
    std::vector<const UserControl*> other_ctrls;
    config->getOtherControls(other_ctrls);
    pimpl->controls.insert(pimpl->controls.end(), other_ctrls.begin(), other_ctrls.end());

    pimpl->knights_log = knights_log;
    pimpl->game_name = game_name;
}

KnightsGame::~KnightsGame()
{
    // If the update thread is still running then we have to kill it
    // here. The thread dtor will not kill the thread for us, instead
    // the thread will just become "detached" and continue running,
    // with disastrous consequences if the KnightsGame object is being
    // destroyed!
    pimpl->update_thread.interrupt();
    pimpl->update_thread.join();
}

int KnightsGame::getNumPlayers() const
{
    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
    int n = 0;
    for (game_conn_vector::const_iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
        if ((*it)->player_num != -1) ++n;
    }
    return n;
}

int KnightsGame::getNumObservers() const
{
    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
    int n = 0;
    for (game_conn_vector::const_iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
        if ((*it)->player_num == -1) ++n;
    }
    return n;
}

void KnightsGame::getPlayerNames(std::string &p0, std::string &p1)
{
    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
    for (game_conn_vector::const_iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
        if ((*it)->player_num == 0) {
            p0 = (*it)->name;
        } else if ((*it)->player_num == 1) {
            p1 = (*it)->name;
        }
    }
}

bool KnightsGame::isSplitScreenAllowed() const
{
    // sub thread won't be changing allow_split_screen so don't need to lock
    return pimpl->allow_split_screen;
}

GameStatus KnightsGame::getStatus() const
{
    const int nplayers = getNumPlayers();
    if (nplayers < 2) return GS_WAITING_FOR_PLAYERS;
    if (pimpl->update_thread.joinable()) return GS_RUNNING;
    else return GS_SELECTING_QUEST;
}

GameConnection & KnightsGame::newClientConnection(const std::string &client_name, const std::string &client_name_2, int client_version)
{
    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);

    // check preconditions. (caller should have checked these already.)
    if (client_name.empty()) throw UnexpectedError("invalid client name");
    for (game_conn_vector::const_iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
        if ((*it)->name == client_name) throw UnexpectedError("Duplicate client name");
        if (!client_name_2.empty() && (*it)->name == client_name_2) throw UnexpectedError("Duplicate client name");
    }
    if (!pimpl->allow_split_screen && !client_name_2.empty()) throw UnexpectedError("Split screen mode not allowed");
    if (!client_name_2.empty() && !pimpl->connections.empty()) throw UnexpectedError("Cannot join in split screen mode while connections exist");

    // determine player num.
    // for now: the first two to connect are players, the rest are observers.
    bool pnum_taken[2] = { false, false };
    int new_pnum = -1;
    for (game_conn_vector::const_iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
        if ((*it)->player_num == 0) pnum_taken[0] = true;
        if ((*it)->player_num == 1) pnum_taken[1] = true;
    }
    if (!pnum_taken[0]) new_pnum = 0;
    else if (!pnum_taken[1]) new_pnum = 1;

    // create the GameConnection
    boost::shared_ptr<GameConnection> conn(new GameConnection(client_name, client_name_2, new_pnum, client_version));
    pimpl->connections.push_back(conn);

    // modify the house colour if necessary.
    if (new_pnum == -1) {
        conn->house_colour = 0;
    } else {
        bool col_ok = false;
        while (!col_ok) {
            col_ok = true;
            for (game_conn_vector::const_iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
                if (*it != conn && (*it)->house_colour == conn->house_colour && (*it)->player_num != -1) {
                    col_ok = false;
                    // try the next colour
                    conn->house_colour ++;
                    std::vector<Coercri::Color> hse_cols;
                    pimpl->knights_config->getHouseColours(hse_cols);  // a bit wasteful as we only want the size. never mind.
                    if (conn->house_colour >= hse_cols.size()) conn->house_colour = 0;
                    break;
                }
            }
        }
    }

    // Send the SERVER_JOIN_GAME_ACCEPTED message (includes initial configuration messages e.g. menu settings)
    Coercri::OutputByteBuf buf(conn->output_data);
    SendJoinGameAccepted(*pimpl, buf, new_pnum);

    // Send any necessary SERVER_PLAYER_JOINED_THIS_GAME messages (but not to the player who just joined).
    for (game_conn_vector::iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
        if (*it != conn && (new_pnum != -1 || (*it)->client_version > 9)) {   // version 9 crashes if you send joined-this-game msgs for observers!
            Coercri::OutputByteBuf out((*it)->output_data);
            out.writeUbyte(SERVER_PLAYER_JOINED_THIS_GAME);
            out.writeString(client_name);
            out.writeVarInt(new_pnum);
            out.writeUbyte(conn->house_colour);
            if (!client_name_2.empty()) {
                out.writeUbyte(SERVER_PLAYER_JOINED_THIS_GAME);
                out.writeString(client_name_2);
                out.writeVarInt(new_pnum);
                out.writeUbyte(conn->house_colour);
            }
        }
    }

    // If the new player is an observer, send him a quick message explaining why he cannot join yet.
    if (pnum_taken[0] && pnum_taken[1]) {
        buf.writeUbyte(SERVER_ANNOUNCEMENT);
        buf.writeString("This game already has two players. You will not be able to join "
                        "until one of the players leaves. However, you can still chat to the "
                        "players if you wish.");
        buf.writeUbyte(SERVER_ANNOUNCEMENT);
        buf.writeString("--");
    }    
                
    return *conn;
}

void KnightsGame::clientLeftGame(GameConnection &conn)
{
    // This is called when a player has left the game (either because
    // KnightsServer sent him a SERVER_LEAVE_GAME msg, or because he
    // disconnected.)

    bool is_player = false;

    std::string name, name2;

    {
        boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
        
        // Find the client in our connections list. 
        game_conn_vector::iterator where = std::find_if(pimpl->connections.begin(), 
                                                        pimpl->connections.end(), ShPtrEq<GameConnection>(&conn));
        ASSERT(where != pimpl->connections.end());
        
        // Determine whether he is a player or observer. Also save his name(s)
        is_player = (*where)->player_num != -1;
        name = (*where)->name;
        name2 = (*where)->name2;
        
        // Erase the connection from our list. Note 'conn' is invalid from now on
        pimpl->connections.erase(where);
    }
        
    if (is_player && pimpl->update_thread.joinable()) {
        // The leaving client is one of the players (as opposed to an observer).
        // The game will now end.
        StopGameAndReturnToMenu(*pimpl);
    }

    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
    
    // Send SERVER_PLAYER_LEFT_THIS_GAME message to all remaining players
    for (game_conn_vector::iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
        if (is_player || (*it)->client_version > 9) { // version 9 didn't handle join/leave msgs about observers very well.
            Coercri::OutputByteBuf buf((*it)->output_data);
            buf.writeUbyte(SERVER_PLAYER_LEFT_THIS_GAME);
            buf.writeString(name);
            if (!name2.empty()) {
                buf.writeUbyte(SERVER_PLAYER_LEFT_THIS_GAME);
                buf.writeString(name2);
            }
        }
    }

    // If there are no connections left then reset the menu selections.
    if (pimpl->connections.empty()) {
        ResetMenuSelections(*pimpl);
    }
}

void KnightsGame::sendChatMessage(GameConnection &conn, const std::string &msg)
{
    // Forward the message to everybody (including the originator)
    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
    for (game_conn_vector::iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
        Coercri::OutputByteBuf buf((*it)->output_data);
        buf.writeUbyte(SERVER_CHAT);
        buf.writeString(conn.name);
        buf.writeUbyte(conn.player_num == -1 ? 2 : 1);  // 2 means observer, 1 means player.
        buf.writeString(msg);
    }
}

void KnightsGame::setReady(GameConnection &conn, bool ready)
{
    if (pimpl->update_thread.joinable()) return;  // Game is running.
    
    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
    if (conn.player_num != -1) {
        conn.is_ready = ready;

        // send notification to players
        for (game_conn_vector::iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
            Coercri::OutputByteBuf out((*it)->output_data);
            out.writeUbyte(SERVER_SET_READY);
            out.writeString(conn.name);
            out.writeUbyte(ready ? 1 : 0);
            if (!conn.name2.empty()) {
                out.writeUbyte(SERVER_SET_READY);
                out.writeString(conn.name2);
                out.writeUbyte(ready ? 1 : 0);
            }
        }

        // Start the game if two players are now ready.
        StartGameIfReady(*pimpl);        
    }
}

void KnightsGame::setHouseColour(GameConnection &conn, int hse_col)
{
    if (pimpl->update_thread.joinable()) return;  // Game is running

    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
    if (conn.player_num != -1) {

        // check that this colour is available.
        for (game_conn_vector::iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
            if ((*it)->player_num != -1 && hse_col == (*it)->house_colour) {
                // not available. set hse_col back to the original house_colour
                hse_col = conn.house_colour;
                break;
            }
        }
        
        conn.house_colour = hse_col;

        // send notification to players
        for (game_conn_vector::iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
            Coercri::OutputByteBuf out((*it)->output_data);
            out.writeUbyte(SERVER_SET_HOUSE_COLOUR);
            // note we don't support house colours in split screen mode at the moment. we assume it's the first player
            // on the connection who is having the house colours set.
            out.writeString(conn.name);
            out.writeUbyte(hse_col);
        }
    }
}
    
void KnightsGame::finishedLoading(GameConnection &conn)
{
    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
    if (conn.player_num != -1) {
        conn.finished_loading = true;
    }
}

void KnightsGame::readyToEnd(GameConnection &conn)
{
    if (!pimpl->game_over) return;
    
    bool all_ready = false;
    {
        boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
        if (conn.player_num != -1) {
            conn.ready_to_end = true;
            all_ready = true;
            for (game_conn_vector::const_iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
                if ((*it)->player_num != -1 && !(*it)->ready_to_end) {
                    all_ready = false;
                    break;
                }
            }
        }
    }
    
    if (all_ready) {
        StopGameAndReturnToMenu(*pimpl);
    }
}

void KnightsGame::requestQuit(GameConnection &conn)
{
    if (!pimpl->update_thread.joinable()) return;  // Game is not running
    
    int pnum;
    {
        boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
        pnum = conn.player_num;
    }
    if (pnum != -1) {

        if (pimpl->knights_log) {
            pimpl->knights_log->logMessage(pimpl->game_name + "\tquit requested\t" + conn.name);
        }
        
        StopGameAndReturnToMenu(*pimpl);

        // tell all players what happened.
        for (game_conn_vector::iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
            Coercri::OutputByteBuf buf((*it)->output_data);
            buf.writeUbyte(SERVER_ANNOUNCEMENT);
            buf.writeString(conn.name + " pressed Escape.");
        }
    }
}

void KnightsGame::setPauseMode(bool pm)
{
    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
    pimpl->pause_mode = pm;
}

void KnightsGame::setMenuSelection(GameConnection &conn, const std::string &key, int value)
{
    if (pimpl->update_thread.joinable()) return;  // Game is running
    
    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
    if (conn.player_num == -1) return;   // only players can adjust the menu.

    SetMenuSelection(*pimpl, key, value);
}

void KnightsGame::sendControl(GameConnection &conn, int p, unsigned char control_num)
{
    if (!pimpl->update_thread.joinable()) return; // Game is not running
    
    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
    if (conn.player_num != -1 && p >= 0 && p < (conn.name2.empty() ? 1 : 2)) {
        const UserControl * control = control_num == 0 ? 0 : pimpl->controls.at(control_num - 1);
        conn.control_queue[p].push_back(control);
    }
}

void KnightsGame::changePlayerNum(GameConnection &conn, int new_num)
{
    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);

    if (conn.player_num == new_num) return;  // Nothing to do
    
    if (new_num != -1) {
        // we have to check whether the requested num is free.

        bool free = true;
        if (new_num < 0 || new_num > 1) free = false;   // illegal player num

        for (game_conn_vector::iterator it = pimpl->connections.begin(); free && it != pimpl->connections.end(); ++it) {
            if ((*it)->player_num == new_num) free = false;
        }

        if (!free) {
            // send him an 'announcement' telling him that the seat is taken.
            Coercri::OutputByteBuf buf(conn.output_data);
            buf.writeUbyte(SERVER_ANNOUNCEMENT);
            buf.writeString("Cannot join - this position has been taken by another player.");
            return;
        }
    }

    // The move is approved.

    // changing player num does not apply to split screen games, so we can ignore name2 and just use name.
    const std::string & my_name = conn.name;

    // Assign him a house colour.
    int new_col = 0;
    if (new_num >= 0 && new_num <= 1) {
        for (game_conn_vector::iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
            if ((*it)->player_num != -1 && (*it)->house_colour == 0) {
                new_col = 1;
                break;
            }
        }
    }

    // Update our data, then send announcements to all players.
    conn.player_num = new_num;
    conn.house_colour = new_col;
    for (game_conn_vector::iterator it = pimpl->connections.begin(); it != pimpl->connections.end(); ++it) {
        const bool me = (it->get() == &conn);
        Coercri::OutputByteBuf buf((*it)->output_data);
        if ((*it)->client_version > 9) { // SERVER_SET_PLAYER_NUM only available from version 010
            buf.writeUbyte(SERVER_SET_PLAYER_NUM);
            buf.writeString(my_name);
            buf.writeVarInt(new_num);
            buf.writeUbyte(me ? 1 : 0);
        } else {
            // for version 009 clients, fake it using SERVER_PLAYER_JOINED_THIS_GAME / SERVER_PLAYER_LEFT_THIS_GAME
            if (new_num >= 0) {
                buf.writeUbyte(SERVER_PLAYER_JOINED_THIS_GAME);
                buf.writeString(my_name);
                buf.writeVarInt(new_num);
                buf.writeUbyte(new_col);
            } else {
                buf.writeUbyte(SERVER_PLAYER_LEFT_THIS_GAME);
                buf.writeString(my_name);
            }
        }
        buf.writeUbyte(SERVER_SET_HOUSE_COLOUR);
        buf.writeString(my_name);
        buf.writeUbyte(new_col);
    }
}

void KnightsGame::getOutputData(GameConnection &conn, std::vector<unsigned char> &data)
{
    boost::lock_guard<boost::mutex> lock(pimpl->my_mutex);
    std::swap(conn.output_data, data);
    conn.output_data.clear();
}
