/*
 * rng.hpp
 *
 * Random number generators.
 *
 * getInt, getFloat return number in the range [a,b).
 *
 * TODO: allow save/restore of internal state
 * TODO: improve the seeding algorithm; it seems a waste to use MT but then only use time(0) as the seed!
 * 
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef RNG_HPP
#define RNG_HPP

#include "boost/random/mersenne_twister.hpp"
#include "boost/thread/mutex.hpp"

#ifdef min
#undef min
#endif
#ifdef max
#undef max
#endif

class RNG {
public:
    virtual ~RNG() { }

    virtual void seedFromTime() = 0;
    virtual float getU01() = 0;

    bool getBool(float p = 0.5f) { return getU01() <= p; }
    int getInt(int a, int b) { return a + int(getU01() * (b-a)); }
    float getFloat(float a, float b) { return a + getU01() * (b-a); }
};

// for std::random_shuffle compatibility
class RNG_Wrapper {
public:
    RNG_Wrapper(RNG &r_) : r(r_) { }
    int operator()(int n) { return r.getInt(0, n); }
private:
    RNG &r;
};



// Implementations of RNG

// concrete RNG - using rand/srand
// NOT necessarily thread safe.
class StandardRNG : public RNG {
public:
    StandardRNG() { seedFromTime(); }
    virtual void seedFromTime();
    virtual float getU01();
};

// concrete RNG - Mersenne Twister from boost
// Thread safe.
class MersenneRNG : public RNG {
public:
    MersenneRNG() { seedFromTime(); }
    virtual void seedFromTime();
    virtual float getU01();
private:
    typedef boost::mt19937 gen_type;
    gen_type gen;
    boost::mutex mutex;
};

// global access point
extern MersenneRNG g_rng;

#endif      // RNG_HPP
