/*
 * title_screen.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "config_map.hpp"
#include "credits_screen.hpp"
#include "find_server_screen.hpp"
#include "knights_app.hpp"
#include "loading_screen.hpp"
#include "options_screen.hpp"
#include "title_screen.hpp"

#include "gui_button.hpp"
#include "gui_centre.hpp"
#include "gui_panel.hpp"

#include "guichan.hpp"

#include "boost/scoped_ptr.hpp"
using namespace boost;

class TitleScreenImpl : public gcn::ActionListener {
public:
    TitleScreenImpl(KnightsApp &app, gcn::Gui &gui);
    void action(const gcn::ActionEvent &event);
    
private:
    KnightsApp &knights_app;
    scoped_ptr<GuiCentre> centre;
    scoped_ptr<gcn::Container> outer_container;
    scoped_ptr<GuiPanel> panel1, panel2;
    scoped_ptr<gcn::Container> container;
    scoped_ptr<gcn::Label> title;
    scoped_ptr<gcn::Button> start_new_game;
    scoped_ptr<gcn::Button> host_lan_game;
    scoped_ptr<gcn::Button> join_lan_game;
    scoped_ptr<gcn::Button> join_internet_game;    
    scoped_ptr<gcn::Button> options;
    scoped_ptr<gcn::Button> credits;
    scoped_ptr<gcn::Button> exit;
};

TitleScreenImpl::TitleScreenImpl(KnightsApp &app, gcn::Gui &gui)
    : knights_app(app)
{
    const int w = 300, h = 40, vspace = 15, pad = 15, vspace2 = 40;
    
    container.reset(new gcn::Container);
    container->setOpaque(false);
    
    const int x = pad;
    int y = pad;
    const int yinc = h + vspace;
    
    start_new_game.reset(new GuiButton("Start Split Screen Game"));
    start_new_game->setSize(w,h);
    start_new_game->addActionListener(this);
    container->add(start_new_game.get(), x, y);
    y += yinc;
    
    join_internet_game.reset(new GuiButton("Connect to Server"));
    join_internet_game->setSize(w,h);
    join_internet_game->addActionListener(this);
    container->add(join_internet_game.get(), x, y);
    y += yinc;
    
    host_lan_game.reset(new GuiButton("Host LAN Game"));
    host_lan_game->setSize(w,h);
    host_lan_game->addActionListener(this);
    container->add(host_lan_game.get(), x, y);
    y += yinc;
    
    join_lan_game.reset(new GuiButton("Join LAN Game"));
    join_lan_game->setSize(w,h);
    join_lan_game->addActionListener(this);
    container->add(join_lan_game.get(), x, y);
    y += yinc;

    options.reset(new GuiButton("Options"));
    options->setSize(w, h);
    options->addActionListener(this);
    container->add(options.get(), x, y);
    y += yinc;

    credits.reset(new GuiButton("Credits"));
    credits->setSize(w, h);
    credits->addActionListener(this);
    container->add(credits.get(), x, y);
    y += yinc;
    
    exit.reset(new GuiButton("Quit"));
    exit->setSize(w,h);
    exit->addActionListener(this);
    container->add(exit.get(), x, y);
    y += yinc;
    
    container->setSize(w + 2*pad, y - vspace + pad);
    panel2.reset(new GuiPanel(container.get()));

    title.reset(new gcn::Label("KNIGHTS"));
    title->setWidth(w+2*pad);
    title->setHeight(title->getHeight() + 2*pad);
    title->setAlignment(gcn::Graphics::CENTER);
    panel1.reset(new GuiPanel(title.get()));

    outer_container.reset(new gcn::Container);
    outer_container->setOpaque(false);
    outer_container->add(panel1.get(), 0, 0);
    outer_container->add(panel2.get(), 0, panel1->getHeight() + vspace2);
    outer_container->setSize(panel2->getWidth(), panel1->getHeight() + vspace2 + panel2->getHeight());

    centre.reset(new GuiCentre(outer_container.get()));
    gui.setTop(centre.get());
}

void TitleScreenImpl::action(const gcn::ActionEvent &event)
{
    auto_ptr<Screen> new_screen;

    if (event.getSource() == start_new_game.get()) {
        // Go to LoadingScreen in Local mode
        new_screen.reset(new LoadingScreen(-1));
        
    } else if (event.getSource() == host_lan_game.get()) {
        // Go to LoadingScreen in LAN mode
        new_screen.reset(new LoadingScreen(knights_app.getConfigMap().getInt("port_number")));

    } else if (event.getSource() == join_lan_game.get()) {
        // Go to FindServerScreen in LAN mode.
        new_screen.reset(new FindServerScreen("Join LAN Game", false));

    } else if (event.getSource() == join_internet_game.get()) {
        // Go to FindServerScreen in Internet mode.
        new_screen.reset(new FindServerScreen("Connect to Server", true));

    } else if (event.getSource() == options.get()) {
        new_screen.reset(new OptionsScreen);
    } else if (event.getSource() == credits.get()) {
        new_screen.reset(new CreditsScreen);
    } else if (event.getSource() == exit.get()) {
        knights_app.requestQuit();
    }
    
    if (new_screen.get()) {
        knights_app.requestScreenChange(new_screen);
    }
}        

bool TitleScreen::start(KnightsApp &knights_app, shared_ptr<Coercri::Window> w, gcn::Gui &gui)
{
    knights_app.resetAll();
    pimpl.reset(new TitleScreenImpl(knights_app, gui));
    return true;
}
