/*
 * options_screen.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "options.hpp"

#include "knights_app.hpp"
#include "options_screen.hpp"
#include "title_screen.hpp"

#include "gui_button.hpp"
#include "gui_centre.hpp"
#include "gui_panel.hpp"

// coercri
#include "gcn/cg_font.hpp"
#include "gfx/window_listener.hpp"

#include "boost/scoped_ptr.hpp"

#include <sstream>

namespace {
    const char * control_names[] = { "UP", "DOWN", "LEFT", "RIGHT", "ACTION", "SUICIDE" };

    class ScalingListModel : public gcn::ListModel {
    public:
        std::string getElementAt(int i) {
            if (i==0) return "Scale2x";
            else return "Pixellated";
        }

        int getNumberOfElements() {
            return 2;
        }
    };

    class DisplayListModel : public gcn::ListModel {
    public:
        std::string getElementAt(int i) {
            if (i == 0) {
                return "Windowed";
            } else if (i == 1) {
                return "Full Screen";
            } else {
                return "";
            }
        }

        int getNumberOfElements() {
            return 2;
        }
    };
}

class OptionsScreenImpl : public gcn::ActionListener, public gcn::KeyListener, public Coercri::WindowListener {
public:
    OptionsScreenImpl(KnightsApp &app, gcn::Gui &gui);
    ~OptionsScreenImpl();
    void action(const gcn::ActionEvent &event);
    void keyPressed(gcn::KeyEvent &ke);
    void keyReleased(gcn::KeyEvent &ke);
    void onKey(Coercri::KeyEvent ke, Coercri::KeyCode kc, int character, int modifiers);

    void transferToGui();
    void setupChangeControls(int);
    
    boost::shared_ptr<Coercri::Window> window;

private:
    KnightsApp &knights_app;
    gcn::Gui &gui_ref;
    boost::scoped_ptr<GuiCentre> centre;
    boost::scoped_ptr<GuiPanel> panel;
    boost::scoped_ptr<gcn::Container> container;
    boost::scoped_ptr<gcn::Button> ok_button, cancel_button;
    boost::scoped_ptr<gcn::Label> control_label[2][7], extra_control_label;
    boost::scoped_ptr<gcn::Label> control_setting[2][6];
    boost::scoped_ptr<gcn::Button> change_button[2];
    boost::scoped_ptr<gcn::Label> change_label;
    boost::scoped_ptr<gcn::Font> red_font, blue_font;
    boost::scoped_ptr<gcn::Label> options_label, scaling_label, display_label;
    boost::scoped_ptr<gcn::ListModel> scaling_listmodel;
    boost::scoped_ptr<DisplayListModel> display_listmodel;
    boost::scoped_ptr<gcn::DropDown> scaling_dropdown, display_dropdown;
    boost::scoped_ptr<gcn::CheckBox> non_integer_checkbox;
    boost::scoped_ptr<gcn::Button> restore_button;

    Options current_opts;

    bool change_active;
    int change_key;
    int change_player;

    int desktop_width, desktop_height;
    bool previous_fullscreen;
};

OptionsScreenImpl::OptionsScreenImpl(KnightsApp &app, gcn::Gui &gui)
    : knights_app(app), gui_ref(gui), current_opts(app.getOptions()), change_active(false)
{
    app.getDesktopResolution(desktop_width, desktop_height);
    previous_fullscreen = current_opts.fullscreen;
    
    container.reset(new gcn::Container);
    container->setOpaque(false);

    const int pad = 15, pad_small = 5, extra_pad = 100;
    int y = pad;

    // Controls
    blue_font.reset(new Coercri::CGFont(knights_app.getFont(), Coercri::Color(0, 0, 128)));
    control_label[0][0].reset(new gcn::Label("Controls (Player 1)"));
    control_label[0][0]->setFont(blue_font.get());
    control_label[1][0].reset(new gcn::Label("Controls (Player 2 /"));
    control_label[1][0]->setFont(blue_font.get());
    extra_control_label.reset(new gcn::Label("Network Games)"));
    extra_control_label->setFont(blue_font.get());
    for (int i = 0; i < 2; ++i) {
        control_label[i][1].reset(new gcn::Label("Up:"));
        control_label[i][2].reset(new gcn::Label("Down:"));
        control_label[i][3].reset(new gcn::Label("Left:"));
        control_label[i][4].reset(new gcn::Label("Right:"));
        control_label[i][5].reset(new gcn::Label("Action:"));
        control_label[i][6].reset(new gcn::Label("Suicide:"));
        for (int j = 0; j < 6; ++j) {
            control_setting[i][j].reset(new gcn::Label);
        }
    }

    red_font.reset(new Coercri::CGFont(knights_app.getFont(), Coercri::Color(128, 0, 0)));
    change_label.reset(new gcn::Label);
    change_label->setVisible(false);
    change_label->setFont(red_font.get());

    const int cw0 = control_label[0][6]->getWidth() + 10;
    const int cw1 = control_label[0][0]->getFont()->getWidth("RIGHT WINDOWS");
    const int intercol = 35;
    const int overall_width = (pad + cw0 + cw1)*2 + intercol;

    container->add(control_label[0][0].get(), pad, y);
    container->add(control_label[1][0].get(), pad + cw0 + cw1 + intercol, y);
    y += control_label[0][0]->getHeight();
    container->add(extra_control_label.get(), pad + cw0 + cw1 + intercol, y);
    y += control_label[0][0]->getHeight() + 7;

    for (int i = 0; i < 6; ++i) {
        container->add(control_label[0][i+1].get(), pad, y);
        container->add(control_setting[0][i].get(), pad + cw0, y);
        container->add(control_label[1][i+1].get(), pad + cw0 + cw1 + intercol, y);
        container->add(control_setting[1][i].get(), pad + cw0 + cw1 + intercol + cw0, y);
        y += control_label[0][i+1]->getHeight() + pad_small;
    }

    y += pad_small;

    for (int i = 0; i < 2; ++i) {
        change_button[i].reset(new GuiButton("Change"));
        change_button[i]->addActionListener(this);
        container->add(change_button[i].get(), pad + i*(cw0+cw1+intercol), y);
    }
    container->add(change_label.get(), pad, y);

    y += change_button[0]->getHeight();
    y += 40;

    options_label.reset(new gcn::Label("Other Options"));
    options_label->setFont(blue_font.get());
    container->add(options_label.get(), pad, y);
    y += options_label->getHeight() + pad_small + 2;
    
    display_label.reset(new gcn::Label("Display Mode:"));
    scaling_label.reset(new gcn::Label("Graphics Scaling:"));
    const int ddx = pad + pad + scaling_label->getWidth();
    display_listmodel.reset(new DisplayListModel);
    display_dropdown.reset(new gcn::DropDown(display_listmodel.get()));
    display_dropdown->setWidth(overall_width - pad - extra_pad - ddx);
    display_dropdown->addActionListener(this);
    container->add(display_label.get(), pad, y);
    container->add(display_dropdown.get(), ddx, y);
    y += display_dropdown->getHeight() + pad_small + 2;

    scaling_listmodel.reset(new ScalingListModel);
    scaling_dropdown.reset(new gcn::DropDown(scaling_listmodel.get()));
    scaling_dropdown->setWidth(overall_width - pad - extra_pad - ddx);
    scaling_dropdown->addActionListener(this);
    container->add(scaling_label.get(), pad, y);
    container->add(scaling_dropdown.get(), ddx, y);
    y += scaling_dropdown->getHeight() + pad_small + 4;

    non_integer_checkbox.reset(new gcn::CheckBox("Allow Non-Integer Scaling"));
    non_integer_checkbox->addActionListener(this);
    container->add(non_integer_checkbox.get(), pad, y);
    y += non_integer_checkbox->getHeight() + 35;
    
    restore_button.reset(new GuiButton("Restore Defaults"));
    restore_button->addActionListener(this);
    container->add(restore_button.get(), overall_width/2 - restore_button->getWidth()/2, y);
    
    ok_button.reset(new GuiButton("OK"));
    ok_button->addActionListener(this);
    container->add(ok_button.get(), pad, y);

    cancel_button.reset(new GuiButton("Cancel"));
    cancel_button->addActionListener(this);
    container->add(cancel_button.get(), overall_width - pad - cancel_button->getWidth(), y);

    y += cancel_button->getHeight();
    y += pad;

    container->setSize(overall_width, y);
    panel.reset(new GuiPanel(container.get()));    
    centre.reset(new GuiCentre(panel.get()));
    gui.setTop(centre.get());

    gui.addGlobalKeyListener(this);
}

OptionsScreenImpl::~OptionsScreenImpl()
{
    gui_ref.removeGlobalKeyListener(this);
}

void OptionsScreenImpl::action(const gcn::ActionEvent &event)
{
    const bool got_ok = event.getSource() == ok_button.get();
    const bool got_cancel = event.getSource() == cancel_button.get();

    if (got_ok || got_cancel) {
        if (got_ok) {
            knights_app.setAndSaveOptions(current_opts);
            if (current_opts.fullscreen && !previous_fullscreen) {
                window->switchToFullScreen(desktop_width, desktop_height);
            } else if (!current_opts.fullscreen && previous_fullscreen) {
                int w, h;
                knights_app.getWindowedModeSize(w, h);
                window->switchToWindowed(w, h);
            }
        }
        auto_ptr<Screen> new_screen(new TitleScreen);
        knights_app.requestScreenChange(new_screen);
        return;
    }

    if (event.getSource() == change_button[0].get()) {
        setupChangeControls(0);
    } else if (event.getSource() == change_button[1].get()) {
        setupChangeControls(1);
    } else if (event.getSource() == scaling_dropdown.get()) {
        current_opts.use_scale2x = (scaling_dropdown->getSelected() == 0);
    } else if (event.getSource() == non_integer_checkbox.get()) {
        current_opts.allow_non_integer_scaling = non_integer_checkbox->isSelected();
    } else if (event.getSource() == display_dropdown.get()) {
        current_opts.fullscreen = (display_dropdown->getSelected() != 0);
    } else if (event.getSource() == restore_button.get()) {
        current_opts = Options();
        transferToGui();
    }
}

void OptionsScreenImpl::keyPressed(gcn::KeyEvent &ke)
{
    if (change_active) ke.consume();
}

void OptionsScreenImpl::keyReleased(gcn::KeyEvent &ke)
{
    if (change_active) ke.consume();
}

void OptionsScreenImpl::transferToGui()
{
    for (int i = 0; i < 2; ++i) {
        for (int j = 0; j < 6; ++j) {
            if (change_active && change_player == i && j >= change_key) {
                if (j == change_key) {
                    control_setting[i][j]->setCaption("<Press>");
                } else {
                    control_setting[i][j]->setCaption("");
                }
            } else {
                control_setting[i][j]->setCaption(Coercri::KeyName(current_opts.ctrls[i][j]));
            }
            control_setting[i][j]->adjustSize();
        }
    }
    if (change_active) {
        change_button[0]->setVisible(false);
        change_button[1]->setVisible(false);
        change_label->setVisible(true);
        change_label->setCaption(std::string("** Please press a key for ") + control_names[change_key] + " **");
        change_label->adjustSize();
    } else {
        change_button[0]->setVisible(true);
        change_button[1]->setVisible(true);
        change_label->setVisible(false);
    }
    scaling_dropdown->setSelected(1-int(current_opts.use_scale2x));
    non_integer_checkbox->setSelected(current_opts.allow_non_integer_scaling);
    display_dropdown->setSelected(current_opts.fullscreen ? 1 : 0);
    if (window) window->invalidateAll();
}

void OptionsScreenImpl::setupChangeControls(int i)
{
    change_active = true;
    change_player = i;
    change_key = 0;
    transferToGui();
}

void OptionsScreenImpl::onKey(Coercri::KeyEvent ke, Coercri::KeyCode kc, int character, int modifiers)
{
    if (ke == Coercri::KE_PRESSED && change_active) {
        current_opts.ctrls[change_player][change_key] = kc;
        ++change_key;
        if (change_key >= 6) {
            change_active = false;
        }
        transferToGui();
    }
}

bool OptionsScreen::start(KnightsApp &app, boost::shared_ptr<Coercri::Window> w, gcn::Gui &gui)
{
    pimpl.reset(new OptionsScreenImpl(app, gui));
    pimpl->transferToGui();
    w->addWindowListener(pimpl.get());
    pimpl->window = w;
    return true;
}

OptionsScreen::~OptionsScreen()
{
    if (pimpl->window) pimpl->window->rmWindowListener(pimpl.get());
}
