/*
 * menu_screen.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "adjust_list_box_size.hpp"
#include "config_map.hpp"
#include "game_manager.hpp"
#include "knights_app.hpp"
#include "knights_client.hpp"
#include "make_scroll_area.hpp"
#include "menu.hpp"
#include "menu_screen.hpp"
#include "title_screen.hpp"

#include "gui_button.hpp"
#include "gui_centre.hpp"
#include "gui_panel.hpp"
#include "gui_text_wrap.hpp"

#include "gcn/cg_font.hpp"

#include "boost/scoped_ptr.hpp"
#include <sstream>

namespace {

    std::string ColToText(const Coercri::Color &c)
    {
        std::string x;
        x += (unsigned char)(c.r);
        x += (unsigned char)(c.g);
        x += (unsigned char)(c.b);
        return x;
    }

    gcn::Color TextToCol(const std::string &x)
    {
        return gcn::Color((unsigned char)x[0], (unsigned char)x[1], (unsigned char)x[2]);
    }

    class HouseColourListModel : public gcn::ListModel {
    public:
        explicit HouseColourListModel(const GameManager &gm) : game_manager(gm) { }
        std::string getElementAt(int i) {
            if (i < 0 || i >= game_manager.getNumAvailHouseColours()) return "";
            else return ColToText(game_manager.getAvailHouseColour(i));
        }
        int getNumberOfElements() { return game_manager.getNumAvailHouseColours(); }
    private:
        const GameManager &game_manager;
    };

    // abuse of gcn::Font to draw the little coloured squares for the knight house colours.
    class HouseColourFont : public gcn::Font {
    public:
        virtual void drawString(gcn::Graphics *graphics, const std::string &text, int x, int y);
        int getHeight() const { return 20; }
        int getWidth(const std::string &) const { return 40; }
    };

    void HouseColourFont::drawString(gcn::Graphics *graphics, const std::string &text, int x, int y)
    {
        if (!graphics || text.size() != 3) return;

        gcn::Color col = TextToCol(text);
        const int w = getWidth(""), h = getHeight();

        graphics->setColor(gcn::Color(0,0,0));
        graphics->drawLine(x+1,   y+1,   x+w-2, y+1  );
        graphics->drawLine(x+w-2, y+1,   x+w-2, y+h-2);
        graphics->drawLine(x+w-2, y+h-2, x+1,   y+h-2);
        graphics->drawLine(x+1,   y+h-2, x+1,   y+1);
        graphics->setColor(col);
        graphics->fillRectangle(gcn::Rectangle(x+2, y+2, w-4, h-4));
    }

    HouseColourFont g_house_colour_font;
}

class MenuScreenImpl : public gcn::ActionListener {
public:
    MenuScreenImpl(boost::shared_ptr<KnightsClient> kc, bool extended_, bool show_obs_, KnightsApp &app, 
                   boost::shared_ptr<Coercri::Window> win, gcn::Gui &gui);
    ~MenuScreenImpl();
    void action(const gcn::ActionEvent &event);
    void updateGui();
    
    boost::shared_ptr<KnightsClient> knights_client;
    KnightsApp &knights_app;
    gcn::Gui &gui;
    boost::shared_ptr<Coercri::Window> window;

    boost::scoped_ptr<GuiCentre> centre;
    boost::scoped_ptr<GuiPanel> panel;
    boost::scoped_ptr<gcn::Container> container;
    boost::scoped_ptr<gcn::CheckBox> ready_checkbox;
    boost::scoped_ptr<GuiButton> start_button;
    boost::scoped_ptr<GuiButton> exit_button;
    boost::scoped_ptr<gcn::Label> label, title;
    boost::scoped_ptr<GuiTextWrap> quest_description_box;
    std::auto_ptr<gcn::ScrollArea> quest_description_scroll_area;
    
    boost::scoped_ptr<gcn::Label> player_label[2], house_colour_label[2], ready_label[2], player_name_label[2];
    boost::scoped_ptr<gcn::TextField> player_name_box[2];
    boost::scoped_ptr<gcn::DropDown> house_colour_dropdown[2];
    boost::scoped_ptr<HouseColourListModel> house_colour_list_model;
    boost::scoped_ptr<gcn::Label> house_colour_square[2];
    boost::scoped_ptr<GuiButton> join_button[2];
    boost::scoped_ptr<gcn::ListBox> observers_listbox;
    std::auto_ptr<gcn::ScrollArea> observers_scrollarea;
    boost::scoped_ptr<gcn::Label> observers_title;
    boost::scoped_ptr<gcn::ListBox> chat_listbox;
    std::auto_ptr<gcn::ScrollArea> chat_scrollarea;
    boost::scoped_ptr<gcn::Label> chat_field_title;
    boost::scoped_ptr<gcn::TextField> chat_field;

    boost::scoped_ptr<Coercri::CGFont> font_grey;
    
    bool extended, show_obs;
    bool chat_reformatted;
};

MenuScreenImpl::MenuScreenImpl(boost::shared_ptr<KnightsClient> kc, bool extended_, bool show_obs_, KnightsApp &app, 
                               boost::shared_ptr<Coercri::Window> win, gcn::Gui &gui_)
    : knights_client(kc), knights_app(app), gui(gui_), window(win), extended(extended_),
      show_obs(show_obs_), chat_reformatted(false)
{
    font_grey.reset(new Coercri::CGFont(knights_app.getFont()));
    font_grey->setColor(Coercri::Color(170, 170, 170));
    
    // create container
    container.reset(new gcn::Container);
    container->setOpaque(false);
    
    const int pad = 6, vpad_pretitle = 6, vpad_posttitle = 15, vpad_mid = 15, vpad_bot = 4;
    const int vpad_before_quest_box = 30;
    const int vpad_rhs = 5, width_rhs = extended ? 450 : 0;
    const int gutter = 15;
    
    // Create title (don't add it yet)
    title.reset(new gcn::Label("QUEST SELECTION"));
    
    // Add menu widgets
    const int menu_y = vpad_pretitle + title->getHeight() + vpad_posttitle;
    int menu_width, y_after_menu;
    app.getGameManager().createMenuWidgets(this, pad, menu_y, *container.get(),
                                           title->getFont(), font_grey.get(),
                                           menu_width, y_after_menu);

    quest_description_box.reset(new GuiTextWrap);
    quest_description_box->setForegroundColor(gcn::Color(0,0,0));
    quest_description_box->setWidth(menu_width - DEFAULT_SCROLLBAR_WIDTH - 4);
    quest_description_box->adjustHeight();
    quest_description_scroll_area = MakeScrollArea(*quest_description_box, menu_width, title->getFont()->getHeight() * 7);
    y_after_menu += vpad_before_quest_box;
    container->add(quest_description_scroll_area.get(), pad, y_after_menu);
    y_after_menu += quest_description_scroll_area->getHeight();

    // Add title
    container->add(title.get(), pad + menu_width/2 - title->getWidth()/2, vpad_pretitle);

    const int rhs_x = pad + menu_width + (extended ? gutter : 0);
    int rhs_y = vpad_pretitle;

    if (extended) {
        house_colour_list_model.reset(new HouseColourListModel(app.getGameManager()));
        
        // Add players area
        for (int i = 0; i < 2; ++i) {
            player_label[i].reset(new gcn::Label(std::string("Player ") + char('1'+i) + ": "));
            const int label_height = player_label[i]->getHeight();
            container->add(player_label[i].get(), rhs_x, rhs_y);

            ready_label[i].reset(new gcn::Label("(Ready)"));
            container->add(ready_label[i].get(), rhs_x + width_rhs - ready_label[i]->getWidth(), rhs_y);

            house_colour_dropdown[i].reset(new gcn::DropDown);
            house_colour_dropdown[i]->setFont(&g_house_colour_font);
            house_colour_dropdown[i]->setListModel(house_colour_list_model.get());
            house_colour_dropdown[i]->setWidth(g_house_colour_font.getWidth("") + g_house_colour_font.getHeight() + 5);
            house_colour_dropdown[i]->addActionListener(this);
            house_colour_dropdown[i]->setSelectionColor(gcn::Color(255,255,255));
            const int hse_col_x = ready_label[i]->getX() - house_colour_dropdown[i]->getWidth() - pad;
            container->add(house_colour_dropdown[i].get(), hse_col_x, rhs_y);

            house_colour_square[i].reset(new gcn::Label);
            house_colour_square[i]->setFont(&g_house_colour_font);
            house_colour_square[i]->adjustSize();
            container->add(house_colour_square[i].get(), hse_col_x + 3, rhs_y);

            house_colour_label[i].reset(new gcn::Label("Col:"));
            const int col_label_x = hse_col_x - house_colour_label[i]->getWidth();
            container->add(house_colour_label[i].get(), col_label_x, rhs_y);

            join_button[i].reset(new GuiButton("Join Game"));
            join_button[i]->addActionListener(this);
            join_button[i]->setVisible(false);
            container->add(join_button[i].get(), rhs_x + width_rhs - join_button[i]->getWidth(), rhs_y - 2);
            
            player_name_box[i].reset(new gcn::TextField);
            player_name_box[i]->setFocusable(false);
            const int name_x = rhs_x + player_label[i]->getWidth();
            player_name_box[i]->setWidth(house_colour_label[i]->getX() - name_x - pad);
            player_name_box[i]->adjustHeight();
            container->add(player_name_box[i].get(), name_x, rhs_y);

            player_name_label[i].reset(new gcn::Label("--"));
            container->add(player_name_label[i].get(), name_x, rhs_y);
            
            rhs_y += label_height + pad + 4;
        }

        // Add chat field (at bottom)
        int ybelow = y_after_menu;

        chat_field.reset(new gcn::TextField);
        chat_field->adjustSize();
        chat_field->setWidth(width_rhs);
        chat_field->addActionListener(this);
        container->add(chat_field.get(), rhs_x, ybelow - chat_field->getHeight());
        ybelow -= chat_field->getHeight();

        chat_field_title.reset(new gcn::Label("Type here to chat"));
        container->add(chat_field_title.get(), rhs_x, ybelow - chat_field_title->getHeight());
        ybelow -= chat_field_title->getHeight();
        ybelow -= (pad + vpad_rhs);

        // Add observers window
        if (show_obs) {
            observers_title.reset(new gcn::Label("Observers"));
            --rhs_y;
            container->add(observers_title.get(), rhs_x, rhs_y);
            rhs_y += observers_title->getHeight() + 4;
            observers_listbox.reset(new gcn::ListBox);
            observers_listbox->setListModel(&app.getGameManager().getObserversList());
            observers_scrollarea = MakeScrollArea(*observers_listbox, width_rhs, 3 * observers_listbox->getFont()->getHeight());
            AdjustListBoxSize(*observers_listbox, *observers_scrollarea);
            container->add(observers_scrollarea.get(), rhs_x, rhs_y);
            rhs_y += observers_scrollarea->getHeight() + vpad_rhs;
        }
        
        // Add chat area (to take up remaining space)
        rhs_y += 12;
        chat_listbox.reset(new gcn::ListBox);
        app.getGameManager().getChatList().setGuiParams(chat_listbox->getFont(), width_rhs);
        chat_listbox->setListModel(&app.getGameManager().getChatList());
        chat_listbox->setWidth(width_rhs);
        chat_scrollarea = MakeScrollArea(*chat_listbox, width_rhs, std::max(10, ybelow - rhs_y));
        chat_scrollarea->setHorizontalScrollPolicy(gcn::ScrollArea::SHOW_NEVER);
        chat_scrollarea->setVerticalScrollPolicy(gcn::ScrollArea::SHOW_NEVER);
        container->add(chat_scrollarea.get(), rhs_x, rhs_y);
    }

    const int container_width = rhs_x + width_rhs + pad;
    
    // Add the start and exit buttons
    int y = y_after_menu + vpad_mid;
    if (extended) {
        ready_checkbox.reset(new gcn::CheckBox("Ready to Start"));
        ready_checkbox->addActionListener(this);
        container->add(ready_checkbox.get(), pad, y);
    } else {
        start_button.reset(new GuiButton("Start"));
        start_button->addActionListener(this);
        container->add(start_button.get(), pad, y);
    }
    exit_button.reset(new GuiButton("Exit"));
    exit_button->addActionListener(this);
    container->add(exit_button.get(), container_width - pad - exit_button->getWidth(), y);
    y += exit_button->getHeight();
    y += vpad_bot;
    
    // resize the container
    container->setSize(container_width, y);

    panel.reset(new GuiPanel(container.get()));
    centre.reset(new GuiCentre(panel.get()));
    gui.setTop(centre.get());

    // make sure everything is up to date
    updateGui();
}

MenuScreenImpl::~MenuScreenImpl()
{
    // get rid of the menu widgets at this point.
    container.reset();
    knights_app.getGameManager().destroyMenuWidgets();
}

void MenuScreenImpl::action(const gcn::ActionEvent &event)
{
    if (event.getSource() == exit_button.get()) {
        knights_client->leaveGame();
        return;
    }

    if (event.getSource() == ready_checkbox.get()) {
        knights_client->setReady(ready_checkbox->isSelected());
        return;
    }

    if (event.getSource() == start_button.get()) {
        knights_client->setReady(true);
        return;
    }

    if (event.getSource() == chat_field.get()) {
        knights_client->sendChatMessage(chat_field->getText());
        chat_field->setText("");
        gui.logic();
        window->invalidateAll();
        return;
    }

    for (int i = 0; i < 2; ++i) {
        if (event.getSource() == house_colour_dropdown[i].get()) {
            const int sel = house_colour_dropdown[i]->getSelected();
            knights_client->setHouseColour(sel);
            return;
        }

        if (event.getSource() == join_button[i].get()) {
            knights_client->changePlayerNum(i);
            return;
        }
    }

    // Must have been one of the menu drop downs
    std::string key;
    int val;
    const bool found = knights_app.getGameManager().getDropdownInfo(event.getSource(), key, val);
    if (found) {
        knights_client->setMenuSelection(key, val);
    }
}

void MenuScreenImpl::updateGui()
{
    GameManager &game_manager = knights_app.getGameManager();

    const bool i_am_observer = (game_manager.getMyPlayerNum() == -1);    
    
    if (chat_listbox) {

        chat_listbox->adjustSize();
        
        if (!chat_reformatted && chat_listbox->getHeight() > chat_scrollarea->getHeight()) {
            chat_listbox->setWidth(chat_listbox->getWidth() - DEFAULT_SCROLLBAR_WIDTH);
            chat_scrollarea->setVerticalScrollPolicy(gcn::ScrollArea::SHOW_ALWAYS);
            game_manager.getChatList().setGuiParams(chat_listbox->getFont(), chat_listbox->getWidth());
            chat_reformatted = true;
        }
    }

    if (player_name_box[0].get()) {
        for (int i = 0; i < 2; ++i) {
            const std::string my_name = game_manager.getPlayerName(i);
            if (my_name.empty()) {
                player_name_box[i]->setVisible(false);
                player_name_label[i]->setVisible(true);
            } else {
                player_name_box[i]->setVisible(true);
                player_name_label[i]->setVisible(false);
                player_name_box[i]->setText(my_name);
            }
            
            const bool ready = !my_name.empty() && game_manager.getPlayerReady(i);
            ready_label[i]->setVisible(ready);
            
            const int my_col = game_manager.getHouseColour(i);
            house_colour_square[i]->setCaption(ColToText(game_manager.getAvailHouseColour(my_col)));
            house_colour_dropdown[i]->setSelected(my_col);

            if (my_name.empty()) {
                house_colour_dropdown[i]->setVisible(false);
                house_colour_square[i]->setVisible(false);
                house_colour_label[i]->setVisible(false);
                join_button[i]->setVisible(i_am_observer);
            } else {
                house_colour_label[i]->setVisible(true);
                if (i == game_manager.getMyPlayerNum()) {
                    house_colour_dropdown[i]->setVisible(true);
                    house_colour_square[i]->setVisible(false);
                } else {
                    house_colour_dropdown[i]->setVisible(false);
                    house_colour_square[i]->setVisible(true);
                }
                join_button[i]->setVisible(false);
            }
        }
    }

    quest_description_box->setText(game_manager.getQuestDescription());
    quest_description_box->adjustHeight();

    if (observers_listbox) AdjustListBoxSize(*observers_listbox, *observers_scrollarea);

    game_manager.setMenuWidgetsEnabled(!i_am_observer);
    if (ready_checkbox) ready_checkbox->setVisible(!i_am_observer);
    
    gui.logic();
    window->invalidateAll();
}

MenuScreen::MenuScreen(boost::shared_ptr<KnightsClient> kc, bool extended_, bool show_obs_)
    : knights_client(kc), extended(extended_), show_obs(show_obs_)
{ }

bool MenuScreen::start(KnightsApp &app, boost::shared_ptr<Coercri::Window> win, gcn::Gui &gui)
{
    pimpl.reset(new MenuScreenImpl(knights_client, extended, show_obs, app, win, gui));
    return true;
}

void MenuScreen::update()
{
    if (pimpl->knights_app.getGameManager().isGuiInvalid()) {
        pimpl->updateGui();
    }

    if (extended && pimpl->knights_app.getGameManager().getChatList().isUpdated()) {
        // auto scroll chat window to bottom
        gcn::Rectangle rect(0, pimpl->chat_listbox->getHeight() - pimpl->chat_listbox->getFont()->getHeight(),
                            1, pimpl->chat_listbox->getFont()->getHeight());
        pimpl->chat_scrollarea->showWidgetPart(pimpl->chat_listbox.get(), rect);
    }
}
