/*
 * main.cpp
 *
 * The main routine. Sets up a couple of error handlers etc then calls into knights_app.cpp.
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "knights_app.hpp"

#ifdef WIN32
#include "windows.h"
#include "SDL.h"
#endif

#include <sstream>

//----------------
 
// this controls whether to catch exceptions from main & print an
// error message, or else to rely on the compiler to print enough info
// about uncaught exceptions
#define CATCH_EXCEPTIONS

//----------------

#ifdef CATCH_EXCEPTIONS
#include "guichan/exception.hpp"
#include <iostream>
using namespace std;
#endif

extern "C" {  // needed for SDL I think

#ifdef WIN32
int KnightsMain(int argc, char **argv)  // On Windows this is called from WinMain below
#else
int main(int argc, char **argv)   // On other systems this is our real main function
#endif
{

    std::string err_msg;

#ifdef CATCH_EXCEPTIONS
    try {
#endif
        
        // Get resource dir.
        // If argv[1] is set then use that, else use DATA_DIR if available, else default to knights_data.
        string rdir;
        if (argc>1) {
            rdir = argv[1];
        } else {
#ifdef DATA_DIR
#define _QUOTEME(x) #x
#define QUOTEME(x) _QUOTEME(x)
            rdir = QUOTEME(DATA_DIR);
#else
            rdir = "knights_data";
#endif
        }

        // Run the game itself:
        KnightsApp app(rdir);
        app.runKnights();
        
#ifdef CATCH_EXCEPTIONS
    } catch (std::exception &e) {
        err_msg = "ERROR: Caught exception:\n";
        err_msg += e.what();
    } catch (gcn::Exception &e) {
        std::ostringstream str;
        str << "ERROR: Caught guichan exception:\n";
        str << e.getMessage() + "\n";
        str << "In " << e.getFunction() << " at " << e.getFilename() << ":" << e.getLine() << "\n";
        err_msg = str.str();
    } catch (...) {
        err_msg = "ERROR: Unknown exception caught\n";
    }
#endif

    if (!err_msg.empty()) {
#ifdef WIN32
        ::MessageBox(0, err_msg.c_str(), "Knights", MB_OK | MB_ICONERROR);
#else
        cout << err_msg << endl;
#endif
    }

    return 0;
}
    
} // extern "C"


//---------------------

//
// boost::assertion_failed
//

#ifdef BOOST_ENABLE_ASSERT_HANDLER

#include "boost/assert.hpp"

void boost::assertion_failed(char const * expr, char const * function, char const * file, long line) // user defined
{
    cout << "<< BOOST ASSERTION FAILED >> " << endl;
    cout << file << ":" << line << ": " << function << ": " << expr << endl;
    exit(1);
}

#endif


#ifdef WIN32

// WinMain
int WINAPI WinMain(HINSTANCE hInst, HINSTANCE hPrev, LPSTR szCmdLine, int sw)
{
    // Start DDHELP.exe - this is some kind of hack to prevent files being kept open, see 
    // SDL_win32_main.c in the SDL source code
    HMODULE handle = LoadLibrary("DDRAW.DLL");
    if (handle) FreeLibrary(handle);

    // Find path to the application
    char app_path[MAX_PATH];
    DWORD pathlen = GetModuleFileName(NULL, app_path, MAX_PATH);
    while (pathlen > 0 && app_path[pathlen] != '\\') --pathlen;
    app_path[pathlen] = '\0';

    // SDL requires this:
    SDL_SetModuleHandle(GetModuleHandle(NULL));

    // Make our own command line.
    // Knights only uses argv[1] which is the path to look for knights_data in. We set that to app_path.
    std::string rdir(app_path, pathlen);
    rdir += "\\knights_data";
    const char* MyArgv[2];
    MyArgv[0] = 0;
    MyArgv[1] = rdir.c_str();
    KnightsMain(2, (char**)MyArgv);

    return 0;
}

#endif

// ---------------------------------

// Fix "bug" with MSVC static libs + global object constructors.
#ifdef _MSC_VER
#pragma comment (linker, "/include:_InitMagicActions")
#pragma comment (linker, "/include:_InitScriptActions")
#pragma comment (linker, "/include:_InitControls")
#endif
