/*
 * local_mini_map.cpp
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "config_map.hpp"
#include "local_mini_map.hpp"

namespace {
    const Coercri::Color col[] = {
        Coercri::Color(255,0,0),     // highlight
        Coercri::Color(255,255,255), // highlight
        Coercri::Color(255,0,0),     // highlight
        Coercri::Color(0,0,0),       // highlight
        Coercri::Color(68,68,34),    // COL_WALL
        Coercri::Color(136,136,85),  // COL_FLOOR
        Coercri::Color(0,0,0),       // COL_UNMAPPED
    };

    const int FIRST_HIGHLIGHT = 0;
    const int NUM_HIGHLIGHTS = 4;
}

void LocalMiniMap::draw(Coercri::GfxContext &gc, int left, int top, int npx, int npy, int t) const
{
    if (width==0 || height==0) return;
    if (npx == 0 || npy == 0) return;  // will cause division by zero otherwise

    // Set highlights to their proper colour.
    Highlighter h(*this, t, config_map);

    const int scale = min(npx/width, npy/height);

    if (scale == 0) {

        // Down-scaling

        const int recip_scale = max(width/npx, height/npy);
        const int xinc = width % npx;
        const int yinc = height % npy;

        int xrem = 0, yrem = 0;
        int xbase = 0, ybase = 0;

        for (int y = top; y < top + npy; ++y) {

            xbase = 0;
            xrem = 0;

            bool extend_up = false;
            if (yrem >= npy) {
                yrem -= npy;
                extend_up = true;
            }

            for (int x = left; x < left + npx; ++x) {

                bool extend_right = false;
                if (xrem >= npx) {
                    xrem -= npx;
                    extend_right = true;
                }

                MiniMapColour csel = COL_UNMAPPED;
                for (int j = 0; j < (extend_up ? recip_scale+1 : recip_scale); ++j) {
                    for (int i = 0; i < (extend_right ? recip_scale+1 : recip_scale); ++i) {
                        ASSERT(xbase + i < width);
                        ASSERT(ybase + j < height);
                        csel = min(csel, data[(xbase+i) + (ybase+j)*width]);
                    }
                }

                gc.plotPixel(x, y, col[csel]);

                xrem += xinc;
                xbase += extend_right ? recip_scale+1 : recip_scale;
            }

            yrem += yinc;
            ybase += extend_up ? recip_scale+1 : recip_scale;
        }
        

    } else {

        // Up-scaling

        const int xinc = npx % width;
        const int yinc = npy % height;
    
        int xrem = 0, yrem = 0;
        int xbase = left, ybase = top;

        for (int y = 0; y < height; ++y) {

            xbase = left;
            xrem = 0;
        
            bool extend_up = false;
            if (yrem >= height) {
                yrem -= height;
                extend_up = true;
            }

            const MiniMapColour *pright = &data[y*width];
            const MiniMapColour *paboveright = (y+1==height) ? 0 : &data[(y+1)*width];
        
            MiniMapColour chere, cabove;
            MiniMapColour cright = *pright++;
            MiniMapColour caboveright = paboveright ? *paboveright++ : COL_UNMAPPED;
        
            for (int x = 0; x < width; ++x) {

                bool extend_right = false;
                if (xrem >= width) {
                    xrem -= width;
                    extend_right = true;
                }

                chere = cright;
                cabove = caboveright;
                if (x != width-1) {
                    cright = *pright++;
                    caboveright = paboveright ? *paboveright++ : COL_UNMAPPED;
                } else {
                    cright = caboveright = COL_UNMAPPED;
                }

                for (int j=0; j<scale; ++j) {
                    for (int i=0; i<scale; ++i) {
                        gc.plotPixel(xbase+i, ybase+j, col[chere]);
                    }
                }

                if (extend_right) {
                    MiniMapColour csel = max(chere, cright);
                    for (int j=0; j<scale; ++j) {
                        gc.plotPixel(xbase+scale, ybase+j, col[csel]);
                    }
                    if (extend_up) {
                        csel = max(csel, cabove);
                        csel = max(csel, caboveright);
                        gc.plotPixel(xbase+scale, ybase+scale, col[csel]);
                    }
                }
                if (extend_up) {
                    const MiniMapColour csel = max(chere, cabove);
                    for (int i=0; i<scale; ++i) {
                        gc.plotPixel(xbase+i, ybase+scale, col[csel]);
                    }
                }

                xrem += xinc;
                xbase += extend_right ? scale+1 : scale;
            }

            yrem += yinc;
            ybase += extend_up ? scale+1 : scale;
        }
    }
}

void LocalMiniMap::setSize(int w, int h)
{
    if (width != 0 || w <= 0 || h <= 0) return;
    width = w;
    height = h;
    data.resize(width*height);
    wipeMap();
}

void LocalMiniMap::setColour(int x, int y, MiniMapColour col)
{
    if (x < 0 || x >= width || y < 0 || y >= height) return;
    data[y*width + x] = col;
}
    
void LocalMiniMap::wipeMap()
{
    std::fill(data.begin(), data.end(), COL_UNMAPPED);
}

void LocalMiniMap::mapKnightLocation(int n, int x, int y)
{
    setHighlight(x, y, n);
}

void LocalMiniMap::mapItemLocation(int x, int y, bool on)
{
    const int id = y * height + x + 1000;  // add 1000 to prevent clashes with other IDs
    if (on) {
        setHighlight(x, y, id);
    } else {
        setHighlight(-1, -1, id);
    }
}

void LocalMiniMap::setHighlight(int x, int y, int id)
{
    // The strategy is to store highlights into a separate map, rather than
    // simply setting colours directly in the "data" array. This simplifies
    // setHighlight and setColour, but it does complicate the draw routines
    // slightly. 
    
    // Erase the old highlight first (if any)
    map<int,Highlight>::iterator it = highlights.find(id);
    if (it != highlights.end()) {
        highlights.erase(it);
    }

    if (x >= 0 && x < width && y >= 0 && y < height) {
        // Set the new highlight.
        Highlight h;
        h.x = x;
        h.y = y;
        h.old_colour = data[y*width+x];
        highlights.insert(make_pair(id, h));
    }
}

LocalMiniMap::Highlighter::Highlighter(const LocalMiniMap &m, int t, const ConfigMap &config_map)
    : mini_map(m)
{
    const MiniMapColour col_highlight = MiniMapColour((t / config_map.getInt("mini_map_flash_time")) % NUM_HIGHLIGHTS + FIRST_HIGHLIGHT);

    for (std::map<int,LocalMiniMap::Highlight>::const_iterator it = mini_map.highlights.begin();
    it != mini_map.highlights.end(); ++it) {
        const int idx = it->second.y * mini_map.width + it->second.x;
        it->second.old_colour = mini_map.data[idx];
        mini_map.data[idx] = col_highlight;
    }
}

LocalMiniMap::Highlighter::~Highlighter()
{
    // Reset highlights to their proper map colour.
    for (std::map<int, LocalMiniMap::Highlight>::const_iterator it = mini_map.highlights.begin();
    it != mini_map.highlights.end(); ++it) {
        mini_map.data[it->second.y * mini_map.width + it->second.x] = it->second.old_colour;
    }
}
