/*
 * local_dungeon_view.hpp
 *
 * Implementation of DungeonView for on-screen drawing.
 * 
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef LOCAL_DUNGEON_VIEW_HPP
#define LOCAL_DUNGEON_VIEW_HPP

#include "dungeon_view.hpp"
#include "entity_map.hpp"

#include "gfx/gfx_context.hpp" // coercri

#include <list>
#include <queue>
#include <vector>

class ConfigMap;
class GfxManager;

class LocalDungeonView : public DungeonView {
public:

    //
    // functions specific to this class
    //

    LocalDungeonView(const ConfigMap &config_map, int approach_offset);
    void addToTime(int delta) { time += delta; }
    void draw(Coercri::GfxContext &gc, GfxManager &gm, bool screen_flash,
              int phy_dungeon_left, int phy_dungeon_top,
              int phy_dungeon_width, int phy_dungeon_height,
              int phy_pixels_per_square, float dungeon_scale_factor);
    boost::shared_ptr<ColourChange> getMyColourChange() const { return my_colour_change; }
    bool isApproached() const { return my_approached; }
    bool aliveRecently() const;

    //
    // functions from DungeonView
    //
    
    void setCurrentRoom(int r, int width, int height);

    void addEntity(unsigned short int id, int x, int y, MapHeight ht, MapDirection facing,
                   const Anim * anim, const Overlay *ovr, int af, int atz_diff,
                   bool ainvuln, // (anim data)
                   int cur_ofs, MotionType motion_type, int motion_time_remaining);
    void rmEntity(unsigned short int id);
    void repositionEntity(unsigned short int id, int new_x, int new_y);
    void moveEntity(unsigned short int id, MotionType motion_type, int motion_duration, bool missile_mode);
    void flipEntityMotion(unsigned short int id, int initial_delay, int motion_duration);
    void setAnimData(unsigned short int id, const Anim *, const Overlay *, int af, int atz_diff, bool ainvuln, bool currently_moving);
    void setFacing(unsigned short int id, MapDirection new_facing);

    void clearTiles(int x, int y);
    void setTile(int x, int y, int depth, const Graphic *gfx, boost::shared_ptr<const ColourChange> cc);

    void setItem(int x, int y, const Graphic *gfx);

    void placeIcon(int x, int y, const Graphic *gfx, int dur);

    void flashMessage(const std::string &msg, int ntimes);
    void cancelContinuousMessages();
    void addContinuousMessage(const std::string &msg);

private:
    // config map
    const ConfigMap &config_map;
    const int approach_offset;

    // time (updated by draw() method)
    int time;
    
    // icons
    struct LocalIcon {
        int room_no;
        int x, y;
        int expiry;
        bool operator<(const LocalIcon &other) const { return expiry < other.expiry; }
    };
    std::priority_queue<LocalIcon> icons;

    // room maps
    struct RoomData {
        // Origin is set to the top-left corner if the room is mapped,
        // or (-1,-1) if it is unmapped.
        // Width, height correspond to the size of gmap.
        int origin_x, origin_y;
        int width, height;

        // Items, Tiles
        struct GfxEntry {
            const Graphic *gfx;
            boost::shared_ptr<const ColourChange> cc;
            int depth;
        };
        std::vector<std::list<GfxEntry> > gmap;
        std::list<GfxEntry> & lookupGfx(int x, int y) { return gmap[y*width + x]; }
        bool valid(int x, int y) const { return x>=0 && y>=0 && x<width && y<height; }

        // Entities
        EntityMap emap;

        // ctor
        RoomData(const ConfigMap &cfg, int approach_offset, int w, int h)
            : origin_x(-1), origin_y(-1), width(w), height(h), gmap(w*h), emap(cfg, approach_offset)
        { }
    };
    std::map<int, RoomData> rooms;
    int current_room;

    // my knight
    int my_x, my_y;
    int last_known_time;
    int last_known_x, last_known_y;
    boost::shared_ptr<ColourChange> my_colour_change;
    bool my_approached;

    // messages
    struct Message {
        std::string message;
        int start_time;
        int stop_time;
    };
    std::deque<Message> messages;  // sorted by start_time
    std::vector<std::string> cts_messages;
    int last_mtime;

    // gfx buffer
    std::map<int, std::vector<GraphicElement> > gfx_buffer;
};

#endif
