/*
 * local_dungeon_view.cpp
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "anim.hpp"
#include "config_map.hpp"
#include "draw.hpp"
#include "gfx_manager.hpp"
#include "local_dungeon_view.hpp"
#include "round.hpp"

namespace {
    // NB item_depth must be greater than the other depths here.
    const int icon_depth = -205;
    const int entity_depth = -200;  // note this is modified (in steps of 10) for MapHeight, 
                                    // and 1 is subtracted for weapon overlays.
    const int item_depth = -100;

    const int MY_X_OFFSCREEN_VALUE = -999;
}

LocalDungeonView::LocalDungeonView(const ConfigMap &cfg, int approach_offset_)
    : config_map(cfg),
      approach_offset(approach_offset_),
      time(0),
      current_room(-1),
      my_x(MY_X_OFFSCREEN_VALUE), my_y(0),
      last_known_time(-999999),
      last_known_x(0), last_known_y(0),
      my_approached(false),
      last_mtime(-999999)
{ }


void LocalDungeonView::draw(Coercri::GfxContext &gc, GfxManager &gm, bool screen_flash,
                            int phy_dungeon_left, int phy_dungeon_top,
                            int phy_dungeon_width, int phy_dungeon_height,
                            int phy_pixels_per_square, float dungeon_scale_factor)
{
    // Clear out expired icons
    while (!icons.empty() && time > icons.top().expiry) {
        map<int,RoomData>::iterator ri = rooms.find(icons.top().room_no);
        if (ri != rooms.end()) {
            list<RoomData::GfxEntry> &glst(ri->second.lookupGfx(icons.top().x,
                                                                icons.top().y));
            for (list<RoomData::GfxEntry>::iterator it = glst.begin(); it != glst.end();
            ++it) {
                if (it->depth == icon_depth) {
                    glst.erase(it);
                    break;
                }
            }
        }
        icons.pop();
    }

    // Clear out expired messages
    // Note - this is not perfect as queue is ordered by start_time, rather than stop_time.
    // But this does not matter much in practice.
    while (!messages.empty() && time >= messages.front().stop_time) {
        messages.pop_front();
    }

    // Draw current room
    if (my_x != MY_X_OFFSCREEN_VALUE) last_known_time = time;

    if (aliveRecently()) {
        // We are alive, or have been dead for no longer than
        // "death_draw_map_time". Draw the map.
        std::map<int, RoomData>::iterator ri = rooms.find(current_room);
        if (ri != rooms.end()) {
            RoomData &rd(ri->second);

            // Clear out the graphics buffer
            for (std::map<int, std::vector<GraphicElement> >::iterator it = gfx_buffer.begin();
            it != gfx_buffer.end(); ++it) {
                it->second.clear();
            }
            
            // Find on-screen top-left corner for the room
            const int room_tl_x = phy_dungeon_left + (phy_dungeon_width - rd.width*phy_pixels_per_square) / 2;
            const int room_tl_y = phy_dungeon_top + (phy_dungeon_height - rd.height*phy_pixels_per_square) / 2;

            // Build a list of graphics to be drawn

            // Draw tiles (if screen not flashing)
            if (!screen_flash) {
                for (int i=0; i<rd.width; ++i) {
                    for (int j=0; j<rd.height; ++j) {
                        const int screen_x = i*phy_pixels_per_square + room_tl_x;
                        const int screen_y = j*phy_pixels_per_square + room_tl_y;
                        
                        const list<RoomData::GfxEntry> &glst(rd.lookupGfx(i,j));
                        for (list<RoomData::GfxEntry>::const_iterator it = glst.begin();
                        it != glst.end(); ++it) {
                            GraphicElement ge;
                            ge.sx = screen_x;
                            ge.sy = screen_y;
                            ge.gr = it->gfx;
                            ge.cc = it->cc.get();
                            gfx_buffer[it->depth].push_back(ge);
                        }
                    }
                }
            }
                
            // Add the entities (always)
            rd.emap.getEntityGfx(time, room_tl_x, room_tl_y, phy_pixels_per_square, entity_depth, gfx_buffer);
                
            // Now draw all buffered graphics
            for (map<int, vector<GraphicElement> >::reverse_iterator it = gfx_buffer.rbegin();
            it != gfx_buffer.rend(); ++it) {
                for (vector<GraphicElement>::const_iterator it2 = it->second.begin();
                it2 != it->second.end(); ++it2) {

                    int old_width, old_height;
                    gm.getGraphicSize(*it2->gr, old_width, old_height);
                    const int new_width = Round(old_width * dungeon_scale_factor);
                    const int new_height = Round(old_height * dungeon_scale_factor);
                    
                    if (it2->cc) {
                        gm.drawTransformedGraphic(gc, it2->sx, it2->sy, *it2->gr, new_width, new_height, *it2->cc);
                    } else {
                        gm.drawTransformedGraphic(gc, it2->sx, it2->sy, *it2->gr, new_width, new_height);
                    }
                }
            }
        }
    } else {
        // Dead - clear out the msg queue as it looks stupid when msgs from your previous life come up!
        messages.clear();
        cts_messages.clear();
    }

    // The rest should not be drawn when screen is flashing
    if (!screen_flash) {

        // Determine whether we should draw a message
        if (messages.empty() && cts_messages.empty()) {
            last_mtime = 0;
        } else {
            if (last_mtime == 0) {
                if (!messages.empty()) last_mtime = messages.front().start_time;
                else last_mtime = time;
            }
            int mtime = time - last_mtime;
                
            const int msg_on_time = config_map.getInt("message_on_time");
            const int msg_off_time = config_map.getInt("message_off_time");
            const int nmsgs = messages.size() + cts_messages.size();
            int mphase = (mtime / (msg_on_time + msg_off_time)) % nmsgs;
            const string &the_msg(mphase < messages.size() ? messages[mphase].message
                                  : cts_messages[mphase - messages.size()]);
            if (mtime % (msg_on_time + msg_off_time) <= msg_on_time) {
                std::map<int,RoomData>::iterator ri = rooms.find(current_room);
                if (ri != rooms.end()) {
                    DrawUI::drawMessage(config_map, gc,
                                        phy_dungeon_left, phy_dungeon_top, phy_dungeon_width, phy_dungeon_height,
                                        gm,
                                        last_known_x, last_known_y, ri->second.width,
                                        ri->second.height, phy_pixels_per_square, the_msg);
                }
            }
        }
    }
}

bool LocalDungeonView::aliveRecently() const
{
    return time < last_known_time + config_map.getInt("death_draw_map_time");
}

void LocalDungeonView::setCurrentRoom(int r, int w, int h)
{
    typedef std::map<int, RoomData>::iterator Iter;

    // Wipe all entities from the current map
    Iter curr_room = rooms.find(current_room);
    if (curr_room != rooms.end()) {
        curr_room->second.emap.clear();
    }

    // Create the new map (if necessary) and reset current_room & current_dmap
    Iter it = rooms.find(r);
    if (it == rooms.end()) {
        rooms.insert(std::make_pair(r, RoomData(config_map, approach_offset, w, h)));
    }
    current_room = r;
}

void LocalDungeonView::addEntity(unsigned short int id, int x, int y, MapHeight ht, MapDirection facing,
                                 const Anim * anim, const Overlay *ovr, int af, int atz_diff,
                                 bool ainvuln,
                                 int cur_ofs, MotionType motion_type, int motion_time_remaining)
{
    std::map<int, RoomData>::iterator ri = rooms.find(current_room);
    if (ri == rooms.end()) return;
    ri->second.emap.addEntity(time, id, x, y, ht, facing, anim, ovr, af, atz_diff ? atz_diff + time : 0, ainvuln,
                              cur_ofs, motion_type, motion_time_remaining);
    if (id == 0) {
        // my knight
        my_x = x;
        my_y = y;
        if (my_x >= 0) {
            last_known_x = my_x;
            last_known_y = my_y;
        }
        my_approached = false;
        if (!my_colour_change && anim) {
            my_colour_change.reset(new ColourChange(anim->getColourChange(false)));
        }
    }
}

void LocalDungeonView::rmEntity(unsigned short int id)
{
    std::map<int, RoomData>::iterator ri = rooms.find(current_room);
    if (ri == rooms.end()) return;
    ri->second.emap.rmEntity(id);
    if (id==0) {
        my_x = MY_X_OFFSCREEN_VALUE;
    }
}

void LocalDungeonView::repositionEntity(unsigned short int id, int new_x, int new_y)
{
    std::map<int, RoomData>::iterator ri = rooms.find(current_room);
    if (ri == rooms.end()) return;
    ri->second.emap.repositionEntity(id, new_x, new_y);
    if (id == 0) {
        my_x = new_x;
        my_y = new_y;
        if (my_x >= 0) {
            last_known_x = my_x;
            last_known_y = my_y;
        }
        my_approached = false;
    }
}

void LocalDungeonView::moveEntity(unsigned short int id, MotionType mt, int motion_dur, bool missile_mode)
{
    std::map<int, RoomData>::iterator ri = rooms.find(current_room);
    if (ri == rooms.end()) return;
    ri->second.emap.moveEntity(time, id, mt, motion_dur, missile_mode);
    if (id==0) {
        my_approached = (mt == MT_APPROACH);
    }
}

void LocalDungeonView::flipEntityMotion(unsigned short int id, int initial_delay, int motion_dur)
{
    std::map<int, RoomData>::iterator ri = rooms.find(current_room);
    if (ri == rooms.end()) return;
    ri->second.emap.flipEntityMotion(time, id, initial_delay, motion_dur);
}

void LocalDungeonView::setAnimData(unsigned short int id, const Anim *a, const Overlay *o,
                                   int af, int atz_diff, bool ainvuln, bool currently_moving)
{
    std::map<int, RoomData>::iterator ri = rooms.find(current_room);
    if (ri == rooms.end()) return;
    ri->second.emap.setAnimData(id, a, o, af, atz_diff ? atz_diff + time : 0, ainvuln, currently_moving);
}

void LocalDungeonView::setFacing(unsigned short int id, MapDirection new_facing)
{
    map<int,RoomData>::iterator ri = rooms.find(current_room);
    if (ri == rooms.end()) return;  
    ri->second.emap.setFacing(id, new_facing);
}

void LocalDungeonView::clearTiles(int x, int y)
{
    map<int,RoomData>::iterator ri = rooms.find(current_room);
    if (ri == rooms.end()) return;
    if (!ri->second.valid(x,y)) return;
    list<RoomData::GfxEntry> &gfx(ri->second.lookupGfx(x, y));
    // We know that the tiles come before items or other stuff like that.
    while (!gfx.empty() && gfx.front().depth != item_depth) gfx.pop_front();
}

void LocalDungeonView::setTile(int x, int y, int depth, const Graphic * gfx,
                               boost::shared_ptr<const ColourChange> cc)
{
    std::map<int,RoomData>::iterator ri = rooms.find(current_room);
    if (ri == rooms.end()) return;
    if (!ri->second.valid(x,y)) return;
    std::list<RoomData::GfxEntry> &glst(ri->second.lookupGfx(x, y));
    std::list<RoomData::GfxEntry>::iterator it = glst.begin();
    while (1) {
        // The list is in sorted order, highest (ie deepest) depth first.
        if (it == glst.end() || depth > it->depth) {
            if (gfx) {
                RoomData::GfxEntry ge;
                ge.gfx = gfx;
                ge.cc = cc;
                ge.depth = depth;
                glst.insert(it, ge);
            }
            break;
        } else if (it->depth == depth) {
            if (gfx) {
                it->gfx = gfx;
                it->cc = cc;
            } else {
                glst.erase(it);
            }
            break;
        } else {
            ++it;
        }
    }
}

void LocalDungeonView::setItem(int x, int y, const Graphic * graphic)
{
    // We can implement this using setTile.
    setTile(x, y, item_depth, graphic, boost::shared_ptr<const ColourChange>());
}

void LocalDungeonView::placeIcon(int x, int y, const Graphic *g, int dur)
{
    if (current_room == -1) return;
    
    // Add the 'icon' to the gmap. Also store in 'icons' list.
    setTile(x, y, icon_depth, g, boost::shared_ptr<const ColourChange>());
    LocalIcon ic;
    ic.room_no = current_room;
    ic.x = x;
    ic.y = y;
    ic.expiry = dur + time;
    icons.push(ic);
}

void LocalDungeonView::flashMessage(const std::string &s, int ntimes)
{
    Message m;
    m.message = s;
    m.start_time = time;
    m.stop_time = time + (config_map.getInt("message_on_time") + config_map.getInt("message_off_time"))*ntimes;
    messages.push_back(m);
}

void LocalDungeonView::cancelContinuousMessages()
{
    cts_messages.clear();
}

void LocalDungeonView::addContinuousMessage(const std::string &s)
{
    cts_messages.push_back(s);
}
