/*
 * local_display.hpp
 *
 * In-game display. Can handle both split screen and single screen
 * modes. Implements KnightsCallbacks.
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef LOCAL_DISPLAY_HPP
#define LOCAL_DISPLAY_HPP

#include "knights_callbacks.hpp"
#include "map_support.hpp"

// coercri stuff
#include "gcn/cg_font.hpp"
#include "gfx/gfx_context.hpp"
#include "timer/timer.hpp"

#include "boost/scoped_ptr.hpp"

class ChatList;
class ConfigMap;
class Controller;
class GfxManager;
class Graphic;
class GuiPanel2;
class LocalDungeonView;
class LocalMiniMap;
class LocalStatusDisplay;
class PotionRenderer;
class SkullRenderer;
class SoundManager;

class LocalDisplay : public KnightsCallbacks, public gcn::ActionListener {
public:
    LocalDisplay(const ConfigMap &config_map_,
                 int approach_offset,
                 const Graphic *winner_image_,
                 const Graphic *loser_image_,
                 const Graphic *menu_gfx_centre,
                 const Graphic *menu_gfx_empty,
                 const Graphic *menu_gfx_highlight,
                 const PotionRenderer *potion_renderer,
                 const SkullRenderer *skull_renderer,
                 boost::shared_ptr<std::vector<std::pair<std::string,std::string> > > menu_strings_,
                 const std::vector<const UserControl*> &standard_controls_,
                 const Controller *ctrlr1,
                 const Controller *ctrlr2,
                 int nplyrs,
                 Coercri::Timer &timer,
                 ChatList &chat_list_,
                 KnightsClient &knights_client_,
                 gcn::Container &container_);
    ~LocalDisplay();

    enum PauseDisplay {
        PD_OFF,
        PD_TRANSPARENT,
        PD_OPAQUE
    };

    //
    // functions specific to this class
    //

    // draw the in-game screen.
    // NOTE: for split screen mode 2 calls to draw are needed (one for each player).
    void draw(Coercri::GfxContext &gc, GfxManager &gm, 
              int plyr_num, bool is_split_screen, 
              PauseDisplay pause_display);

    // play all queued sounds, then clear the sound queue.
    void playSounds(SoundManager &sm);

    // This routine reads the joystick/controller, and converts this
    // information to a UserControl. It also activates the on-screen
    // menu (when fire is held down) if necessary.
    const UserControl * readControl(int plyr);

    // Tell us if anyone has won yet. (ie has winGame() or loseGame() been called yet.)
    bool isGameOver() const { return won[0] || won[1] || lost[0] || lost[1]; }

    void setReadyFlag() { ready_msg_sent = true; }
    bool getReadyFlag() const { return ready_msg_sent; }

    void setChatUpdated() { chat_updated = true; }


    void action(const gcn::ActionEvent &event);
    
    
    //
    // functions overridden from KnightsCallbacks
    //
    
    DungeonView & getDungeonView(int plyr);
    MiniMap & getMiniMap(int plyr);
    StatusDisplay & getStatusDisplay(int plyr);

    void playSound(int plyr, const Sound &sound, int frequency);
    
    void winGame(int plyr);
    void loseGame(int plyr);
    
    void setAvailableControls(int plyr, const std::vector<std::pair<const UserControl*, bool> > &available_controls);
    void setMenuHighlight(int plyr, const UserControl *highlight);
    
    void flashScreen(int plyr, int delay);


    
private:
    void setMenuControl(int plyr, MapDirection d, const UserControl *ctrl, const UserControl *prev);
    void recalculateTime();
    void setupGui(int x, int y, int width, int height, GfxManager &gm);
    
private:
    const ConfigMap &config_map;
    
    // cached config variables
    const int ref_vp_width, ref_vp_height, ref_gutter;
    const int ref_pixels_per_square, dungeon_tiles_x, dungeon_tiles_y;
    const int min_font_size;
    const float ref_font_size;
    const int game_over_t1, game_over_t2, game_over_t3;
    const float game_over_ratio;

    // controllers.
    const Controller *controller1, *controller2;
    const std::vector<const UserControl*> &standard_controls;
    
    // controller state
    bool attack_mode[2], allow_menu_open[2], approached_when_menu_was_opened[2];
    int fire_start_time[2];
    enum MenuNullEnum { M_OK, M_CTS, M_NULL };
    MenuNullEnum menu_null[2];
    MapDirection menu_null_dir[2];
    MapDirection my_facing[2];
    const UserControl * tap_control[2];
    const UserControl * menu_control[2][4];
    
    // display components
    std::auto_ptr<LocalDungeonView> dungeon_view[2];
    std::auto_ptr<LocalMiniMap> mini_map[2];
    std::auto_ptr<LocalStatusDisplay> status_display[2];

    // other display stuff.
    int flash_screen_start[2];
    boost::shared_ptr<std::vector<std::pair<std::string,std::string> > > menu_strings;
    const Graphic *winner_image;
    const Graphic *loser_image;

    // we need to keep track of time (for animations, making entities move at the correct speed, etc).
    // the time is stored here and updated by calls to draw() (caller must pass in an updated time).
    int time;

    // have we won or lost yet
    bool won[2];
    bool lost[2];
    int game_over_time[2];
    bool ready_msg_sent;
    
    // sounds waiting to be played
    struct MySound {
        const Sound *sound;
        int frequency;
        bool plyr[2];
    };
    std::vector<MySound> sounds;

    // timer
    Coercri::Timer & timer;
    int last_time;

    // gui stuff
    ChatList &chat_list;
    KnightsClient &knights_client;
    gcn::Container &container;
    bool chat_updated;
    int prev_gui_width[2], prev_gui_height[2];
    boost::scoped_ptr<Coercri::CGFont> gui_font, gui_small_font;
    boost::scoped_ptr<gcn::ListBox> chat_listbox;
    std::auto_ptr<gcn::ScrollArea> chat_scrollarea;
    boost::scoped_ptr<gcn::Label> chat_label;
    boost::scoped_ptr<gcn::TextField> chat_field;
    boost::scoped_ptr<gcn::Button> send_button, clear_button;
    /*
    boost::scoped_ptr<gcn::Label> msg0_label[2], msg1_label[2], msg2_label[2];
    boost::scoped_ptr<GuiPanel2> msg0_panel[2], msg1_panel[2], msg2_panel[2];
    std::vector<boost::shared_ptr<GuiPanel2> > icon_panels[2];
    */
};

#endif
