/*
 * local_display.cpp
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "config_map.hpp"
#include "controller.hpp"
#include "game_manager.hpp"  // for ChatList
#include "gfx_manager.hpp"
#include "gfx_resizer.hpp"
#include "graphic.hpp"
#include "knights_client.hpp"
#include "local_display.hpp"
#include "local_dungeon_view.hpp"
#include "local_mini_map.hpp"
#include "local_status_display.hpp"
#include "make_scroll_area.hpp"
#include "my_exceptions.hpp"
#include "round.hpp"
#include "sound_manager.hpp"
#include "user_control.hpp"

#include "gui_panel_2.hpp"

#include "gfx/rectangle.hpp"  // coercri

namespace {
    // a version of ListBox that doesn't react to mouse clicks on it.
    class ListBoxNoMouse : public gcn::ListBox {
    public:
        virtual void mousePressed(gcn::MouseEvent &) { }
    };
}

LocalDisplay::LocalDisplay(const ConfigMap &cfg,
                           int approach_offset,
                           const Graphic *winner_image_,
                           const Graphic *loser_image_,
                           const Graphic *menu_gfx_centre,
                           const Graphic *menu_gfx_empty,
                           const Graphic *menu_gfx_highlight,
                           const PotionRenderer *potion_renderer,
                           const SkullRenderer *skull_renderer,
                           boost::shared_ptr<std::vector<std::pair<std::string,std::string> > > menu_strings_,
                           const std::vector<const UserControl*> &std_ctrls,
                           const Controller *ctrlr1, 
                           const Controller *ctrlr2,
                           int nplyrs,
                           Coercri::Timer & timer_,
                           ChatList &chat_list_,
                           KnightsClient &knights_client_,
                           gcn::Container &container_)
    : config_map(cfg),

      // cached config variables
      ref_vp_width(cfg.getInt("dpy_viewport_width")),
      ref_vp_height(cfg.getInt("dpy_viewport_height")),
      ref_gutter(cfg.getInt("dpy_gutter")),
      
      ref_pixels_per_square(cfg.getInt("dpy_pixels_per_square")),
      dungeon_tiles_x(cfg.getInt("dpy_dungeon_width") / ref_pixels_per_square),
      dungeon_tiles_y(cfg.getInt("dpy_dungeon_height") / ref_pixels_per_square),

      min_font_size(cfg.getInt("dpy_font_min_size")),
      ref_font_size(cfg.getInt("dpy_font_base_size") / 100.0f),

      game_over_t1(cfg.getInt("game_over_fade_time")),
      game_over_t2(game_over_t1 + cfg.getInt("game_over_black_time")),
      game_over_t3(game_over_t2 + game_over_t1),
      game_over_ratio(cfg.getInt("game_over_ratio") / 100.0f),
      
      // other stuff
      controller1(ctrlr1),
      controller2(ctrlr2),
      standard_controls(std_ctrls),
      menu_strings(menu_strings_),
      winner_image(winner_image_),
      loser_image(loser_image_),
      time(0),
      ready_msg_sent(false),

      timer(timer_),
      last_time(-1),

      chat_list(chat_list_),
      knights_client(knights_client_),
      container(container_),
      chat_updated(false)
{
    if (nplyrs != 1 && nplyrs != 2) {
        throw UnexpectedError("bad number of players in LocalDisplay");
    }
    
    for (int i = 0; i < nplyrs; ++i) {
        attack_mode[i] = false;
        allow_menu_open[i] = true;
        approached_when_menu_was_opened[i] = false;
        fire_start_time[i] = 0;
        menu_null[i] = M_OK;
        menu_null_dir[i] = D_NORTH;
        my_facing[i] = D_NORTH;
        tap_control[i] = 0;
        for (int j = 0; j < 4; ++j) menu_control[i][j] = 0;
        
        dungeon_view[i].reset(new LocalDungeonView(cfg, approach_offset));
        mini_map[i].reset(new LocalMiniMap(cfg));
        status_display[i].reset(new LocalStatusDisplay(cfg, potion_renderer, skull_renderer,
                                                       menu_gfx_centre, menu_gfx_empty, menu_gfx_highlight));

        won[i] = lost[i] = false;
        game_over_time[i] = -9999;
        flash_screen_start[i] = -9999;

        prev_gui_width[i] = prev_gui_height[i] = -1;
    }

    won[1] = lost[1] = false;
}

LocalDisplay::~LocalDisplay()
{
    // empty dtor - needed because of auto_ptrs in the header file.
}

void LocalDisplay::recalculateTime()
{
    const int time_now = timer.getMsec();
    if (last_time == -1) last_time = time_now;  // initialization.
    const int time_delta = time_now - last_time;
    last_time = time_now;
    
    time += time_delta;
    for (int plyr_num = 0; plyr_num < 2; ++plyr_num) {
        if (dungeon_view[plyr_num].get()) dungeon_view[plyr_num]->addToTime(time_delta);
    }
}

void LocalDisplay::setupGui(int x, int y, int width, int height, GfxManager &gm)
{
    const gcn::Color bg_col(0x66, 0x66, 0x44);
    
    // TODO: better font loading method?
    if (!gui_font) {
        gm.setFontSize(14);
        gui_font.reset(new Coercri::CGFont(gm.getFont()));
        gui_font->setColor(Coercri::Color(255,255,255));
        gm.setFontSize(13);
        gui_small_font.reset(new Coercri::CGFont(gm.getFont()));
        gui_small_font->setColor(Coercri::Color(255,255,255));
    }

    // Resize and reposition the container
    container.setSize(width, height);
    container.setPosition(x, y);

    // Add text field and label, and the two buttons.

    send_button.reset(new gcn::Button("Send"));
    send_button->addActionListener(this);
    send_button->setFocusable(false);
    send_button->setBaseColor(gcn::Color(0x66, 0x66, 0x44));
    send_button->setFont(gui_small_font.get());
    send_button->adjustSize();
    const int chat_y = height - send_button->getHeight();
    container.add(send_button.get(), width - send_button->getWidth(), chat_y);

    clear_button.reset(new gcn::Button("Clear"));
    clear_button->addActionListener(this);
    clear_button->setFocusable(false);
    clear_button->setBaseColor(gcn::Color(0x66, 0x66, 0x44));
    clear_button->setFont(gui_small_font.get());
    clear_button->adjustSize();
    container.add(clear_button.get(), send_button->getX() - clear_button->getWidth() - 2, chat_y);
        
    chat_field.reset(new gcn::TextField);
    chat_field->setBackgroundColor(bg_col);
    chat_field->setForegroundColor(gcn::Color(255,255,255));
    chat_field->setFont(gui_font.get());
    chat_field->adjustSize();
    chat_field->setWidth(clear_button->getX() - 2);
    chat_field->addActionListener(this);
    const int chat_y2 = chat_y + std::max(0, (send_button->getHeight() - chat_field->getHeight())/2);
    container.add(chat_field.get(), 0, chat_y2);
    chat_field->requestFocus();

    chat_label.reset(new gcn::Label("Type here to chat"));
    chat_label->setFont(gui_font.get());
    chat_label->adjustSize();
    container.add(chat_label.get(), 0, height - send_button->getHeight() - 1 - chat_label->getHeight());

    const int y2 = chat_y - 1 - chat_label->getHeight() - 7;


    // Add the listbox for the chat
    chat_listbox.reset(new ListBoxNoMouse);
    chat_listbox->setBackgroundColor(gcn::Color(0,0,0));
    chat_listbox->setSelectionColor(gcn::Color(0,0,0));
    chat_listbox->setFont(gui_font.get());
    chat_list.setGuiParams(chat_listbox->getFont(), width - DEFAULT_SCROLLBAR_WIDTH);
    chat_listbox->setListModel(&chat_list);
    chat_listbox->setWidth(width - DEFAULT_SCROLLBAR_WIDTH);
    chat_listbox->setFocusable(false);
    
    chat_scrollarea.reset(new gcn::ScrollArea);
    chat_scrollarea->setContent(chat_listbox.get());
    chat_scrollarea->setSize(width, y2);
    chat_scrollarea->setScrollbarWidth(DEFAULT_SCROLLBAR_WIDTH);
    chat_scrollarea->setFrameSize(0);
    chat_scrollarea->setHorizontalScrollPolicy(gcn::ScrollArea::SHOW_NEVER);
    chat_scrollarea->setVerticalScrollPolicy(gcn::ScrollArea::SHOW_ALWAYS);
    chat_scrollarea->setBaseColor(bg_col);
    chat_scrollarea->setBackgroundColor(gcn::Color(0,0,0));
    container.add(chat_scrollarea.get(), 0, 0);


    /*
    std::string msg0 = status_display[plyr_num]->getQuestMsg();
    const bool special_quest_msg = !msg0.empty();

    if (!special_quest_msg) {
        if (status_display[plyr_num]->getQuestIcons().empty()) {
            msg0 = "Quest: Duel to the Death";
        } else {
            msg0 = "Quest Requirements:";
        }
    }

    typedef std::vector<StatusDisplay::QuestIconInfo> QuestIcons;
    const QuestIcons & qi = status_display[plyr_num]->getQuestIcons();
    std::string msg1;
    if (special_quest_msg && !qi.empty()) {
        msg1 = "Gems Required:";
    }

    std::string msg2 = "Exit Point: TODO";
    
    msg0_label[plyr_num].reset(new gcn::Label(msg0));
    msg0_label[plyr_num]->setFont(gui_font.get()); 
    msg0_label[plyr_num]->adjustSize();
    msg0_panel[plyr_num].reset(new GuiPanel2(msg0_label[plyr_num].get()));
    container.add(msg0_panel[plyr_num].get(), x, y);

    int xnext = x + msg0_panel[plyr_num]->getWidth();

    if (!msg1.empty()) {
        msg1_label[plyr_num].reset(new gcn::Label(msg1));
        msg1_label[plyr_num]->setFont(gui_font.get());
        msg1_label[plyr_num]->adjustSize();
        msg1_panel[plyr_num].reset(new GuiPanel2(msg1_label[plyr_num].get()));
        container.add(msg1_panel[plyr_num].get(), xnext, y);
        xnext += msg1_panel[plyr_num]->getWidth();
    }

    icon_panels[plyr_num].clear();
    for (QuestIcons::const_iterator it = qi.begin(); it != qi.end(); ++it) {
        for (int i = 0; i < it->num_required; ++i) {
            boost::shared_ptr<GuiPanel2> panel(new GuiPanel2(0));
            panel->setSize(32, msg0_panel[plyr_num]->getHeight());  // TODO set size properly
            icon_panels[plyr_num].push_back(panel);
            container.add(panel.get(), xnext, y);
            xnext += panel->getWidth();
        }
    }
    
    if (!msg2.empty()) {
        msg2_label[plyr_num].reset(new gcn::Label(msg2));
        msg2_label[plyr_num]->setFont(gui_font.get());
        msg2_label[plyr_num]->adjustSize();
        msg2_panel[plyr_num].reset(new GuiPanel2(msg2_label[plyr_num].get()));
        container.add(msg2_panel[plyr_num].get(), xnext, y);
        xnext += msg2_panel[plyr_num]->getWidth();
    }
    */
}

void LocalDisplay::action(const gcn::ActionEvent &event)
{
    if (event.getSource() == chat_field.get() || event.getSource() == send_button.get()) {
        const std::string & msg = chat_field->getText();
        if (!msg.empty()) {
            knights_client.sendChatMessage(msg);
            chat_field->setText("");
        }
    } else if (event.getSource() == clear_button.get()) {
        chat_field->setText("");
    }
}

void LocalDisplay::draw(Coercri::GfxContext &gc, GfxManager &gm, 
                        int plyr_num, bool is_split_screen, 
                        PauseDisplay pause_display)
{
    recalculateTime();
    
    if ((plyr_num != 0 && plyr_num != 1) || !dungeon_view[plyr_num].get()) {
        throw UnexpectedError("bad player num in LocalDisplay");
    }

    if (chat_updated) {
        chat_updated = false;
        if (chat_scrollarea.get()) {
            // auto scroll chat window to bottom
            chat_listbox->adjustSize();
            gcn::Rectangle rect(0, chat_listbox->getHeight() - chat_listbox->getFont()->getHeight(),
                                1, chat_listbox->getFont()->getHeight());
            chat_scrollarea->showWidgetPart(chat_listbox.get(), rect);
        }
    }
    

    // work out scale factors    
    const int dpy_width = gc.getWidth();
    const int dpy_height = gc.getHeight();

    const int ref_width = ref_vp_width * 2 + ref_gutter;
    const int ref_height = ref_vp_height;

    const float ideal_x_scale = float(dpy_width) / float(ref_width);
    const float ideal_y_scale = float(dpy_height) / float(ref_height);

    float x_scale, y_scale, dummy;
    gm.getGfxResizer()->roundScaleFactor(ideal_x_scale, x_scale, dummy);
    gm.getGfxResizer()->roundScaleFactor(ideal_y_scale, y_scale, dummy);

    const float scale = std::min(x_scale, y_scale);


    // work out dungeon and status area sizes
    const int pixels_per_square = ref_pixels_per_square * scale;
    const int dungeon_width = pixels_per_square * dungeon_tiles_x;
    const int dungeon_height = pixels_per_square * dungeon_tiles_y;
    const float dungeon_scale_factor = float(pixels_per_square) / float(ref_pixels_per_square);

    int status_area_width, status_area_height;
    status_display[plyr_num]->getSize(scale, status_area_width, status_area_height);

    const int excess_height = std::max(0, dpy_height - dungeon_height - status_area_height);
    const int dungeon_area_height = dungeon_height + excess_height / 2;
    
    // work out dungeon position
    int dungeon_x;
    if (is_split_screen) {
        // draw to left or right side
        const int margin = std::max(0, dpy_width/2 - dungeon_width)/5;
        dungeon_x = plyr_num == 0 ? margin : dpy_width - dungeon_width - margin;
    } else {
        // centre horizontally
        dungeon_x = (dpy_width - dungeon_width)/2;
    }
    const int dungeon_y = excess_height / 4;

    // work out status area position
    int status_area_x;
    if (is_split_screen) {
        const int margin = std::max(0, dpy_width/2 - status_area_width)/5;
        status_area_x = plyr_num == 0 ? margin : dpy_width - status_area_width - margin;
    } else {
        status_area_x = dpy_width/2 + (dpy_width/2 - status_area_width)/2;
    }
    const int status_area_y = dungeon_area_height + excess_height / 4;


    // if split screen mode, then set clip rectangle
    // (this is to stop anything accidentally being drawn on the opponent's side of the screen).
    if (is_split_screen) {
        Coercri::Rectangle r(plyr_num == 0 ? 0 : dpy_width/2, 0, dpy_width/2, dpy_height);
        gc.setClipRectangle(r);
    }

    // work out what we are drawing
    bool draw_in_game_screen = true;
    bool draw_winner_loser_screen = false;
    if (won[plyr_num] || lost[plyr_num]) {
        if (game_over_time[plyr_num] < 0) {
            game_over_time[plyr_num] = time;
        }

        if (time >= game_over_time[plyr_num] + game_over_t1) {
            draw_in_game_screen = false;
        }
        if (time > game_over_time[plyr_num] + game_over_t2) {
            draw_winner_loser_screen = true;
        }
    }

    bool draw_pause_display = false;
    if (pause_display != PD_OFF && !won[plyr_num] && !lost[plyr_num]) {
        if (pause_display == PD_OPAQUE) draw_in_game_screen = false;
        draw_pause_display = true;
    }
    
    
    // set an appropriate font size, based on dungeon_scale_factor
    int font_size = Round(ref_font_size * dungeon_scale_factor);
    if (draw_winner_loser_screen) font_size = int(font_size * game_over_ratio);
    gm.setFontSize(std::max(min_font_size, font_size));


    // recreate gui if the screen size has changed.
    const int gui_margin = 6;
    const int gui_width = dpy_width/2 - 2*gui_margin;
    const int gui_height_reduction = draw_winner_loser_screen ? 3*gm.getFont()->getTextHeight() : 0;
    const int gui_height = dpy_height - dungeon_area_height - 2*gui_margin - gui_height_reduction;
    if (!is_split_screen) {
        if (gui_width != prev_gui_width[plyr_num] || gui_height != prev_gui_height[plyr_num] 
        || status_display[plyr_num]->needGuiUpdate()) {
            prev_gui_width[plyr_num] = gui_width;
            prev_gui_height[plyr_num] = gui_height;
            setupGui(gui_margin, dungeon_area_height + gui_margin + gui_height_reduction,
                     gui_width, gui_height, gm);
        }
    }


    int base_x, base_w;
    if (is_split_screen) {
        base_x = plyr_num == 0 ? 0 : dpy_width/2;
        base_w = dpy_width/2;
    } else {
        base_x = 0;
        base_w = dpy_width;
    }    

    // Draw Winner/Loser screen
    if (draw_winner_loser_screen) {
        boost::shared_ptr<ColourChange> my_colour_change = dungeon_view[plyr_num]->getMyColourChange();

        const Graphic * image = 0;
        
        if (won[plyr_num]) {
            if (winner_image) {
                image = winner_image;
            } else {
                gc.drawText(base_x + 10, 50, *gm.getFont(), "WINNER", Coercri::Color(255,255,255), true);
            }
        } else {
            if (loser_image) {
                image = loser_image;
            } else {
                gc.drawText(base_x + 10, 50, *gm.getFont(), "LOSER", Coercri::Color(255,255,255), true);
            }
        }

        if (image) {
            int width, height;
            gm.loadGraphic(*image);
            gm.getGraphicSize(*image, width, height);
            const int new_width = scale*width;
            const int new_height = scale*height;
            const int th = gm.getFont()->getTextHeight();
            const int image_y = is_split_screen ? (dpy_height - (new_height + 2*th))/2 : dungeon_y;
            gm.drawTransformedGraphic(gc, base_x + (base_w - new_width)/2, image_y,
                                      *image, new_width, new_height,
                                      my_colour_change ? *my_colour_change : ColourChange());

            if (!ready_msg_sent) {
                const std::string msg = config_map.getString("game_over_msg");
                const int x = base_x + base_w/2 - gm.getFont()->getTextWidth(msg)/2;
                gc.drawText(x,
                            image_y + new_height + 2*th,
                            *gm.getFont(),
                            msg,
                            Coercri::Color(config_map.getInt("game_over_r"), 
                                           config_map.getInt("game_over_g"), 
                                           config_map.getInt("game_over_b")),
                            true);
            }
        }
    }

    // Draw In-Game Screen
    if (draw_in_game_screen) {

        // Check whether screen should flash
        const bool screen_flash = (time >= flash_screen_start[plyr_num] 
                                  && time <= flash_screen_start[plyr_num] + config_map.getInt("screen_flash_time"));

        // Draw the coloured background if screen is flashing
        if (screen_flash) {
            Coercri::Color col(config_map.getInt("screen_flash_r"), 
                               config_map.getInt("screen_flash_g"), 
                               config_map.getInt("screen_flash_b"));
            Coercri::Rectangle rect(dungeon_x, dungeon_y, dungeon_width, dungeon_height);
            gc.fillRectangle(rect, col);
        }
        
        // Draw dungeon view. Pass in screen_flash as this only draws
        // part of the dungeon when screen is flashing.
        dungeon_view[plyr_num]->draw(gc, gm, screen_flash, dungeon_x, dungeon_y, dungeon_width,
            dungeon_height, pixels_per_square, dungeon_scale_factor);

        // Draw status area
        status_display[plyr_num]->draw(gc, gm, time, scale, status_area_x, status_area_y,
                                       dungeon_view[plyr_num]->aliveRecently(),
                                       *mini_map[plyr_num]);
    }

    // Handle fade to black if necessary.
    // This is done in a quick-n-dirty way, by drawing a huge
    // semi-transparent black rectangle over the whole screen...
    if (won[plyr_num] || lost[plyr_num]) {
        float alpha = 0.0f;
        const int delta = time - game_over_time[plyr_num];
        if (delta <= 0) {
            alpha = 0.0f;
        } else if (delta < game_over_t1) {
            alpha = float(delta) / float(game_over_t1);
        } else if (delta < game_over_t2) {
            alpha = 1.0f;
        } else if (delta < game_over_t3) {
            alpha = 1.0f - float(delta - game_over_t2) / float(game_over_t3 - game_over_t2);
        } else {
            alpha = 0.0f;
        }
        if (alpha != 0.0f) {
            Coercri::Rectangle rect(0, 0, dpy_width, dpy_height);
            gc.fillRectangle(rect, Coercri::Color(0, 0, 0, int(alpha*255)));
        }
    }
    
    // Pause Display
    if (draw_pause_display) {

        if (pause_display == PD_TRANSPARENT) {
            // Darken the visible area.
            Coercri::Rectangle rect(base_x, 0, base_w, dpy_height);
            gc.fillRectangle(rect, Coercri::Color(0, 0, 0, config_map.getInt("pausealpha")));
        }
        
        const Coercri::Color col1(config_map.getInt("pause1r"), config_map.getInt("pause1g"), config_map.getInt("pause1b"));
        const Coercri::Color col2(config_map.getInt("pause2r"), config_map.getInt("pause2g"), config_map.getInt("pause2b"));
        const Coercri::Color col3(config_map.getInt("pause3r"), config_map.getInt("pause3g"), config_map.getInt("pause3b"));
        const Coercri::Color col4(config_map.getInt("pause4r"), config_map.getInt("pause4g"), config_map.getInt("pause4b"));

        const int th = gm.getFont()->getTextHeight();

        const int nlines =
            (pause_display != PD_TRANSPARENT ? 4 : 1)
            + (menu_strings ? menu_strings->size() : 0);

        int y = std::max(0, dpy_height - nlines*th) / 2;
        
        if (pause_display != PD_TRANSPARENT) {
            const string str1 = "GAME PAUSED", str2 = "Press TAB to return to game";
            const int str1w = gm.getFont()->getTextWidth(str1), str2w = gm.getFont()->getTextWidth(str2);
            
            gc.drawText(base_x + base_w/2 - str1w/2, y, *gm.getFont(), str1, col1, true);
            y += th;
            gc.drawText(base_x + base_w/2 - str2w/2, y, *gm.getFont(), str2, col2, true);
            y += th;
            y += th;
            y += th;
        } else {
            y += th;
        }

        if (menu_strings) {
            int maxw1 = 0, maxw2 = 0;
            for (int i = 0; i < menu_strings->size(); ++i) {
                const pair<string, string> & p = (*menu_strings)[i];
                maxw1 = std::max(maxw1, gm.getFont()->getTextWidth(p.first + ":"));
                maxw2 = std::max(maxw2, gm.getFont()->getTextWidth(p.second));
            }

            const int xofs = std::max(0, base_w - maxw1 - maxw2) / 2;
            
            for (int i = 0; i < menu_strings->size(); ++i) {
                pair<string, string> p = (*menu_strings)[i];
                if (!p.first.empty()) p.first += ":";

                const int w1 = gm.getFont()->getTextWidth(p.first);
                const int w2 = gm.getFont()->getTextWidth(p.second);

                const int left_bound = w1 + 5;
                const int right_bound = base_w - w2;
                int x = maxw1 + 10;
                // Move left if necessary, so as not to run off right side of screen.
                if (x > right_bound) x = right_bound;
                // Move right if necessary, so as not to overlap left-hand text.
                if (x < left_bound) x = left_bound;
                
                gc.drawText(xofs + base_x,     y, *gm.getFont(), p.first, col3, true);
                gc.drawText(xofs + base_x + x, y, *gm.getFont(), p.second, col4, true);
                y += th;
            }
        }
    }

    gc.clearClipRectangle();
}

void LocalDisplay::playSounds(SoundManager &sm)
{
    for (std::vector<MySound>::iterator it = sounds.begin(); it != sounds.end(); ++it) {
        sm.playSound(*it->sound, it->frequency);
    }
    sounds.clear();
}

const UserControl * LocalDisplay::readControl(int plyr)
{
    if (plyr != 0 && plyr != 1) {
        throw UnexpectedError("bad player number in LocalDisplay");
    }
        
    // set up some constants
    const int menu_delay = config_map.getInt("menu_delay"); // cutoff btwn 'tapping' and 'holding' fire.
    const bool approached = dungeon_view[plyr]->isApproached();
    
    // get current controller state
    MapDirection ctrlr_dir = D_NORTH;
    bool ctrlr_centred = true, ctrlr_fire = false, suicide_keys = false;
    if (plyr == 0 && controller1) {
        controller1->get(ctrlr_dir, ctrlr_centred, ctrlr_fire, suicide_keys);
    } else if (plyr == 1 && controller2) {
        controller2->get(ctrlr_dir, ctrlr_centred, ctrlr_fire, suicide_keys);
    }
    
    // if suicide keys held, then we can exit immediately...
    if (suicide_keys) {
        return standard_controls[SC_SUICIDE];
    }

    // work out 'long' fire state
    enum FireState { NONE, TAPPED, HELD } long_fire = NONE;
    if (ctrlr_fire) {
        if (fire_start_time[plyr] == 0) {
            // fire_start_time: set to time when fire was pressed down (or 0 if fire is
            // currently released)
            // attack_mode: if fire+direction is used to attack, then attack_mode is set true,
            // which disables menus and 'tapping'.
            fire_start_time[plyr] = time;
            attack_mode[plyr] = false;
        }
        if (time >= fire_start_time[plyr] + menu_delay && !attack_mode[plyr]) {
            long_fire = HELD;
        }
    } else {
        // do a 'tap' if allowed
        if (fire_start_time[plyr] > 0 && fire_start_time[plyr] < time + menu_delay && !attack_mode[plyr]) {
            long_fire = TAPPED;
        }
        // reset fire state when fire is released
        fire_start_time[plyr] = 0;
        attack_mode[plyr] = false;
        // reset allow_menu_open when they let go of fire when the menu is closed.
        if (!status_display[plyr]->isMenuOpen()) allow_menu_open[plyr] = true;
    }
    
    if (status_display[plyr]->isMenuOpen()) {
        // Menu open

        if (long_fire != HELD) {
            // close the menu
            status_display[plyr]->setMenuOpen(false);
            menu_null[plyr] = M_OK;
        } else if (approached != approached_when_menu_was_opened[plyr]) {
            // This traps the situation where we have picked a lock and now start moving towards the door.
            // We want to automatically close the menu in this case, so that the knight is ready to attack
            // whatever is beyond the door. Also turn off allow_menu_open until fire is released.
            status_display[plyr]->setMenuOpen(false);
            allow_menu_open[plyr] = false;
        } else {
            if (menu_null[plyr]==M_NULL && (ctrlr_centred || ctrlr_dir != menu_null_dir[plyr])) {
                // Cancel null dir.
                menu_null[plyr] = M_OK;
            }
            if (!ctrlr_centred && (menu_null[plyr]!=M_NULL || ctrlr_dir != menu_null_dir[plyr])) {
                // A menu option has been selected
                const UserControl *c = menu_control[plyr][ctrlr_dir];
                if (c) {
                    // stop the action being executed more than once in a row.
                    // (continuous actions will be stopped only if the control disappears,
                    // which is what M_CTS is for.)
                    menu_null[plyr] = (c->isContinuous()? M_CTS: M_NULL);
                    menu_null_dir[plyr] = ctrlr_dir;
                }
                return c;
            }
        }

    } else if (approached) {
        // Approached, and menu closed
        if ((ctrlr_centred || ctrlr_dir != my_facing[plyr]) && !ctrlr_fire) {
            // Withdraw if controller is no longer pointing in the approach direction
            // and fire is released.
            return standard_controls[SC_WITHDRAW];
        } else {
            if (long_fire == HELD && allow_menu_open) {
                status_display[plyr]->setMenuOpen(true);
                approached_when_menu_was_opened[plyr] = true;
                menu_null[plyr] = M_NULL;
                menu_null_dir[plyr] = my_facing[plyr];
            } else if (long_fire == TAPPED) { 
                return tap_control[plyr];
            }
        }
    } else {
        // Not approached, and menu closed
        if (ctrlr_centred) {
            if (long_fire == HELD && allow_menu_open[plyr]) {
                status_display[plyr]->setMenuOpen(true);
                approached_when_menu_was_opened[plyr] = false;
                menu_null[plyr] = M_OK;
            } else if (long_fire == TAPPED) {
                return tap_control[plyr];
            }
        } else {
            if (ctrlr_fire) {
                attack_mode[plyr] = true;
                return standard_controls[SC_ATTACK + ctrlr_dir];
            } else {
                my_facing[plyr] = ctrlr_dir;
                return standard_controls[SC_MOVE + ctrlr_dir];
            }
        }
    }

    // No commands coming from the controller at present.
    return 0;
}

DungeonView & LocalDisplay::getDungeonView(int plyr)
{
    if (plyr == 0 || plyr == 1) {
        return *dungeon_view[plyr];
    } else {
        throw UnexpectedError("bad player number in LocalDisplay");
    }
}

MiniMap & LocalDisplay::getMiniMap(int plyr)
{
    if (plyr == 0 || plyr == 1) {
        return *mini_map[plyr];
    } else {
        throw UnexpectedError("bad player number in LocalDisplay");
    }
}

StatusDisplay & LocalDisplay::getStatusDisplay(int plyr)
{
    if (plyr == 0 || plyr == 1) {
        return *status_display[plyr];
    } else {
        throw UnexpectedError("bad player number in LocalDisplay");
    }
}

void LocalDisplay::playSound(int plyr, const Sound &sound, int frequency)
{
    if (plyr != 0 && plyr != 1) {
        throw UnexpectedError("bad player number in LocalDisplay");
    } else {
        // see if we have this sound already. this prevents the same sound
        // being played twice (and therefore at double volume!) in the
        // split screen mode when both knights are in the same room.
        for (std::vector<MySound>::iterator it = sounds.begin(); it != sounds.end(); ++it) {
            if (it->sound == &sound && it->frequency == frequency && !it->plyr[plyr]) {
                it->plyr[plyr] = true;
                return;
            }
        }
        
        // otherwise, add it to the list
        MySound s;
        s.sound = &sound;
        s.frequency = frequency;
        s.plyr[plyr] = true;
        s.plyr[1-plyr] = false;
        sounds.push_back(s);
    }
}

void LocalDisplay::winGame(int plyr)
{
    if (plyr != 0 && plyr != 1) {
        throw UnexpectedError("bad player number in LocalDisplay");
    } else {
        won[plyr] = true;
    }
}

void LocalDisplay::loseGame(int plyr)
{
    if (plyr != 0 && plyr != 1) {
        throw UnexpectedError("bad player number in LocalDisplay");
    } else {
        lost[plyr] = true;
    }
}

void LocalDisplay::setAvailableControls(int plyr, const std::vector<std::pair<const UserControl*, bool> > &controls)
{
    if (plyr != 0 && plyr != 1) {
        throw UnexpectedError("bad player number in LocalDisplay");
    }
    
    const UserControl * prev_menu_control[4];
    for (int i = 0; i < 4; ++i) {
        prev_menu_control[i] = menu_control[plyr][i];
        menu_control[plyr][i] = 0;
    }
    tap_control[plyr] = 0;
    int tap_pri = 0;

    // All controls in 'controls' are available to our
    // knight. So put them into menu_control or tap_control as
    // appropriate.

    status_display[plyr]->clearMenuGraphics();

    for (std::vector<std::pair<const UserControl*, bool> >::const_iterator ctrl = controls.begin();
    ctrl != controls.end(); ++ctrl) {
        // Find the (primary) control with the highest tap priority:
        // (NB control is primary iff ctrl->second == true)
        if (ctrl->first->getTapPriority() > tap_pri && ctrl->second) {
            tap_pri = ctrl->first->getTapPriority();
            tap_control[plyr] = ctrl->first;
        }
        if (ctrl->first->getMenuGraphic() != 0) {
            MapDirection d = ctrl->first->getMenuDirection();
            // If no control in direction d, or if the control in direction d has
            // property MS_WEAK, then save ctrl->first into menu position d.
            if (menu_control[plyr][d] == 0 ||
            (menu_control[plyr][d]->getMenuSpecial() & UserControl::MS_WEAK)) {
                setMenuControl(plyr, d, ctrl->first, prev_menu_control[d]);
            }
        }
    }

    // Now try to place controls that could not be put in their preferred position.
    for (std::vector<std::pair<const UserControl *, bool> >::const_iterator ctrl = controls.begin(); 
    ctrl != controls.end(); ++ctrl) {
        if (ctrl->first->getMenuGraphic() != 0) {
            MapDirection d = ctrl->first->getMenuDirection();
            if (menu_control[plyr][d] != ctrl->first &&
            (ctrl->first->getMenuSpecial() & UserControl::MS_WEAK) == 0) {
                for (int i=0; i<4; ++i) {
                    if (menu_control[plyr][i] == 0) {
                        setMenuControl(plyr, MapDirection(i), ctrl->first, prev_menu_control[i]);
                        break;
                    }
                }
            }
        }
    }
}

void LocalDisplay::setMenuControl(int plyr, MapDirection d, const UserControl *ctrl, const UserControl *prev)
{
    if (menu_null[plyr]==M_CTS && menu_null_dir[plyr]==d && ctrl != prev) {
        menu_null[plyr] = M_NULL;
    }
    menu_control[plyr][d] = ctrl;
    status_display[plyr]->setMenuGraphic(d, ctrl ? ctrl->getMenuGraphic() : 0);
}

void LocalDisplay::setMenuHighlight(int plyr, const UserControl *highlight)
{
    if (plyr != 0 && plyr != 1) {
        throw UnexpectedError("bad player number in LocalDisplay");
    }

    if (highlight) {
        for (int i = 0; i < 4; ++i) {
            if (menu_control[plyr][i] == highlight) {
                status_display[plyr]->setMenuHighlight(MapDirection(i));
                return;
            }
        }
    }
    status_display[plyr]->clearMenuHighlight();
}

void LocalDisplay::flashScreen(int plyr, int delay)
{
    if (plyr != 0 && plyr != 1) {
        throw UnexpectedError("bad player number in LocalDisplay");
    } else {
        flash_screen_start[plyr] = time + delay;
    }
}
