/*
 * loading_screen.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "config_map.hpp"
#include "error_screen.hpp"
#include "game_manager.hpp"
#include "kconfig_loader.hpp" // for KConfigError
#include "knights_app.hpp"
#include "knights_client.hpp"
#include "knights_server.hpp"
#include "loading_screen.hpp"
#include "x_centre.hpp"

LoadingScreen::Loader::Loader(KnightsApp &app)
    : knights_app(app)
{ }

void LoadingScreen::Loader::operator()()
{
    try {
        // the loading of the config is slow, especially in a debug
        // build, so we do it in a separate thread. the loading of
        // everything else is faster, and requires calls into coercri
        // (which complicates thread safety) so we do it in the main
        // thread.
        knights_app.loadKnightsConfig();
    } catch (KConfig::KConfigError &kce) {
        kconfig_error.reset(new KConfig::KConfigError(kce));
    } catch (std::exception &e) {
        error_msg = e.what();
        if (error_msg.empty()) error_msg = " ";
    } catch (...) {
        error_msg = "Unknown Error";
    }
}

bool LoadingScreen::start(KnightsApp &ka, boost::shared_ptr<Coercri::Window>, gcn::Gui &)
{
    knights_app = &ka;
    loader.reset(new Loader(ka));

    boost::thread new_thread(boost::ref(*loader));
    loader_thread.swap(new_thread);
    return false;
}

LoadingScreen::LoadingScreen(int port)
    : knights_app(0), server_port(port)
{ }

LoadingScreen::~LoadingScreen()
{
    // We shouldn't really be able to get here unless the loader has finished,
    // but just in case it is still running:
    loader_thread.join();
}

void LoadingScreen::update()
{
    if (!knights_app || !loader) return;

    if (loader_thread.joinable()) {
        loader_thread.timed_join(boost::posix_time::milliseconds(10));
        return;
    }

    // KConfigErrors should be re-thrown in this thread. Other errors should just go directly
    // to ErrorScreen.
    if (loader->kconfig_error.get())
    {
        throw *loader->kconfig_error;
    }
    if (!loader->error_msg.empty()) {
        std::auto_ptr<Screen> error_screen(new ErrorScreen("Loading Failed: " + loader->error_msg));
        knights_app->requestScreenChange(error_screen);
        return;
    }
    
    if (server_port < 0) {
        // create server & game
        KnightsServer *server = knights_app->createLocalServer(true);
        server->startNewGame("#SplitScreenGame");

        // create a local client
        boost::shared_ptr<KnightsClient> client = knights_app->openLocalConnection();
        knights_app->createGameManager(client);
        client->setClientCallbacks(&knights_app->getGameManager());

        // set dummy player name.
        client->setPlayerName("#SplitScreenPlayer");
        
        // Join the game in split screen mode. This will take us to MenuScreen automatically.
        knights_app->getGameManager().tryJoinGameSplitScreen("#SplitScreenGame");

    } else {
        // create server & game
        KnightsServer *server = knights_app->createServer(server_port, false);
        server->startNewGame("#LanGame");

        // create a local client
        boost::shared_ptr<KnightsClient> client = knights_app->openLocalConnection();
        knights_app->createGameManager(client);
        client->setClientCallbacks(&knights_app->getGameManager());

        // Start responding to broadcasts
        knights_app->startBroadcastReplies(server_port);

        // Set our player name.
        client->setPlayerName("Player 1");
        
        // Join the game -- this will take us to MenuScreen automatically.
        knights_app->getGameManager().tryJoinGame("#LanGame");
    }

    loader.reset();
}

void LoadingScreen::draw(Coercri::GfxContext &gc)
{
    if (!knights_app) return;
    XCentre(gc, *knights_app->getFont(), knights_app->getFont()->getTextHeight() + 15, "LOADING");
}
