/*
 * knights_app.hpp
 *
 * Top level class for Knights. The main() function constructs this
 * class and calls runKnights().
 * 
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef KNIGHTS_APP_HPP
#define KNIGHTS_APP_HPP

// coercri
#include "gfx/font.hpp"
#include "gfx/key_code.hpp"
#include "network/network_driver.hpp"
#include "timer/timer.hpp"

#include "boost/shared_ptr.hpp"

#include <iosfwd>
#include <memory>
#include <string>
using namespace std;

class ConfigMap;
class Controller;
class GameManager;
class GfxManager;
class Graphic;
class KnightsAppImpl;
class KnightsClient;
class KnightsServer;
class Options;
class PotionRenderer;
class Screen;
class SkullRenderer;
class SoundManager;

class KnightsApp {
public:
    explicit KnightsApp(const string &resource_dir);

    // Start the main loop
    void runKnights();

    // Screen change and Quit requests
    void requestScreenChange(auto_ptr<Screen> screen);
    void requestQuit();

    // Resets global game data (called when we go back to the title screen)
    void resetAll();

    // Just reset graphics/sounds, don't clear network connections
    void unloadGraphicsAndSounds();
    

    // "Get" methods
    boost::shared_ptr<Coercri::Font> getFont() const;  // the Coercri font being used for the guichan global widget font.
    const Controller & getLeftController() const;
    const Controller & getRightController() const;
    GfxManager & getGfxManager() const;
    SoundManager & getSoundManager() const;
    void getDesktopResolution(int &w, int &h) const;
    const Graphic * getWinnerImage() const;
    const Graphic * getLoserImage() const;
    const Graphic * getMenuGfxCentre() const;
    const Graphic * getMenuGfxEmpty() const;
    const Graphic * getMenuGfxHighlight() const;
    const PotionRenderer * getPotionRenderer() const;
    const SkullRenderer * getSkullRenderer() const;
    const ConfigMap & getConfigMap() const;
    Coercri::Timer & getTimer() const;
    Coercri::NetworkDriver & getNetworkDriver() const;

    // get the width/height to be used for windowed mode.
    void getWindowedModeSize(int &width, int &height);
    
    // Options. KnightsApp ctor will load options from disk. setAndSaveOptions will set new
    // options and save them back to disk.
    const Options & getOptions() const;
    void setAndSaveOptions(const Options &);

    // Filter GUI keys. This is used in-game to prevent the controller keys from
    // affecting the chat box.
    void setFilteredKeys(const std::vector<Coercri::KeyCode> &keys);
    
    // to work around a guichan bug. This should be called if new gui-buttons are created
    void repeatLastMouseInput();


    //
    // GameManager methods
    //

    void createGameManager(boost::shared_ptr<KnightsClient> knights_client);
    void destroyGameManager();
    GameManager & getGameManager();
    

    //
    // KnightsClient methods
    //
    // Create KnightsClient objects. Packets will be automatically
    // routed to/from the network (or locally) as needed, and
    // callbacks will be called automatically from within the main
    // loop. All connections will be closed (and the corresponding
    // KnightsClient objects deleted) when resetAll() is called.
    //
    // (Currently the code only ever creates one KnightsClient at a
    // time, but it is theoretically possible to create multiple
    // KnightsClients if required.)
    //
    
    boost::shared_ptr<KnightsClient> openRemoteConnection(const std::string &address, int port);
    boost::shared_ptr<KnightsClient> openLocalConnection();
    void closeConnection(KnightsClient *);


    //
    // KnightsServer methods
    //
    // Create KnightsServer objects. In the case of createServer() we
    // start listening for incoming connections on the given port, as
    // well as "local" connections (via openLocalConnection). For
    // createLocalServer() we will accept local connections only. Only
    // one server can be created at a time. The server will be
    // deleted (and any network connections closed) when resetAll is
    // called.
    //

    void loadKnightsConfig();
    KnightsServer * createServer(int port, bool allow_split_screen);
    KnightsServer * createLocalServer(bool allow_split_screen);

    //
    // LAN broadcast replies. (Reset when we return to title screen.)
    // NOTE: the port number for this is fixed currently, see net_msgs.hpp.
    //
    void startBroadcastReplies(int server_port);

        
private:
    void executeScreenChange();
    void setupControllers();
    void setupGfxResizer();
    
private:
    boost::shared_ptr<KnightsAppImpl> pimpl;
};

#endif
