/*
 * knights_app.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "config_map.hpp"
#include "error_screen.hpp"
#include "game_manager.hpp"
#include "gfx_manager.hpp"
#include "gfx_resizer_compose.hpp"
#include "gfx_resizer_nearest_nbr.hpp"
#include "gfx_resizer_scale2x.hpp"
#include "graphic.hpp"
#include "keyboard_controller.hpp"
#include "knights_app.hpp"
#include "knights_client.hpp"
#include "knights_config.hpp"
#include "knights_server.hpp"
#include "load_font.hpp"
#include "menu_selections.hpp"
#include "my_exceptions.hpp"
#include "net_msgs.hpp"
#include "options.hpp"
#include "potion_renderer.hpp"
#include "rstream.hpp"
#include "skull_renderer.hpp"
#include "sound_manager.hpp"
#include "title_screen.hpp"

#include "kconfig_loader.hpp"
#include "kfile.hpp"

// coercri
#include "enet/enet_network_driver.hpp"
#include "gcn/cg_font.hpp"
#include "gcn/cg_listener.hpp"
#include "gfx/window_listener.hpp"
#include "network/network_connection.hpp"
#include "network/udp_socket.hpp"
#include "sdl/gfx/sdl_gfx_driver.hpp"
#include "sdl/gfx/sdl_ttf_loader.hpp"
#include "sdl/sound/sdl_sound_driver.hpp"
#include "sdl/timer/sdl_timer.hpp"

#include <curl/curl.h>

#include "guichan.hpp"

#include "boost/scoped_ptr.hpp"

#include <cstdlib>

#ifdef WIN32
#include <shlobj.h>
#include "SDL_syswm.h"
#endif

#ifdef min
#undef min
#endif
#ifdef max
#undef max
#endif

/////////////////////////////////////////////////////////////
// WindowListener derived classes
/////////////////////////////////////////////////////////////

namespace {
    class WindowCloseListener : public Coercri::WindowListener {
    public:
        explicit WindowCloseListener(KnightsApp &k) : app(k) { }
        void onClose() { app.requestQuit(); }
    private:
        KnightsApp &app;
    };

    class FilteringCGListener : public Coercri::CGListener {
    public:
        explicit FilteringCGListener(const std::vector<Coercri::KeyCode> &k,
                                     boost::shared_ptr<Coercri::Window> window,
                                     boost::shared_ptr<gcn::Gui> gui,
                                     boost::shared_ptr<Coercri::Timer> timer)
            : Coercri::CGListener(window, gui, timer), keys(k) { }
        
        void onKey(Coercri::KeyEvent ke, Coercri::KeyCode kc, int character, int modifiers)
        {
            // If the key is being filtered then don't let the gui see it at all
            // (although we do allow the modifier keys through...).
            // Otherwise pass it through to the base class.
            if (std::find(keys.begin(), keys.end(), kc) == keys.end()) {
                Coercri::CGListener::onKey(ke, kc, character, modifiers);
            }
        }

    private:
        const std::vector<Coercri::KeyCode> &keys;
    };
}


/////////////////////////////////////////////////////////////
// Definition of KnightsAppImpl
/////////////////////////////////////////////////////////////

class KnightsAppImpl {
public:
    boost::shared_ptr<Coercri::GfxDriver> gfx_driver;
    boost::shared_ptr<Coercri::SoundDriver> sound_driver;
    boost::shared_ptr<Coercri::Timer> timer;
    boost::shared_ptr<Coercri::TTFLoader> ttf_loader;
    boost::shared_ptr<Coercri::Window> window;
    boost::shared_ptr<WindowCloseListener> wcl;

    boost::shared_ptr<gcn::Gui> gui;
    boost::shared_ptr<FilteringCGListener> cg_listener;

    // we do not want these shared, we want to be sure there is only one copy of each:
    auto_ptr<Screen> current_screen, requested_screen;
    
    unsigned int last_time;  // last time update() was called
    unsigned int update_interval;
    
    bool running;

    boost::shared_ptr<GfxManager> gfx_manager;
    boost::shared_ptr<SoundManager> sound_manager;
    boost::shared_ptr<Controller> left_controller, right_controller;

    boost::shared_ptr<Coercri::Font> font;
    boost::scoped_ptr<gcn::Font> gcn_font;

    boost::scoped_ptr<Options> options;
    std::string options_filename;

    ConfigMap config_map;

    std::auto_ptr<Graphic> winner_image;
    std::auto_ptr<Graphic> loser_image;
    std::auto_ptr<Graphic> menu_gfx_centre, menu_gfx_empty, menu_gfx_highlight;
    std::vector<boost::shared_ptr<Graphic> > extra_graphics;
    std::auto_ptr<PotionRenderer> potion_renderer;
    std::auto_ptr<SkullRenderer> skull_renderer;


    // network driver
    boost::scoped_ptr<Coercri::NetworkDriver> net_driver;

    
    // server object.
    boost::shared_ptr<const KnightsConfig> knights_config;
    boost::scoped_ptr<KnightsServer> knights_server;

    // incoming network connections (to the knights_server above)
    struct IncomingConn {
        ServerConnection * server_conn;
        boost::shared_ptr<Coercri::NetworkConnection> remote;
    };
    std::vector<IncomingConn> incoming_conns;

    // outgoing network connections
    struct OutgoingConn {
        boost::shared_ptr<KnightsClient> knights_client;
        boost::shared_ptr<Coercri::NetworkConnection> remote;
    };
    std::vector<OutgoingConn> outgoing_conns;

    // local "loopback" connections
    struct LocalConn {
        boost::shared_ptr<KnightsClient> knights_client;
        ServerConnection * server_conn;
    };
    std::vector<LocalConn> local_conns;

    // broadcast socket
    boost::shared_ptr<Coercri::UDPSocket> broadcast_socket;
    unsigned int broadcast_last_time;
    int server_port;
    
    // game manager
    boost::scoped_ptr<GameManager> game_manager;

    std::vector<Coercri::KeyCode> filtered_keys;
    
    // functions
    KnightsAppImpl() : last_time(0), update_interval(0), running(true) { }

    void saveOptions();
    
    void popPotionSetup(KConfig::KFile&);
    void popSkullSetup(KConfig::KFile&);

    bool processIncomingNetMsgs();
    bool processOutgoingNetMsgs();
    bool processLocalNetMsgs();
    bool processBroadcastMsgs();
};

///////////////////////////////////////////////////////////
// Implementation of KConfigSource to use RStreams
///////////////////////////////////////////////////////////
namespace {
    struct MyFileLoader : KConfig::KConfigSource {
        virtual boost::shared_ptr<std::istream> openFile(const std::string &filename) {
            boost::shared_ptr<std::istream> result(new RStream(filename));
            return result;
        }
    };
}

/////////////////////////////////////////////////////
// Constructor (One-Time Initialization)
/////////////////////////////////////////////////////

KnightsApp::KnightsApp(const string &resource_dir)
    : pimpl(new KnightsAppImpl)
{
    const char * game_name = "Knights";

    // initialize resource lib
    RStream::Initialize(resource_dir);

    // read the client config
    {
        MyFileLoader file_loader;
        KConfig::RandomIntContainer dummy;
        KConfig::KFile kf("client_config.txt", file_loader, dummy);
        kf.pushSymbol("MISC_CONFIG");
        PopConfigMap(kf, pimpl->config_map);
        kf.pushSymbol("WINNER_IMAGE");
        pimpl->winner_image = PopGraphic(kf);
        kf.pushSymbol("LOSER_IMAGE");
        pimpl->loser_image = PopGraphic(kf);
        kf.pushSymbol("MENU_CENTRE");
        pimpl->menu_gfx_centre = PopGraphic(kf);
        kf.pushSymbol("MENU_EMPTY");
        pimpl->menu_gfx_empty = PopGraphic(kf);
        kf.pushSymbol("MENU_HIGHLIGHT");
        pimpl->menu_gfx_highlight = PopGraphic(kf);
        kf.pushSymbol("POTION_SETUP");
        pimpl->popPotionSetup(kf);
        kf.pushSymbol("SKULL_SETUP");
        pimpl->popSkullSetup(kf);
    }

    // initialize game options
    pimpl->options.reset(new Options);
#ifdef WIN32
    // options stored in "app data" directory
    TCHAR szPath[MAX_PATH];
    if(SUCCEEDED(SHGetFolderPath(NULL, 
                                 CSIDL_APPDATA,
                                 NULL,
                                 0,
                                 szPath))) {
        pimpl->options_filename = szPath;
        pimpl->options_filename += "/knights_config.txt";
    }
#else
    // options stored in home directory (assume getenv("HOME") will work)
    pimpl->options_filename = std::getenv("HOME");
    pimpl->options_filename += "/.knights_config";
#endif
    if (!pimpl->options_filename.empty()) {
        ifstream str(pimpl->options_filename.c_str());
        *pimpl->options = LoadOptions(str);
    }
    
    
    // Set up Coercri
    // (Here we use SDL, but a different backend could be selected
    // just by changing the following few lines.)
    pimpl->gfx_driver.reset(new Coercri::SDLGfxDriver);
    pimpl->gfx_driver->setKeyRepeat(true);

#ifndef DISABLE_SOUND
    try {
        pimpl->sound_driver.reset(new Coercri::SDLSoundDriver(pimpl->config_map.getInt("sound_volume") / 100.0f));
    } catch (std::exception &e) {
        // Print warning message to cout, and continue without a sound_driver
        std::cout << "Problem initializing audio: " << e.what() << std::endl;
    }
#endif
    
    pimpl->timer.reset(new Coercri::SDLTimer);
    pimpl->ttf_loader.reset(new Coercri::SDLTTFLoader);

    // use the enet network driver
    pimpl->net_driver.reset(new Coercri::EnetNetworkDriver(1, 1));
    pimpl->net_driver->enableServer(false);  // start off disabled.

    // initialize curl. tell it not to init winsock since EnetNetworkDriver will have done that already.
    curl_global_init(CURL_GLOBAL_NOTHING);
    
    // Set the Windows resource number for the window icon
    pimpl->gfx_driver->setWindowsIcon(1);
    
    // Open the game window.
    bool fullscreen = pimpl->options->fullscreen;
    int width, height;
    if (fullscreen) {
        getDesktopResolution(width, height);
    } else {
        getWindowedModeSize(width, height);
    }
    pimpl->window = pimpl->gfx_driver->createWindow(width, height, true, fullscreen, game_name);
    
    // Get the font names.
    // NOTE: we always try to load TTF before bitmap fonts, irrespective of the order they are in the file.
    vector<string> ttf_font_names, bitmap_font_names;
    {
        RStream str("fonts.txt");

        while (str) {
            string x;
            getline(str, x);
        
            // Left trim
            x.erase(x.begin(), find_if(x.begin(), x.end(), not1(ptr_fun<int,int>(isspace))));
            // Right trim
            x.erase(find_if(x.rbegin(), x.rend(), not1(ptr_fun<int,int>(isspace))).base(), x.end());
            
            if (x.empty()) continue;
        
            if (x[0] == '+') bitmap_font_names.push_back(x.substr(1));
            else if (x[0] != '#') ttf_font_names.push_back(x);
        }
    }
    
    // Font for the gui
    pimpl->font = LoadFont(*pimpl->ttf_loader,
                           ttf_font_names, 
                           bitmap_font_names, 
                           pimpl->config_map.getInt("font_size"));

    // Setup gfx & sound managers
    pimpl->gfx_manager.reset(new GfxManager(pimpl->gfx_driver, pimpl->ttf_loader, ttf_font_names, bitmap_font_names,
                                            pimpl->config_map.getInt("chat_font_size")));
    setupGfxResizer();
    pimpl->sound_manager.reset(new SoundManager(pimpl->sound_driver));

    // Load all "extra graphics" at this point
    // These are added as "permanent" so that we don't have to keep reloading them after a reset.
    pimpl->gfx_manager->loadGraphic(*pimpl->menu_gfx_centre, true);
    pimpl->gfx_manager->loadGraphic(*pimpl->menu_gfx_empty, true);
    pimpl->gfx_manager->loadGraphic(*pimpl->menu_gfx_highlight, true);
    for (std::vector<boost::shared_ptr<Graphic> >::iterator it = pimpl->extra_graphics.begin(); it != pimpl->extra_graphics.end(); ++it) {
        pimpl->gfx_manager->loadGraphic(**it, true);
    }
    
    // Setup Controllers
    setupControllers();

    // Set up Guichan and CGListener
    // NOTE: No need to set Input and Graphics for guichan since this is handled by CGListener.
    pimpl->gui.reset(new gcn::Gui);
    pimpl->cg_listener.reset(new FilteringCGListener(pimpl->filtered_keys, pimpl->window, pimpl->gui, pimpl->timer));
    pimpl->window->addWindowListener(pimpl->cg_listener.get());

    // Add the WindowCloseListener
    pimpl->wcl.reset(new WindowCloseListener(*this));
    pimpl->window->addWindowListener(pimpl->wcl.get());
    
    // Set guichan's global widget font
    pimpl->gcn_font.reset(new Coercri::CGFont(pimpl->font, 
                                              Coercri::Color(0,0,0), 
                                              pimpl->config_map.getInt("font_antialias") != 0));
    gcn::Widget::setGlobalFont(pimpl->gcn_font.get());
}

void KnightsApp::getWindowedModeSize(int &width, int &height)
{
    getDesktopResolution(width, height);
    width = std::min(width, std::max(100, pimpl->options->window_width));
    height = std::min(height, std::max(100, pimpl->options->window_height));
}


//////////////////////////////////////////////
// Screen handling
//////////////////////////////////////////////

void KnightsApp::requestScreenChange(auto_ptr<Screen> screen)
{
    pimpl->requested_screen = screen;
}

void KnightsApp::executeScreenChange()
{
    if (pimpl->requested_screen.get() != pimpl->current_screen.get() && pimpl->requested_screen.get()) {
        // Close the gui
        pimpl->gui->setTop(0);
        pimpl->cg_listener->disableGui();
        pimpl->gui->setTabbingEnabled(true);

        // Swap screens
        pimpl->current_screen = pimpl->requested_screen;   // clears requested_screen

        // Initialize the new screen, and bring up the gui if required
        const bool requires_gui = pimpl->current_screen->start(*this, pimpl->window, *pimpl->gui);
        if (requires_gui) pimpl->cg_listener->enableGui();

        // Work around bug in guichan where Button::mHasMouse will be set to false for new buttons
        // even if the mouse is actually within the button's area when it is created.
        repeatLastMouseInput();
        
        // Reset timer.
        pimpl->last_time = pimpl->timer->getMsec();
        pimpl->update_interval = pimpl->current_screen->getUpdateInterval();

        // Ensure the screen gets repainted
        pimpl->window->invalidateAll();
    }
}

void KnightsApp::requestQuit()
{
    pimpl->running = false;
}

void KnightsApp::repeatLastMouseInput()
{
    pimpl->cg_listener->repeatLastMouseInput();
}

//////////////////////////////////////////////
// Game state handling
//////////////////////////////////////////////

void KnightsApp::resetAll()
{
    // Clean up outgoing connections
    for (std::vector<KnightsAppImpl::OutgoingConn>::iterator it = pimpl->outgoing_conns.begin();
    it != pimpl->outgoing_conns.end(); ++it) {
        ASSERT(it->knights_client && "resetAll: outgoing");
        it->knights_client->setClientCallbacks(0);
        it->knights_client->setKnightsCallbacks(0);
        it->knights_client->connectionClosed();
        it->remote->close();
    }
    pimpl->outgoing_conns.clear();

    // Clean up incoming connections
    for (std::vector<KnightsAppImpl::IncomingConn>::iterator it = pimpl->incoming_conns.begin();
    it != pimpl->incoming_conns.end(); ++it) {
        ASSERT(pimpl->knights_server && "resetAll: incoming");  // otherwise there would not be any incoming_conns!
        pimpl->knights_server->connectionClosed(*it->server_conn);
        it->remote->close();
    }
    pimpl->incoming_conns.clear();

    // Clean up local connections
    for (std::vector<KnightsAppImpl::LocalConn>::iterator it = pimpl->local_conns.begin();
    it != pimpl->local_conns.end(); ++it) {
        ASSERT(pimpl->knights_server && "resetAll: local");  // otherwise there would not be any incoming_conns!
        ASSERT(it->knights_client && "resetAll: local");
        it->knights_client->setClientCallbacks(0);
        it->knights_client->setKnightsCallbacks(0);
        it->knights_client->connectionClosed();
        pimpl->knights_server->connectionClosed(*it->server_conn);
    }
    pimpl->local_conns.clear();

    // Shut down the server
    pimpl->knights_server.reset();
    pimpl->net_driver->enableServer(false);

    // Shut down the GameManager (important to do this BEFORE accessing gfxmanager/soundmanager)
    destroyGameManager();

    // Wipe out all loaded graphics / sounds.
    unloadGraphicsAndSounds();

    // Stop responding to broadcasts
    pimpl->broadcast_socket.reset();
    pimpl->broadcast_last_time = 0;
    pimpl->server_port = 0;
}    

void KnightsApp::unloadGraphicsAndSounds()
{
    pimpl->gfx_manager->deleteAllGraphics();
    pimpl->sound_manager->clear();
}

//////////////////////////////////////////////
// Other methods
//////////////////////////////////////////////

void KnightsApp::setupGfxResizer()
{
    boost::shared_ptr<GfxResizer> gfx_resizer_nearest_nbr(new GfxResizerNearestNbr);
    boost::shared_ptr<GfxResizer> gfx_resizer_scale2x(new GfxResizerScale2x);
    boost::shared_ptr<GfxResizer> gfx_resizer_comp(new GfxResizerCompose(gfx_resizer_nearest_nbr,
        pimpl->options->use_scale2x ? gfx_resizer_scale2x : boost::shared_ptr<GfxResizer>(),
        !pimpl->options->allow_non_integer_scaling));
    pimpl->gfx_manager->setGfxResizer(gfx_resizer_comp);
}

void KnightsApp::setupControllers()
{
    pimpl->left_controller.reset(new KeyboardController(pimpl->options->ctrls[0][0],
                                                        pimpl->options->ctrls[0][3],
                                                        pimpl->options->ctrls[0][1],
                                                        pimpl->options->ctrls[0][2],
                                                        pimpl->options->ctrls[0][4],
                                                        pimpl->options->ctrls[0][5],
                                                        pimpl->window));
    pimpl->right_controller.reset(new KeyboardController(pimpl->options->ctrls[1][0],
                                                         pimpl->options->ctrls[1][3],
                                                         pimpl->options->ctrls[1][1],
                                                         pimpl->options->ctrls[1][2],
                                                         pimpl->options->ctrls[1][4],
                                                         pimpl->options->ctrls[1][5],
                                                         pimpl->window));
}    

boost::shared_ptr<Coercri::Font> KnightsApp::getFont() const
{
    return pimpl->font;
}

const Controller & KnightsApp::getLeftController() const
{
    return *pimpl->left_controller;
}

const Controller & KnightsApp::getRightController() const
{
    return *pimpl->right_controller;
}

GfxManager & KnightsApp::getGfxManager() const
{
    return *pimpl->gfx_manager;
}

SoundManager & KnightsApp::getSoundManager() const
{
    return *pimpl->sound_manager;
}

void KnightsApp::getDesktopResolution(int &w, int &h) const
{
    Coercri::GfxDriver::DisplayMode mode = pimpl->gfx_driver->getDesktopMode();
    w = mode.width;
    h = mode.height;
}

const Graphic * KnightsApp::getLoserImage() const
{
    return pimpl->loser_image.get();
}

const Graphic * KnightsApp::getWinnerImage() const
{
    return pimpl->winner_image.get();
}

const Graphic * KnightsApp::getMenuGfxCentre() const
{
    return pimpl->menu_gfx_centre.get();
}

const Graphic * KnightsApp::getMenuGfxEmpty() const
{
    return pimpl->menu_gfx_empty.get();
}

const Graphic * KnightsApp::getMenuGfxHighlight() const
{
    return pimpl->menu_gfx_highlight.get();
}

const PotionRenderer * KnightsApp::getPotionRenderer() const
{
    return pimpl->potion_renderer.get();
}

const SkullRenderer * KnightsApp::getSkullRenderer() const
{
    return pimpl->skull_renderer.get();
}

const ConfigMap & KnightsApp::getConfigMap() const
{
    return pimpl->config_map;
}

Coercri::Timer & KnightsApp::getTimer() const
{
    return *pimpl->timer;
}

Coercri::NetworkDriver & KnightsApp::getNetworkDriver() const
{
    return *pimpl->net_driver;
}

const Options & KnightsApp::getOptions() const
{
    return *pimpl->options;
}

void KnightsApp::setAndSaveOptions(const Options &opts)
{
    *pimpl->options = opts;
    pimpl->saveOptions();
    setupControllers();
    setupGfxResizer();
}

void KnightsApp::setFilteredKeys(const std::vector<Coercri::KeyCode> &keys)
{
    pimpl->filtered_keys = keys;
}

void KnightsAppImpl::saveOptions()
{
    if (!options_filename.empty()) {
        ofstream str(options_filename.c_str(), std::ios_base::out | std::ios_base::trunc);
        SaveOptions(*options, str);
    }
}

void KnightsAppImpl::popPotionSetup(KConfig::KFile &kf)
{
    potion_renderer.reset(new PotionRenderer(config_map));
    
    KConfig::KFile::Table tab(kf, "PotionSetup");
    {
        tab.push("colours");
        KConfig::KFile::List lst(kf, "PotionColours");
        for (int i = 0; i < lst.getSize(); ++i) {
            lst.push(i);
            int x = kf.popInt();
            potion_renderer->addColour(Colour((x>>16)&255, (x>>8)&255, x&255));
        }
    }
    {
        tab.push("graphics");
        KConfig::KFile::List lst(kf, "PotionGraphics");
        for (int i = 0; i < lst.getSize(); ++i) {
            lst.push(i);
            boost::shared_ptr<Graphic> g(PopGraphic(kf).release());
            extra_graphics.push_back(g);
            potion_renderer->addGraphic(g.get());
        }
    }
}

void KnightsAppImpl::popSkullSetup(KConfig::KFile &kf)
{
    skull_renderer.reset(new SkullRenderer);

    KConfig::KFile::Table tab(kf, "SkullSetup");
    {
        tab.push("columns");
        KConfig::KFile::List lst(kf, "SkullColumns");
        for (int i=0; i<lst.getSize(); ++i) {
            lst.push(i);
            skull_renderer->addColumn(kf.popInt());
        }
    }
    {
        tab.push("graphics");
        KConfig::KFile::List lst(kf, "SkullGraphics");
        for (int i=0; i<lst.getSize(); ++i) {
            lst.push(i);
            boost::shared_ptr<Graphic> g(PopGraphic(kf).release());
            extra_graphics.push_back(g);
            skull_renderer->addGraphic(g.get());
        }
    }
    {
        tab.push("rows");
        KConfig::KFile::List lst(kf, "SkullRows");
        for (int i=0; i<lst.getSize(); ++i) {
            lst.push(i);
            skull_renderer->addRow(kf.popInt());
        }
    }
}

//////////////////////////////////////////////
// Main Routine (runKnights)
//////////////////////////////////////////////

void KnightsApp::runKnights()
{
    // Go to title screen
    auto_ptr<Screen> title_screen(new TitleScreen);
    requestScreenChange(title_screen);
    executeScreenChange();

    // Error Handling system.
    string error;
    int num_errors = 0;
    
    // Main Loop
    while (pimpl->running) {

        bool did_something = false;
        
        if (!error.empty() && num_errors > 100) {
            // Too many errors
            // Abort game.
            throw UnexpectedError("FATAL: " + error);
        }
        
        try {
            if (!error.empty()) {
                // Error detected: try to go to ErrorScreen.
                // (If going to ErrorScreen itself throws an error, then num_errors will keep increasing
                // and eventually we will abort, see above.)
                auto_ptr<Screen> screen;
                screen.reset(new ErrorScreen(error));
                requestScreenChange(screen);
                executeScreenChange();
                error.clear();
                ++num_errors;
                did_something = true;
            }

            // Handle screen changes if necessary
            executeScreenChange();

            // Before running network events, make sure outgoing messages have been sent
            did_something = pimpl->processOutgoingNetMsgs() || did_something;
            
            // Empty out all event queues
            while (pimpl->gfx_driver->pollEvents()) did_something = true;
            while (pimpl->cg_listener->processInput()) did_something = true;
            while (pimpl->net_driver && pimpl->net_driver->doEvents()) did_something = true;

            // Make sure any incoming net messages (or dropped connections) get processed at this point.
            did_something = pimpl->processIncomingNetMsgs() || did_something;

            // Also: make sure local net messages are routed properly
            did_something = pimpl->processLocalNetMsgs() || did_something;

            // Also: do the broadcast msgs
            did_something = pimpl->processBroadcastMsgs() || did_something;
            
            // Do updates
            if (pimpl->update_interval > 0) {
                const unsigned int time_now = pimpl->timer->getMsec();
                const unsigned int time_since_last_update = time_now - pimpl->last_time;

                // if it's time for an update then do one.
                if (time_since_last_update >= pimpl->update_interval) {
                    pimpl->last_time = time_now;
                    pimpl->current_screen->update();
                    did_something = true;
                }
            }

            // Draw the screen if necessary
            if (pimpl->window->needsRepaint()) {
                std::auto_ptr<Coercri::GfxContext> gc = pimpl->window->createGfxContext();
                gc->clearClipRectangle();
                gc->clearScreen(Coercri::Color(0,0,0));
                pimpl->cg_listener->draw(*gc);
                pimpl->current_screen->draw(*gc);
                pimpl->window->cancelInvalidRegion();
                did_something = true;
            }

            // Sleep if necessary (so that we don't consume 100% CPU).
            if (!did_something) {
                unsigned int delay = pimpl->timer->getMsec() - pimpl->last_time;
                if (delay > 20u) delay = 20u;
                if (delay > 0u) pimpl->timer->sleepMsec(delay);
                num_errors = 0;  // if we can get to a sleep w/o error, then assume things
                                 // are going well & can reset num_errors.
            }

        } catch (KConfig::KConfigError&) {
            // These are big and better displayed on stdout than in a guichan dialog box:
            throw;
        } catch (std::exception &e) {
            error = string("ERROR: ") + e.what();
        } catch (gcn::Exception &e) {
            error = string("Guichan Error: ") + e.getMessage();
        } catch (...) {
            error = "Unknown Error";
        }
    }

#ifdef WIN32
    if (pimpl->options) {
        // Get our HWND from SDL.
        SDL_SysWMinfo inf = {0};
        SDL_VERSION(&inf.version);
        if (SDL_GetWMInfo(&inf)) {

            // If the window is maximized then don't try to save the size
            // as it then looks weird when we next start up the program.
            // Ditto if we are in full screen mode.
            if (!IsZoomed(inf.window) && !pimpl->options->fullscreen) {
            
                // Get current window size
                int width = 0, height = 0;
                pimpl->window->getSize(width, height);

                // If it differs from the saved window size, then update the
                // saved window size and re-save the options file.
                if (width != pimpl->options->window_width || height != pimpl->options->window_height) {
                    pimpl->options->window_width = width;
                    pimpl->options->window_height = height;
                    pimpl->saveOptions();
                }
            }
        }
    }
    // TODO: is there some equivalent to the above on Linux?
#endif

    // Delete things -- this is to make sure that destructors run in the order we want them to.
    pimpl->current_screen.reset();
    pimpl->requested_screen.reset();
    pimpl->left_controller.reset();
    pimpl->right_controller.reset();

    pimpl->sound_driver.reset();
    pimpl->window.reset();
    pimpl->wcl.reset();
    pimpl->cg_listener.reset();
    pimpl->gfx_driver.reset();

    // Call resetAll -- this closes network connections among other things.
    resetAll();

    // delete GfxManager, this has the last reference to the Coercri::GfxDriver. This 
    // will close the main window.
    pimpl->gfx_manager.reset();
    
    // Shut down curl.
    curl_global_cleanup();
    
    // Wait up to ten seconds for network connections to be cleaned up.
    const int WAIT_SECONDS = 10;
    for (int i = 0; i < WAIT_SECONDS*10; ++i) {
        if (!pimpl->net_driver->outstandingConnections()) break;
        while (pimpl->net_driver->doEvents()) ;
        pimpl->timer->sleepMsec(100);
    }
}


//////////////////////////////////////////////
// Network Game Handling
//////////////////////////////////////////////

boost::shared_ptr<KnightsClient> KnightsApp::openRemoteConnection(const std::string &address, int port)
{
    KnightsAppImpl::OutgoingConn out;
    out.knights_client.reset(new KnightsClient);
    out.remote = pimpl->net_driver->openConnection(address, port);
    pimpl->outgoing_conns.push_back(out);
    return out.knights_client;
}

boost::shared_ptr<KnightsClient> KnightsApp::openLocalConnection()
{
    if (!pimpl->knights_server) throw UnexpectedError("server must be created before calling openLocalConnection");
    KnightsAppImpl::LocalConn conn;
    conn.knights_client.reset(new KnightsClient);
    conn.server_conn = &pimpl->knights_server->newClientConnection();
    pimpl->local_conns.push_back(conn);
    return conn.knights_client;
}

void KnightsApp::closeConnection(KnightsClient *knights_client)
{
    // see if it's an outgoing (network) connection
    std::vector<KnightsAppImpl::OutgoingConn>::iterator it;
    for (it = pimpl->outgoing_conns.begin(); it != pimpl->outgoing_conns.end(); ++it) {
        if (it->knights_client.get() == knights_client) break;
    }
    if (it != pimpl->outgoing_conns.end()) {
        knights_client->connectionClosed();
        it->remote->close();
        pimpl->outgoing_conns.erase(it);
    } else {

        // see if it's a local connection
        std::vector<KnightsAppImpl::LocalConn>::iterator it;
        for (it = pimpl->local_conns.begin(); it != pimpl->local_conns.end(); ++it) {
            if (it->knights_client.get() == knights_client) break;
        }
        if (it != pimpl->local_conns.end()) {
            knights_client->connectionClosed();
            pimpl->knights_server->connectionClosed(*it->server_conn);
            pimpl->local_conns.erase(it);
        }
    }
}

void KnightsApp::loadKnightsConfig()
{
    if (!pimpl->knights_config) {
        pimpl->knights_config.reset(new KnightsConfig("server_config.txt"));
    }
}

KnightsServer * KnightsApp::createServer(int port, bool allow_split_screen)
{
    if (pimpl->knights_server) throw UnexpectedError("KnightsServer created twice");
    if (!pimpl->knights_config) throw UnexpectedError("KnightsConfig not loaded");
    pimpl->knights_server.reset(new KnightsServer(pimpl->knights_config, pimpl->timer, allow_split_screen, "", ""));
    pimpl->net_driver->setServerPort(port);
    pimpl->net_driver->enableServer(true);
    return pimpl->knights_server.get();
}

KnightsServer * KnightsApp::createLocalServer(bool allow_split_screen)
{
    if (pimpl->knights_server) throw UnexpectedError("KnightsServer created twice");
    if (!pimpl->knights_config) throw UnexpectedError("KnightsConfig not loaded");
    pimpl->knights_server.reset(new KnightsServer(pimpl->knights_config, pimpl->timer, allow_split_screen, "", ""));
    return pimpl->knights_server.get();
}

bool KnightsAppImpl::processIncomingNetMsgs()
{
    // Check for any incoming network data and route it to the
    // appropriate KnightsClient or KnightsServer object

    bool did_something = false;
    std::vector<unsigned char> net_msg;

    // Do "outgoing" connections first
    
    for (int i = 0; i < outgoing_conns.size(); ) {
        OutgoingConn &out = outgoing_conns[i];
        ASSERT(out.knights_client);

        // see if there is any data, if so, route it to the KnightsClient
        out.remote->receive(net_msg);
        if (!net_msg.empty()) {
            did_something = true;
            out.knights_client->receiveInputData(net_msg);
        }
        
        // if connection has dropped, then remove it from the list
        const Coercri::NetworkConnection::State state = out.remote->getState();
        if (state == Coercri::NetworkConnection::CLOSED || state == Coercri::NetworkConnection::FAILED) {
            if (state == Coercri::NetworkConnection::CLOSED) {
                out.knights_client->connectionClosed();
            } else {
                out.knights_client->connectionFailed();
            }
            outgoing_conns.erase(outgoing_conns.begin() + i);
            did_something = true;
        } else {
            ++i;
        }
    }

    // Now do "incoming" connections
    
    for (int i = 0; i < incoming_conns.size(); /* incremented below */) {
        ASSERT(knights_server && "processIncomingNetMsgs: loop over incoming connections");
        IncomingConn &in = incoming_conns[i];

        in.remote->receive(net_msg);
        if (!net_msg.empty()) {
            did_something = true;
            knights_server->receiveInputData(*in.server_conn, net_msg);
        }
        
        const Coercri::NetworkConnection::State state = in.remote->getState();
        ASSERT(state != Coercri::NetworkConnection::FAILED); // incoming connections can't fail...
        if (state == Coercri::NetworkConnection::CLOSED) {
            // connection lost: remove it from the list, and inform the server
            knights_server->connectionClosed(*in.server_conn);
            incoming_conns.erase(incoming_conns.begin() + i);
            did_something = true;
        } else {
            ++i;
        }
    }

    // Listen for new incoming connections.
    Coercri::NetworkDriver::Connections new_conns = net_driver->pollIncomingConnections();
    for (Coercri::NetworkDriver::Connections::const_iterator it = new_conns.begin(); it != new_conns.end(); ++it) {
        ASSERT(knights_server && "processIncomingNetMsgs: listen for new incoming connections");
        IncomingConn in;
        in.server_conn = &knights_server->newClientConnection();
        in.remote = *it;
        incoming_conns.push_back(in);
        did_something = true;
    }

    return did_something;
}

bool KnightsAppImpl::processOutgoingNetMsgs()
{
    // Pick up any outgoing network data and route it to the
    // appropriate NetworkConnection

    // NOTE: we don't bother to check for dropped connections here,
    // instead that is done in processIncomingNetMsgs.

    bool did_something = false;
    std::vector<unsigned char> net_msg;
    
    for (std::vector<OutgoingConn>::iterator it = outgoing_conns.begin(); it != outgoing_conns.end(); ++it) {
        ASSERT(it->knights_client && "processOutgoingNetMsgs");
        it->knights_client->getOutputData(net_msg);
        if (!net_msg.empty()) {
            did_something = true;
            it->remote->send(net_msg);
        }
    }

    for (std::vector<IncomingConn>::iterator it = incoming_conns.begin(); it != incoming_conns.end(); ++it) {
        ASSERT(knights_server && "processOutgoingNetMsgs");
        knights_server->getOutputData(*it->server_conn, net_msg);
        if (!net_msg.empty()) {
            did_something = true;
            it->remote->send(net_msg);
        }
    }

    return did_something;
}

bool KnightsAppImpl::processLocalNetMsgs()
{
    bool did_something = false;
    std::vector<unsigned char> net_msg;
    
    for (std::vector<LocalConn>::iterator it = local_conns.begin(); it != local_conns.end(); ++it) {
        ASSERT(knights_server && "processLocalNetMsgs");
        ASSERT(it->knights_client && "processLocalNetMsgs");

        it->knights_client->getOutputData(net_msg);
#ifdef DEBUG_NET_MSGS
        if (!net_msg.empty()) {
            OutputDebugString("client: ");
            for (int i = 0; i < net_msg.size(); ++i) {
                char buf[256] = {0};
                sprintf(buf, "%d ", int (net_msg[i]));
                OutputDebugString(buf);
            }
            OutputDebugString("\n");
        }
#endif
        if (!net_msg.empty()) did_something = true;
        knights_server->receiveInputData(*it->server_conn, net_msg);

        knights_server->getOutputData(*it->server_conn, net_msg);
#ifdef DEBUG_NET_MSGS
        if (!net_msg.empty()) {
            OutputDebugString("server: ");
            for (int i = 0; i < net_msg.size(); ++i) {
                char buf[256] = {0};
                sprintf(buf, "%d ", int (net_msg[i]));
                OutputDebugString(buf);
            }
            OutputDebugString("\n");
        }
#endif
        if (!net_msg.empty()) did_something = true;
        it->knights_client->receiveInputData(net_msg);
    }

    return did_something;
}

//////////////////////////////////////////////
// Broadcast Replies
//////////////////////////////////////////////

void KnightsApp::startBroadcastReplies(int server_port)
{
    pimpl->broadcast_socket = pimpl->net_driver->createUDPSocket(BROADCAST_PORT, true);
    pimpl->broadcast_last_time = 0;
    pimpl->server_port = server_port;
}

bool KnightsAppImpl::processBroadcastMsgs()
{
    if (!broadcast_socket) return false;
    
    // don't run this more than once per second.
    const unsigned int time_now = timer->getMsec();
    if (time_now - broadcast_last_time < 1000) return false;
    broadcast_last_time = time_now;
    
    const int num_players = knights_server ? knights_server->getNumberOfPlayers() : 0;

    // check for incoming messages, send replies if necessary.
    bool need_reply = false;
    std::string msg, address;
    while (broadcast_socket->receive(address, msg)) {
        if (num_players < 2 && msg == BROADCAST_PING_MSG) {
            need_reply = true;
        }
    }

    if (need_reply) {  // only need to send one reply at the most
        std::string reply = BROADCAST_PONG_HDR;
        reply += static_cast<unsigned char>(server_port >> 8);
        reply += static_cast<unsigned char>(server_port & 0xff);
        reply += 'L';
        reply += static_cast<unsigned char>(num_players >> 8);
        reply += static_cast<unsigned char>(num_players & 0xff);
        broadcast_socket->broadcast(BROADCAST_PORT, reply);
    }

    return (need_reply);
}


//////////////////////////////////////////////
// Game Manager
//////////////////////////////////////////////

void KnightsApp::createGameManager(boost::shared_ptr<KnightsClient> knights_client)
{
    if (pimpl->game_manager) throw UnexpectedError("GameManager created twice");
    pimpl->game_manager.reset(new GameManager(*this, knights_client, pimpl->timer));
}

void KnightsApp::destroyGameManager()
{
    pimpl->game_manager.reset();
}

GameManager & KnightsApp::getGameManager()
{
    if (!pimpl->game_manager) throw UnexpectedError("GameManager unavailable");
    return *pimpl->game_manager;
}
