/*
 * in_game_screen.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "client_config.hpp"
#include "config_map.hpp"
#include "controller.hpp"
#include "error_screen.hpp"
#include "game_manager.hpp"
#include "gfx_manager.hpp"
#include "in_game_screen.hpp"
#include "knights_app.hpp"
#include "knights_client.hpp"
#include "local_display.hpp"
#include "menu.hpp"
#include "menu_selections.hpp"
#include "menu_screen.hpp"
#include "my_exceptions.hpp"
#include "options.hpp"

// coercri
#include "gfx/rectangle.hpp"

InGameScreen::InGameScreen(KnightsApp &ka, boost::shared_ptr<KnightsClient> kc, 
                           boost::shared_ptr<const ClientConfig> cfg, int nplayers_)
    : pause_mode(false),
      window_minimized(false),
      auto_mouse(false),
      nplayers(nplayers_),
      prev_update_time(0),
      stage(LOADING_STAGE),
      knights_app(ka),
      knights_client(kc),
      client_config(cfg)
{
    if (nplayers == 1) {
        ka.setFilteredKeys(ka.getRightController().getReservedKeys());
    }
}

InGameScreen::~InGameScreen()
{
    // Delete LocalDisplay and remove it from the callback mechanism
    knights_client->setKnightsCallbacks(0);
    display.reset();
    
    if (window) {
        // Make sure the mouse pointer is shown after leaving in-game screen
        window->showMousePointer(true);
        // Remove myself from the window if needed
        window->rmWindowListener(this);
    }

    knights_app.setFilteredKeys(std::vector<Coercri::KeyCode>());
}

unsigned int InGameScreen::getUpdateInterval()
{
    return 1000 / static_cast<unsigned int>(knights_app.getConfigMap().getInt("fps"));
}

void InGameScreen::setupDisplay()
{
    // Set up menu strings.
    boost::shared_ptr<vector<pair<string, string> > > menu_strings(new vector<pair<string, string> >);
    knights_app.getGameManager().getMenuStrings(*menu_strings);
    
    // Create displays / callbacks.
    display.reset(new LocalDisplay(knights_app.getConfigMap(),
                                   client_config->approach_offset,
                                   knights_app.getWinnerImage(),
                                   knights_app.getLoserImage(),
                                   knights_app.getMenuGfxCentre(),
                                   knights_app.getMenuGfxEmpty(),
                                   knights_app.getMenuGfxHighlight(),
                                   knights_app.getPotionRenderer(),
                                   knights_app.getSkullRenderer(),
                                   menu_strings,
                                   client_config->standard_controls,
                                   nplayers == 2 ? &knights_app.getLeftController() : &knights_app.getRightController(),
                                   nplayers == 2 ? &knights_app.getRightController() : 0,
                                   nplayers,
                                   knights_app.getTimer(),
                                   knights_app.getGameManager().getChatList(),
                                   *knights_client,
                                   *container));
    knights_client->setKnightsCallbacks(display.get());
}

void InGameScreen::draw(Coercri::GfxContext &gc)
{
    if (!display) return;  // 'display' is not set until loading finishes.

    GfxManager &gm = knights_app.getGfxManager();
    
    // work out whether we want the pause screen.
    LocalDisplay::PauseDisplay pd = LocalDisplay::PD_OFF;
    if (pause_mode) {
        if (nplayers == 2) pd = LocalDisplay::PD_OPAQUE;  // local game
        else pd = LocalDisplay::PD_TRANSPARENT;          // network game
    }

    const bool split_screen = (nplayers==2);
    display->draw(gc, gm, 0, split_screen, pd);
    if (split_screen) display->draw(gc, gm, 1, true, pd);
}

void InGameScreen::update()
{
    if (!container) return;  // just in case

    switch (stage) {
    case LOADING_STAGE:
        // create the displays then tell the server we have finished
        // loading.

        // (There used to be more code here... but now the loading has
        // been moved to GameManager, there isn't much to do other
        // than setup the display.)
        
        setupDisplay();
        knights_client->finishedLoading();
        
        // ready to rock 'n' roll!
        stage = INGAME_STAGE;
        break;
        
    case INGAME_STAGE:
        {
            // Read controllers
            knights_client->sendControl(0, display->readControl(0));
            if (nplayers == 2) knights_client->sendControl(1, display->readControl(1));
            
            // Make sure sounds get played
            display->playSounds(knights_app.getSoundManager());

            // force a repaint
            window->invalidateAll();
        }
        break;
    }

    if (knights_app.getGameManager().getChatList().isUpdated()) {
        display->setChatUpdated();
    }
}

void InGameScreen::onMouseMove(int new_x, int new_y)
{
    if (auto_mouse) window->showMousePointer(true);
}

void InGameScreen::onMouseDown(int x, int y, Coercri::MouseButton b)
{
    if (b != Coercri::MB_LEFT) return;
    if (stage != INGAME_STAGE) return;
    if (display->isGameOver() && !display->getReadyFlag()) {
        knights_client->readyToEnd();
        display->setReadyFlag();
    }
}

void InGameScreen::onKey(Coercri::KeyEvent ke, Coercri::KeyCode kc, int character, int modifiers)
{
    if (auto_mouse) window->showMousePointer(false);
    
    // Keys only work when in game
    if (stage != INGAME_STAGE) return;

    // ESC in game to quit.
    if (ke == Coercri::KE_PRESSED && kc == Coercri::KC_ESCAPE) {
        knights_client->requestQuit();
        return;
    }

    // TAB to pause/unpause
    if (ke == Coercri::KE_PRESSED && kc == Coercri::KC_TAB) {
        pause_mode = !pause_mode;
        if (nplayers == 2) knights_client->setPauseMode(pause_mode || window_minimized);
        window->invalidateAll();   // make sure screen gets redrawn!
        return;
    }
}

void InGameScreen::onActivate()
{
    window_minimized = false;
    if (nplayers == 2) knights_client->setPauseMode(pause_mode || window_minimized);
}

void InGameScreen::onDeactivate()
{
    window_minimized = true;
    if (nplayers == 2) knights_client->setPauseMode(pause_mode || window_minimized);
}

bool InGameScreen::start(KnightsApp &app, boost::shared_ptr<Coercri::Window> win, gcn::Gui &gui)
{
    if (app.getOptions().fullscreen) win->showMousePointer(false);
    auto_mouse = app.getOptions().fullscreen && (nplayers == 1);
    
    win->addWindowListener(this);
    window = win;

    container.reset(new gcn::Container);
    container->setOpaque(false);
    gui.setTop(container.get());

    gui.setTabbingEnabled(false);
    
    return true;
}
