/*
 * game_manager.hpp
 *
 * Implements ClientCallbacks interface.
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef GAME_MANAGER_HPP
#define GAME_MANAGER_HPP

#include "client_callbacks.hpp"

// coercri includes
#include "gfx/font.hpp"
#include "timer/timer.hpp"

#include "guichan.hpp"

#include "boost/shared_ptr.hpp"
#include <deque>
#include <vector>

class GameManagerImpl;
class KnightsApp;
class KnightsClient;


// internal structs, used in ChatList
struct FormattedLine {
    bool firstline;
    std::string text;
};
class FontWrapper;

class ChatList : public gcn::ListModel {
public:
    explicit ChatList(int mm) : max_msgs(mm), width(999999), is_updated(false) { }
    
    void add(const std::string &msg);
    void setGuiParams(const Coercri::Font *new_font, int new_width);
    void setGuiParams(const gcn::Font *new_font, int new_width);
    void clear();

    int getNumberOfElements() { return formatted_lines.size(); }
    std::string getElementAt(int i) { if (i>=0 && i<formatted_lines.size()) return formatted_lines[i].text; else return ""; }

    bool isUpdated();  // clears is_updated flag afterwards. used for auto-scrolling to bottom.
    
private:
    void doSetWidth(int);
    void addFormattedLine(const std::string &msg);
    void rmFormattedLine();
    
private:
    int max_msgs;
    boost::shared_ptr<FontWrapper> font;
    int width;
    std::deque<std::string> lines;
    std::deque<FormattedLine> formatted_lines;
    bool is_updated;
};

// list of unique player names
class NameList : public gcn::ListModel {
public:
    void add(const std::string &x, const std::string &extra, int ord);
    void add(const std::vector<std::string> &x, const std::string &extra, int ord);
    void add(const std::vector<std::pair<std::string, std::string> > &x, int ord);
    void alter(const std::string &x, const std::string &extra, int ord);
    void clearReady();
    void clear();
    void remove(const std::string &x);
    int getNumberOfElements();
    std::string getElementAt(int i);
private:
    struct Name {
        std::string name;
        std::string extra;
        int ord;
        bool operator<(const Name &other) const {
            return ord < other.ord || (ord == other.ord && name < other.name);
        }
    };
    std::vector<Name> names;
};


class GameManager : public ClientCallbacks {
public:
    GameManager(KnightsApp &ka, boost::shared_ptr<KnightsClient> client, boost::shared_ptr<Coercri::Timer> timer);

    // join game
    void tryJoinGame(const std::string &game_name);
    void tryJoinGameSplitScreen(const std::string &game_name);
    
    //
    // gui management
    //

    // lobby
    void setServerName(const std::string &server_name);
    const std::string & getServerName() const;
    const std::vector<GameInfo> & getGameInfos() const;
    bool isGameListUpdated();  // clears game_list_updated flag afterwards
    
    // this is set true if any of the following are invalid:
    // all players list, this game players list (incl. ready flags and house colours), chat list, quest description
    // after the call, the gui_invalid flag is cleared.
    bool isGuiInvalid();
    
    const std::string & getCurrentGameName() const;

    
    // quest selection menu
    void createMenuWidgets(gcn::ActionListener *listener,
                           int initial_x,
                           int initial_y,
                           gcn::Container &container,
                           gcn::Font *font_black,
                           gcn::Font *font_grey,
                           int &menu_width,
                           int &y_after_menu);
    void destroyMenuWidgets();
    void setMenuWidgetsEnabled(bool enabled);
    void getMenuStrings(std::vector<std::pair<std::string, std::string> > &menu_strings) const;
    bool getDropdownInfo(gcn::Widget *source, std::string &key, int &val) const;
    const std::string &getQuestDescription() const;
    
    // player lists etc.
    NameList & getAllPlayersList() const;   // names of players in lobby (NOT all players on the server).
    NameList & getObserversList() const;   // names of observers in the current game. excludes the two players.
    ChatList & getChatList() const;
    const std::string & getPlayerName(int) const;  // names of player 0/1 for this game
    bool getPlayerReady(int) const;  // ready-flags for players 0/1
    int getHouseColour(int) const;   // house-colour for players 0/1
    Coercri::Color getAvailHouseColour(int) const;  // translate house-colour-code into RGB colour.
    int getNumAvailHouseColours() const;
    int getMyPlayerNum() const;
    
    
    //
    // callback implementations
    //
    
    virtual void connectionLost();     // goes to ErrorScreen
    virtual void connectionFailed();   // goes to ErrorScreen
    virtual void serverError(const std::string &error);    // goes to ErrorScreen

    virtual void gameListReceived(const std::vector<GameInfo> &games);
    virtual void joinGameAccepted(boost::shared_ptr<const ClientConfig> conf,
                                  int my_player_num,
                                  const std::string &plyr1, bool ready1, int hse_col_1,
                                  const std::string &plyr2, bool ready2, int hse_col_2,
                                  const std::vector<std::string> &observers);    // goes to MenuScreen
    virtual void joinGameDenied(const std::string &reason);     // goes to ErrorScreen

    virtual void initialPlayerList(const std::vector<std::pair<std::string, std::string> > &names);
    virtual void passwordRequested(bool first_attempt);
    virtual void playerConnected(const std::string &name);
    virtual void playerDisconnected(const std::string &name);
    virtual void playerJoinedGame(const std::string &player, const std::string &game);
    virtual void playerLeftGame(const std::string &player, const std::string &game);
    virtual void setPlayerNum(const std::string &player, int new_num, bool is_me);
    
    virtual void leaveGame();     // goes to lobby
    virtual void setMenuSelection(const std::string &key, int val, const std::vector<int> &allowed_vals);
    virtual void setQuestDescription(const std::string &quest_descr);
    virtual void startGame(int ndisplays);  // goes to InGameScreen
    virtual void gotoMenu();     // goes to MenuScreen

    virtual void playerJoinedThisGame(const std::string &name, int pnum, int house_col);
    virtual void playerLeftThisGame(const std::string &name);
    virtual void playerChangedName(const std::string &new_name, int player_num);
    virtual void setPlayerHouseColour(const std::string &name, int house_col);
    virtual void setAvailableHouseColours(const std::vector<Coercri::Color> &cols);
    virtual void setReady(const std::string &name, bool ready);
    
    virtual void chat(const std::string &whofrom, bool observer, const std::string &msg);
    virtual void announcement(const std::string &msg);
    

private:
    void updateMenuDropdown(const std::string &key);
    
private:
    boost::shared_ptr<GameManagerImpl> pimpl;
};

#endif
