/*
 * game_manager.cpp
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "client_config.hpp"
#include "config_map.hpp"
#include "copy_if.hpp"
#include "error_screen.hpp"
#include "game_manager.hpp"
#include "gfx_manager.hpp"
#include "in_game_screen.hpp"
#include "knights_app.hpp"
#include "knights_client.hpp"
#include "lobby_screen.hpp"
#include "menu.hpp"
#include "menu_item.hpp"
#include "menu_screen.hpp"
#include "menu_selections.hpp"
#include "my_exceptions.hpp"
#include "password_screen.hpp"
#include "sound_manager.hpp"
#include "text_formatter.hpp"
#include "title_screen.hpp"

#include "guichan.hpp"

#include <algorithm>

// annoyingly, since we have 2 types of fonts in the system, need to create
// a small font wrapper class here.
class FontWrapper {
public:
    virtual ~FontWrapper() { }
    virtual int getWidth(const std::string &) const = 0;
};

class GCNFontWrapper : public FontWrapper {
public:
    explicit GCNFontWrapper(const gcn::Font *f_) : f(f_) { }
    int getWidth(const std::string &s) const { return f ? f->getWidth(s) : 0; }
private:
    const gcn::Font *f;
};

class CoercriFontWrapper : public FontWrapper {
public:
    explicit CoercriFontWrapper(const Coercri::Font*f_) : f(f_) { }
    int getWidth(const std::string &s) const { return f? f->getTextWidth(s) : 0; }
private:
    const Coercri::Font *f;
};

namespace {
    struct ChatPrinter : Printer {
        ChatPrinter(std::deque<FormattedLine> &o, const FontWrapper *f) : output(o), font(f), printed(false) { }
        
        int getTextWidth(const std::string &text) { return font ? font->getWidth(text) : 0; }
        int getTextHeight() { return 1; }  // we want to count lines not pixels.
        void printLine(const std::string &text, int, bool) {
            FormattedLine f;
            f.firstline = !printed;
            f.text = text;
            output.push_back(f);
            printed = true;
        }

        std::deque<FormattedLine> &output;
        const FontWrapper *font;
        bool printed;
    };
}      

void ChatList::add(const std::string &msg)
{
    lines.push_back(msg);
    addFormattedLine(msg);
    if (lines.size() > max_msgs) {
        lines.pop_front();
        rmFormattedLine();
    }
}

void ChatList::setGuiParams(const gcn::Font *new_font, int new_width)
{
    font.reset(new GCNFontWrapper(new_font));
    doSetWidth(new_width);
}

void ChatList::setGuiParams(const Coercri::Font *new_font, int new_width)
{
    font.reset(new CoercriFontWrapper(new_font));
    doSetWidth(new_width);
}

void ChatList::doSetWidth(int new_width)
{
    width = new_width;
    formatted_lines.clear();
    for (std::deque<std::string>::const_iterator it = lines.begin(); it != lines.end(); ++it) {
        addFormattedLine(*it);
    }
}

void ChatList::clear()
{
    lines.clear();
    formatted_lines.clear();
}

void ChatList::addFormattedLine(const std::string &msg)
{
    ChatPrinter p(formatted_lines, font.get());
    TextFormatter formatter(p, width, false);
    formatter.printString(msg);
    is_updated = true;
}

void ChatList::rmFormattedLine()
{
    if (!formatted_lines.empty()) formatted_lines.pop_front();
    while (!formatted_lines.empty() && !formatted_lines.front().firstline) formatted_lines.pop_front();
    is_updated = true;
}

bool ChatList::isUpdated()
{
    const bool result = is_updated;
    is_updated = false;
    return result;
}

void NameList::add(const std::string &x, const std::string &extra, int ord)
{
    add(std::vector<std::string>(1, x), extra, ord);
}

void NameList::add(const std::vector<std::string> &x, const std::string &extra, int ord)
{
    for (std::vector<std::string>::const_iterator it = x.begin(); it != x.end(); ++it) {
        Name n;
        n.name = *it;
        n.extra = extra;
        n.ord = ord;
        names.push_back(n);
    }
    std::stable_sort(names.begin(), names.end());
}

void NameList::add(const std::vector<std::pair<std::string, std::string> > &x, int ord)
{
    for (std::vector<std::pair<std::string, std::string> >::const_iterator it = x.begin(); it != x.end(); ++it) {
        Name n;
        n.name = it->first;
        n.extra = it->second;
        n.ord = ord;
        names.push_back(n);
    }
    std::stable_sort(names.begin(), names.end());
}

void NameList::alter(const std::string &x, const std::string &extra, int ord)
{
    std::vector<NameList::Name>::iterator it;
    for (it = names.begin(); it != names.end(); ++it) {
        if (it->name == x) break;
    }
    if (it != names.end()) {
        it->extra = extra;
        it->ord = ord;
        std::stable_sort(names.begin(), names.end());
    }
}

void NameList::clearReady()
{
    for (std::vector<NameList::Name>::iterator it = names.begin(); it != names.end(); ++it) {
        if (it->extra == "Ready") it->extra = "";
    }
}

void NameList::clear()
{
    names.clear();
}

void NameList::remove(const std::string &x)
{
    std::vector<NameList::Name>::iterator it;
    for (it = names.begin(); it != names.end(); ++it) {
        if (it->name == x) break;
    }
    if (it != names.end()) {
        names.erase(it);
    }
}

int NameList::getNumberOfElements()
{
    return names.size();
}

std::string NameList::getElementAt(int i)
{
    if (i < 0 || i >= names.size()) return "";
    std::string result = names[i].name;
    if (!names[i].extra.empty()) result += " (" + names[i].extra + ")";
    return result;
}


namespace {
    class MenuListModel : public gcn::ListModel {
    public:
        virtual int getNumberOfElements() { return elts.size(); }
        virtual std::string getElementAt(int i) { if (i >= 0 && i < elts.size()) return elts[i]; else return ""; }
        std::vector<std::string> elts;
        std::vector<int> vals;
    };

    class MenuWidgets {
    public:
        MenuWidgets(const MenuItem &mi, gcn::ActionListener *listener)
            : menu_item(mi)
        {
            label.reset(new gcn::Label(menu_item.getTitleString()));
            label->adjustSize();
            list_model.reset(new MenuListModel);
            dropdown.reset(new gcn::DropDown(list_model.get()));
            dropdown->adjustHeight();
            dropdown->addActionListener(listener);
        }

        const MenuItem &menu_item;
        
        // NOTE: these have to be shared_ptrs since the MenuWidgets object gets copied.
        boost::shared_ptr<gcn::Label> label;
        boost::shared_ptr<gcn::DropDown> dropdown;
        boost::shared_ptr<MenuListModel> list_model;
    };

    int GetDropdownWidth(const gcn::Font &font, const MenuItem &item)
    {
        int widest_text = 0;
        for (int i = item.getMinValue(); i < item.getMinValue() + item.getNumValues(); ++i) {
            const std::string & text = item.getValueString(i);
            const int text_width = font.getWidth(text);
            if (text_width > widest_text) widest_text = text_width;
        }
        return widest_text + 8 + font.getHeight();  // getHeight is to allow for the 'drop down button' which is approximately square.
    }
}

class GameManagerImpl {
public:
    GameManagerImpl(KnightsApp &ka, boost::shared_ptr<KnightsClient> kc, boost::shared_ptr<Coercri::Timer> timer_) 
        : knights_app(ka), knights_client(kc), menu(0), font_black(0), font_grey(0), gui_invalid(false), are_menu_widgets_enabled(true),
          timer(timer_),
          game_list_updated(false), my_player_num(-1), is_split_screen(false), is_lan_game(false),
          chat_list(ka.getConfigMap().getInt("max_chat_lines"))
    { player_ready[0] = player_ready[1] = false; house_colour[0] = house_colour[1] = 0; }
    
    KnightsApp &knights_app;
    boost::shared_ptr<KnightsClient> knights_client;
    boost::shared_ptr<const ClientConfig> client_config;
    
    const Menu *menu;
    MenuSelections menu_selections;
    std::map<std::string, MenuWidgets> menu_widgets_map;
    std::string quest_description;
    gcn::Font *font_black, *font_grey;
    bool gui_invalid;
    bool are_menu_widgets_enabled;

    boost::shared_ptr<Coercri::Timer> timer;

    // lobby info
    std::string server_name;
    NameList all_players_list, observers_list;
    std::vector<GameInfo> game_infos;
    bool game_list_updated;

    // current game info
    std::string current_game_name;   // empty if not connected to a game. set if we have a
                                     // join game request in flight, or are currently in a game.
    std::string player_name[2];     // empty if the seat is open.
    int house_colour[2];
    std::vector<Coercri::Color> avail_house_colours;
    int my_player_num;
    bool player_ready[2];
    bool is_split_screen;
    bool is_lan_game;
    
    // chat
    ChatList chat_list;
};

void GameManager::setServerName(const std::string &sname)
{
    pimpl->server_name = sname;
}

const std::string & GameManager::getServerName() const
{
    return pimpl->server_name;
}

const std::vector<GameInfo> & GameManager::getGameInfos() const
{
    return pimpl->game_infos;
}

bool GameManager::isGameListUpdated()
{
    const bool result = pimpl->game_list_updated;
    pimpl->game_list_updated = false;
    return result;
}

bool GameManager::isGuiInvalid()
{
    const bool r = pimpl->gui_invalid;
    pimpl->gui_invalid = false;
    return r;
}

const std::string & GameManager::getCurrentGameName() const
{
    return pimpl->current_game_name;
}

void GameManager::tryJoinGame(const std::string &game_name)
{
    if (!pimpl->current_game_name.empty()) return;    
    pimpl->knights_client->joinGame(game_name);
    pimpl->current_game_name = game_name;
}

void GameManager::tryJoinGameSplitScreen(const std::string &game_name)
{
    if (!pimpl->current_game_name.empty()) return;    
    pimpl->knights_client->joinGameSplitScreen(game_name);
    pimpl->current_game_name = game_name;
}

void GameManager::createMenuWidgets(gcn::ActionListener *listener,
                                    int initial_x,
                                    int initial_y,
                                    gcn::Container &container,
                                    gcn::Font *font_black_,
                                    gcn::Font *font_grey_,
                                    int &menu_width, int &y_after_menu)
{
    const int pad = 10;  // gap between labels and dropdowns.
    const int vspace = 2;
    const int extra_vspace = 10;
    
    const Menu *menu = pimpl->menu;
    if (!menu) throw UnexpectedError("cannot create menu widgets");

    pimpl->font_black = font_black_;
    pimpl->font_grey = font_grey_;
    
    // create widgets, work out widths
    int max_label_w = 0, max_dropdown_w = 0;
    for (int i = 0; i < menu->getNumItems(); ++i) {
        const MenuItem &item = menu->getItem(i);
        MenuWidgets menu_widgets(item, listener);
        pimpl->menu_widgets_map.insert(std::make_pair(item.getKey(), menu_widgets));
        max_label_w = std::max(max_label_w, menu_widgets.label->getWidth());
        max_dropdown_w = std::max(max_dropdown_w, GetDropdownWidth(*menu_widgets.dropdown->getFont(), item));
    }

    // resize widgets & add to container
    int y = initial_y;
    for (int i = 0; i < menu->getNumItems(); ++i) {
        const MenuItem &item(menu->getItem(i));
        std::map<std::string,MenuWidgets>::const_iterator mw = pimpl->menu_widgets_map.find(item.getKey());
        mw->second.dropdown->setWidth(max_dropdown_w);
        container.add(mw->second.label.get(), initial_x, y+1);
        container.add(mw->second.dropdown.get(), initial_x + max_label_w + pad, y);
        y += std::max(mw->second.label->getHeight(), mw->second.dropdown->getHeight()) + vspace;
        if (item.getSpaceAfter()) y += extra_vspace;
    }

    // make sure the list models are updated
    for (int i = 0; i < menu->getNumItems(); ++i) {
        updateMenuDropdown(menu->getItem(i).getKey());
    }
    
    y_after_menu = y;
    menu_width = max_label_w + pad + max_dropdown_w;
}

void GameManager::destroyMenuWidgets()
{
    pimpl->menu_widgets_map.clear();
    pimpl->font_black = pimpl->font_grey = 0;
}

void GameManager::setMenuWidgetsEnabled(bool enabled)
{
    // note: this routine is called by MenuScreen::updateGui as needed.
    
    if (pimpl->are_menu_widgets_enabled == enabled) return;
    pimpl->are_menu_widgets_enabled = enabled;
    
    for (std::map<std::string, MenuWidgets>::iterator it = pimpl->menu_widgets_map.begin(); it != pimpl->menu_widgets_map.end(); ++it) {
        updateMenuDropdown(it->second.menu_item.getKey());
    }
}

void GameManager::getMenuStrings(std::vector<std::pair<std::string, std::string> > &menu_strings) const
{
    const Menu * menu = pimpl->menu;
    if (!menu) throw UnexpectedError("cannot get menu strings");

    for (int i = 0; i < menu->getNumItems(); ++i) {
        std::pair<std::string, std::string> p;
        const MenuItem &item = menu->getItem(i);
        p.first = item.getTitleString();
        p.second = item.getValueString(pimpl->menu_selections.getValue(item.getKey()));
        
        menu_strings.push_back(p);
        if (item.getSpaceAfter()) {
            menu_strings.push_back(std::make_pair(std::string(), std::string()));
        }
    }
}

bool GameManager::getDropdownInfo(gcn::Widget *source, std::string &out_key, int &out_val) const
{
    const Menu *menu = pimpl->menu;
    if (!menu) return false;

    for (int i = 0; i < menu->getNumItems(); ++i) {
        const MenuItem &item = menu->getItem(i);
        const std::string &key = item.getKey();

        const MenuWidgets &mw = pimpl->menu_widgets_map.find(key)->second;
        const gcn::DropDown *dropdown = mw.dropdown.get();
        if (dropdown == source) {
            // Find out what was selected
            const int selected = dropdown->getSelected();
            if (selected >= 0 && selected < mw.list_model->vals.size()) {
                out_key = key;
                out_val = mw.list_model->vals[selected];
                return true;
            }
        }
    }
    // Source widget was not one of the dropdowns
    return false;
}

void GameManager::updateMenuDropdown(const std::string &key)
{
    std::map<std::string, MenuWidgets>::iterator mw_iter = pimpl->menu_widgets_map.find(key);
    if (mw_iter != pimpl->menu_widgets_map.end()) {
        MenuWidgets &mw = mw_iter->second;
        
        mw.list_model->elts.clear();
        mw.list_model->vals.clear();

        const std::vector<int> &allowed_vals = pimpl->menu_selections.getAllowedValues(key);
        for (std::vector<int>::const_iterator it = allowed_vals.begin(); it != allowed_vals.end(); ++it) {
            mw.list_model->elts.push_back(mw.menu_item.getValueString(*it));
            mw.list_model->vals.push_back(*it);
        }

        const int val_selected = pimpl->menu_selections.getValue(key);
        for (int idx = 0; idx < mw.list_model->vals.size(); ++idx) {
            const int val_at_idx = mw.list_model->vals[idx];
            if (val_selected == val_at_idx) {
                mw.dropdown->setSelected(idx);
                break;
            }
        }

        const bool is_enabled = (allowed_vals.size() != 1 || allowed_vals[0] != 0) && pimpl->are_menu_widgets_enabled;
        mw.dropdown->setFont(is_enabled ? pimpl->font_black : pimpl->font_grey);
        mw.dropdown->setEnabled(is_enabled);
        
        pimpl->gui_invalid = true;
    }
}

const std::string & GameManager::getPlayerName(int i) const
{
    if (i < 0 || i > 1) throw UnexpectedError("Bad player num in GameManager::getPlayerName");
    else return pimpl->player_name[i];
}

bool GameManager::getPlayerReady(int i) const
{
    if (i < 0 || i > 1) throw UnexpectedError("Bad player num in GameManager::getPlayerReady");
    else return pimpl->player_ready[i];
}

int GameManager::getHouseColour(int i) const
{
    if (i < 0 || i > 1) throw UnexpectedError("Bad player num in GameManager::getHouseColour");
    else return pimpl->house_colour[i];
}

Coercri::Color GameManager::getAvailHouseColour(int i) const
{
    if (i < 0 || i >= pimpl->avail_house_colours.size()) return Coercri::Color();
    else return pimpl->avail_house_colours[i];
}

int GameManager::getNumAvailHouseColours() const
{
    return pimpl->avail_house_colours.size();
}

int GameManager::getMyPlayerNum() const
{
    return pimpl->my_player_num;
}

NameList & GameManager::getAllPlayersList() const
{
    return pimpl->all_players_list;
}

NameList & GameManager::getObserversList() const
{
    return pimpl->observers_list;
}

ChatList & GameManager::getChatList() const
{
    return pimpl->chat_list;
}


//
// Callback functions
//

void GameManager::connectionLost()
{
    // Go to ErrorScreen
    std::auto_ptr<Screen> error_screen(new ErrorScreen("The network connection has been lost"));
    pimpl->knights_app.requestScreenChange(error_screen);
}

void GameManager::connectionFailed()
{
    // Go to ErrorScreen
    std::auto_ptr<Screen> error_screen(new ErrorScreen("Connection failed"));
    pimpl->knights_app.requestScreenChange(error_screen);
}

void GameManager::serverError(const std::string &error)
{
    // Go to ErrorScreen
    std::auto_ptr<Screen> error_screen(new ErrorScreen("Error: " + error));
    pimpl->knights_app.requestScreenChange(error_screen);
}

void GameManager::gameListReceived(const std::vector<GameInfo> &games)
{
    pimpl->game_infos = games;
    pimpl->game_list_updated = true;
}

void GameManager::joinGameAccepted(boost::shared_ptr<const ClientConfig> conf,
                                   int my_player_num,
                                   const std::string &player1,
                                   bool ready1,
                                   int house_col_1,
                                   const std::string &player2,
                                   bool ready2,
                                   int house_col_2,
                                   const std::vector<std::string> &observers)
{
    if (my_player_num < -1 || my_player_num > 1) throw UnexpectedError("bad player num in GameManager::joinGameAccepted");

    pimpl->is_split_screen = (pimpl->current_game_name == "#SplitScreenGame");
    pimpl->is_lan_game = (pimpl->current_game_name == "#LanGame");

    // update my player/observer lists
    pimpl->player_name[0] = player1;
    pimpl->player_ready[0] = ready1;
    pimpl->house_colour[0] = house_col_1;
    pimpl->player_name[1] = player2;
    pimpl->player_ready[1] = ready2;
    pimpl->house_colour[1] = house_col_2;

    pimpl->my_player_num = my_player_num;

    pimpl->observers_list.clear();
    for (std::vector<std::string>::const_iterator it = observers.begin(); it != observers.end(); ++it) {
        pimpl->observers_list.add(*it, "", 0);
    }

    pimpl->gui_invalid = true;
    
    // Store a pointer to the config, and the menu
    pimpl->client_config = conf;
    pimpl->menu = conf->menu.get();

    // Load the graphics and sounds
    for (std::vector<const Graphic *>::const_iterator it = conf->graphics.begin(); it != conf->graphics.end(); ++it) {
        pimpl->knights_app.getGfxManager().loadGraphic(**it);
    }
    for (std::vector<const Sound *>::const_iterator it = conf->sounds.begin(); it != conf->sounds.end(); ++it) {
        pimpl->knights_app.getSoundManager().loadSound(**it);
    }
    
    // Clear messages, and add new "Joined game" messages for lan games
    pimpl->chat_list.clear();
    if (pimpl->is_lan_game) {
        if (my_player_num == 0) {
            std::string msg("LAN game created.");
            if (player2.empty()) { // I think this will always be true
                msg += " Waiting for Player 2 to connect.";
            }
            pimpl->chat_list.add(msg);
        }
    }

    // Go to MenuScreen
    auto_ptr<Screen> menu_screen(new MenuScreen(pimpl->knights_client, !pimpl->is_split_screen, !pimpl->is_lan_game));
    pimpl->knights_app.requestScreenChange(menu_screen);
}

void GameManager::joinGameDenied(const std::string &reason)
{
    pimpl->chat_list.add("Could not join game! " + reason);
    pimpl->current_game_name.clear();
    pimpl->gui_invalid = true;
}

namespace {
    struct SecondIsEmpty {
        bool operator()(const std::pair<std::string, std::string> &x) const
        {
            return x.second.empty();
        }
    };
}

void GameManager::initialPlayerList(const std::vector<std::pair<std::string, std::string> > &names)
{
    std::vector<std::pair<std::string, std::string> > new_names;
    copy_if(names.begin(), names.end(), std::back_inserter(new_names), SecondIsEmpty());
    
    pimpl->all_players_list.add(new_names, 0);
    pimpl->gui_invalid = true;

    // For Internet games we want to leave the connecting screen and
    // go to the lobby screen as soon as we get this initial player
    // list (as this means we have successfully connected to the
    // server). For LAN or split screen games we wait until the join
    // game accepted message, and go directly into the menu screen.

    if (!pimpl->is_lan_game && !pimpl->is_split_screen) {
        auto_ptr<Screen> lobby_screen(new LobbyScreen(pimpl->knights_client, pimpl->server_name));
        pimpl->knights_app.requestScreenChange(lobby_screen);
    }
}

void GameManager::passwordRequested(bool first_attempt)
{
    // Go to PasswordScreen
    auto_ptr<Screen> password_screen(new PasswordScreen(pimpl->knights_client, first_attempt));
    pimpl->knights_app.requestScreenChange(password_screen);
}

void GameManager::playerConnected(const std::string &name)
{
    pimpl->all_players_list.add(name, "", 0);

    if (pimpl->current_game_name.empty()) {
        // only show connection/disconnection messages if in main lobby.
        pimpl->chat_list.add(name + " has connected.");
    }

    pimpl->gui_invalid = true;
}

void GameManager::playerDisconnected(const std::string &name)
{
    pimpl->all_players_list.remove(name);

    if (pimpl->current_game_name.empty()) {
        pimpl->chat_list.add(name + " has disconnected.");
    }

    pimpl->gui_invalid = true;
}

void GameManager::playerJoinedGame(const std::string &name, const std::string &game)
{
    pimpl->all_players_list.remove(name);
    pimpl->gui_invalid = true;
}

void GameManager::playerLeftGame(const std::string &name, const std::string &game)
{
    pimpl->all_players_list.add(name, "", 0);
    pimpl->gui_invalid = true;
}

void GameManager::setPlayerNum(const std::string &name, const int new_num, bool is_me)
{
    if (new_num < -1 || new_num > 1) return;
    
    if (is_me) pimpl->my_player_num = new_num;

    // remove player
    pimpl->observers_list.remove(name);
    for (int i = 0; i < 2; ++i) {
        if (pimpl->player_name[i] == name) pimpl->player_name[i] = "";
    }

    // re-add player
    if (new_num != -1) {
        pimpl->player_name[new_num] = name;
    } else {
        pimpl->observers_list.add(name, "", 0);
    }
    
    // do message if needed
    if (is_me) {
        if (new_num == -1) {
            pimpl->chat_list.add("You are now observing this game.");
        } else {
            pimpl->chat_list.add("You have joined the game.");
        }
    } else {
        if (new_num == -1) {
            pimpl->chat_list.add(name + " is now observing this game.");
        } else {
            pimpl->chat_list.add(name + " has joined the game.");
        }
    }

    pimpl->gui_invalid = true;
}

void GameManager::leaveGame()
{
    // clear chat buffer.
    pimpl->chat_list.clear();

    pimpl->current_game_name.clear();
    pimpl->observers_list.clear();
    
    // go to lobby (for internet games) or title (for split screen and lan games)
    if (pimpl->is_split_screen || pimpl->is_lan_game) {
        auto_ptr<Screen> title_screen(new TitleScreen);
        pimpl->knights_app.requestScreenChange(title_screen);
    } else {
        auto_ptr<Screen> lobby_screen(new LobbyScreen(pimpl->knights_client, pimpl->server_name));
        pimpl->knights_app.requestScreenChange(lobby_screen);
    }

    // unload gfx/sounds
    pimpl->knights_app.unloadGraphicsAndSounds();
    
    pimpl->current_game_name.clear();
    pimpl->player_name[0] = pimpl->player_name[1] = "";
    pimpl->my_player_num = -1;

    pimpl->gui_invalid = true;
}

void GameManager::setMenuSelection(const std::string &key, int val, const std::vector<int> &allowed_vals)
{
    // Update menu selections
    pimpl->menu_selections.setAllowedValues(key, allowed_vals);
    pimpl->menu_selections.setValue(key, val);
    
    // Update GUI if necessary
    updateMenuDropdown(key);
}

void GameManager::setQuestDescription(const std::string &quest_descr)
{
    pimpl->quest_description = quest_descr;
    pimpl->gui_invalid = true;
}

const std::string & GameManager::getQuestDescription() const
{
    return pimpl->quest_description;
}

void GameManager::startGame(int ndisplays)
{
    if (!pimpl->client_config) throw UnexpectedError("Cannot start game -- config not loaded");
    
    // Go to InGameScreen
    std::auto_ptr<Screen> in_game_screen(new InGameScreen(pimpl->knights_app, 
                                                          pimpl->knights_client, 
                                                          pimpl->client_config,
                                                          ndisplays));
    pimpl->knights_app.requestScreenChange(in_game_screen);

    // clear chat buffer when game starts
    pimpl->chat_list.clear();

    // clear all ready-flags when game starts
    pimpl->player_ready[0] = pimpl->player_ready[1] = false;

    pimpl->gui_invalid = true;
}

void GameManager::gotoMenu()
{
    std::auto_ptr<Screen> menu_screen(new MenuScreen(pimpl->knights_client, !pimpl->is_split_screen, !pimpl->is_lan_game));
    pimpl->knights_app.requestScreenChange(menu_screen);

    // clear chat buffer after game ends
    pimpl->chat_list.clear();

    pimpl->gui_invalid = true;
}

void GameManager::playerJoinedThisGame(const std::string &name, int pnum, int house_col)
{
    if (pnum < -1 || pnum > 1) throw UnexpectedError("Bad player num in GameManager::playerJoinedThisGame");    
    if (pnum >= 0) {
        pimpl->chat_list.add(name + " has joined the game.");
        pimpl->player_name[pnum] = name;
        pimpl->player_ready[pnum] = false;
        pimpl->house_colour[pnum] = house_col;
    } else {
        pimpl->chat_list.add(name + " is now observing this game.");
        pimpl->observers_list.add(name, "", 0);
    }
    pimpl->gui_invalid = true;
}

void GameManager::playerChangedName(const std::string &name, int pnum)
{
    // this can only happen in "LAN" games -- where there are no observers (yet).
    if (pnum < 0 || pnum > 1) throw UnexpectedError("Bad player num in GameManager::playerChangedName");
    pimpl->player_name[pnum] = name;
    pimpl->gui_invalid = true;
}

void GameManager::setPlayerHouseColour(const std::string &name, int house_col)
{
    for (int i = 0; i < 2; ++i) {
        if (pimpl->player_name[i] == name) {
            pimpl->house_colour[i] = house_col;
        }
    }
    pimpl->gui_invalid = true;
}

void GameManager::setAvailableHouseColours(const std::vector<Coercri::Color> &cols)
{
    pimpl->avail_house_colours = cols;
    pimpl->gui_invalid = true;
}

void GameManager::playerLeftThisGame(const std::string &name)
{
    bool both_seats_filled = true;
    int pnum = -1;
    for (int i = 0; i < 2; ++i) {
        if (pimpl->player_name[i].empty()) both_seats_filled = false;
        if (pimpl->player_name[i] == name) {
            pnum = i;
            pimpl->player_name[i] = "";
            pimpl->player_ready[i] = false;
            pimpl->house_colour[i] = 0;
        }
    }
    std::string msg = name;
    bool do_join_msg = false;
    if (pnum != -1) {
        msg += " has left the game.";
        if (both_seats_filled && pimpl->my_player_num == -1) do_join_msg = true;
    } else {
        msg += " is no longer observing this game.";
        pimpl->observers_list.remove(name);
    }
    pimpl->chat_list.add(msg);
    pimpl->gui_invalid = true;

    if (do_join_msg) {
        pimpl->chat_list.add("You may now join this game - click Join Game (above) if you wish to do so.");
    }
}

void GameManager::setReady(const std::string &name, bool ready)
{
    for (int i = 0; i < 2; ++i) {
        if (pimpl->player_name[i] == name) {
            pimpl->player_ready[i] = ready;
        }
    }
    if (ready) {
        pimpl->chat_list.add(name + " is ready to start.");
    } else {
        pimpl->chat_list.add(name + " is no longer ready to start.");
    }
    pimpl->gui_invalid = true;
}

void GameManager::chat(const std::string &whofrom, bool observer, const std::string &msg)
{
    std::string out = whofrom;
    if (observer) out += " (Observer)";
    out += ": ";
    out += msg;
    pimpl->chat_list.add(out);
    pimpl->gui_invalid = true;
}

void GameManager::announcement(const std::string &msg)
{
    pimpl->chat_list.add(msg);
    pimpl->gui_invalid = true;
}


//
// ctor/dtor
//

GameManager::GameManager(KnightsApp &ka, boost::shared_ptr<KnightsClient> kc, boost::shared_ptr<Coercri::Timer> timer)
    : pimpl(new GameManagerImpl(ka, kc, timer))
{ }
