/*
 * random_int.hpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef KCFG_RANDOM_INT_HPP
#define KCFG_RANDOM_INT_HPP

#include <utility>
#include <vector>

namespace KConfig {

//
// GetRandom function: this is used by the library as a way of
// generating random numbers. It must be defined externally by the
// client. It should return a uniformly distributed integer between 0
// and N-1 (inclusive).
//
int GetRandom(int N);


//
// RandomInt interface
//

class RandomInt {
public:
	virtual ~RandomInt() { }
	virtual int get() const = 0;
};


//
// Subclasses
//

class RIConstant : public RandomInt {
	// Wrapper for a constant (non-random) integer.
public:
	explicit RIConstant(int x_) : x(x_) { }
	void reset(int x_) { x = x_; }
	virtual int get() const { return x; }
private:
	int x;
};

class RIDice : public RandomInt {
	// Dice expression, eg 2d6.
public:
	RIDice(int n_, int d_) : n(n_), d(d_) { }
	virtual int get() const;
private:
	int n, d;
};

class RIList : public RandomInt {
	// A random selection from a list of RandomInts.
public:
	explicit RIList(int nrsrv)
		: total_weight(0) { contents.reserve(nrsrv); }
	void add(const RandomInt *r, int w)
		{ if (r) { contents.push_back(std::make_pair(r,w)); total_weight += w; } }
	virtual int get() const;
	
private:
	std::vector<std::pair<const RandomInt*, int> > contents;
	int total_weight;
};

class RIBinExpr : public RandomInt {
	// Base class for binary operators (+,-,*,/).
	// (Unary minus can be implemented via RISub(0,x).)
public:
	RIBinExpr(const RandomInt *l, const RandomInt *r) : lhs(l), rhs(r) { }
	int getLhs() const { return lhs? lhs->get() : 0; }
	int getRhs() const { return rhs? rhs->get() : 0; }
private:
	const RandomInt *lhs;
	const RandomInt *rhs;
};

class RIAdd : public RIBinExpr {
public:
	RIAdd(const RandomInt *lhs, const RandomInt *rhs) : RIBinExpr(lhs, rhs) { }
	virtual int get() const { return getLhs() + getRhs(); }
};

class RISub : public RIBinExpr {
public:
	RISub(const RandomInt *lhs, const RandomInt *rhs) : RIBinExpr(lhs, rhs) { }
	virtual int get() const { return getLhs() - getRhs(); }
};

class RIMul : public RIBinExpr {
public:
	RIMul(const RandomInt *lhs, const RandomInt *rhs) : RIBinExpr(lhs, rhs) { }
	virtual int get() const { return getLhs() * getRhs(); }
};

class RIDiv : public RIBinExpr {
public:
	RIDiv(const RandomInt *lhs, const RandomInt *rhs) : RIBinExpr(lhs, rhs) { }
	virtual int get() const { return getLhs() / getRhs(); }
};

}

#endif
