/*
 * kconfig_loader.hpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef KCONFIG_LOADER_HPP
#define KCONFIG_LOADER_HPP

#include "gold_parser.hpp"

#include "boost/shared_ptr.hpp"

#include <exception>
#include <iosfwd>
#include <map>
#include <set>
#include <stack>
#include <string>
#include <vector>
using namespace std;

namespace KConfig {

class RandomInt;
class Value;

class KConfigError : public std::exception {
public:
    KConfigError(const string &msg, const Value *from_where = 0);
    virtual ~KConfigError() throw() { }
    void addTraceback(const Value &from_where);
    const char * what() const throw() { return error_msg.c_str(); }
private:
    string error_msg;
};

// RandomIntContainer holds all of the randomints generated during
// the config loading process. They will all be released when the
// RandomIntContainer is destroyed.
class RandomIntContainer {
public:
    RandomIntContainer() { }
    ~RandomIntContainer();  // deletes all contained RandomInts
    void add(const RandomInt *ri); // transfers ownership of "ri"
private:
    RandomIntContainer(const RandomIntContainer &);//not defined
    void operator=(const RandomIntContainer &);//not defined
    vector<const RandomInt*> contents; 
};

class Value {
public:
    Value(const string &file_, int line_) : file(file_), line(line_) { }
    virtual ~Value() { }
    
    virtual bool isInt() const { return false; }
    virtual bool isFloat() const { return false; }
    virtual bool isString() const { return false; }
    virtual bool isRandomInt() const { return false; }
    virtual bool isTable() const { return false; }
    virtual bool isList() const { return false; }
    virtual bool isDirective() const { return false; }
    
    virtual int getInt() const;
    virtual float getFloat() const;
    virtual string getString() const;
    virtual const RandomInt * getRandomInt(RandomIntContainer &c) const;
    virtual const Value & getTable(const string &key) const;
    virtual const Value * getTableOptional(const string &key) const;
    virtual string getTableNext(const string &prev_key) const; // returns next key alphabetically after prev_key (or "" if prev_key was the last key). 
    virtual int getListSize() const;
    virtual const Value & getList(int index) const;
    virtual int getListSizeP() const;
    virtual pair<const Value *, int> getListP(int index) const; // includes repeat count. The value will never be NULL
    virtual pair<string, const Value *> getDirective() const;  // returns dir-name and list-of-args. The value will never be NULL

    virtual string describe() const = 0;
    virtual const Value & getTarget() const { return *this; } // Only defined for "symbols"
    virtual const Value * getTargetOptional() const { return this; }  // Ditto
    
    const string &getFilename() const { return file; }
    int getLineNumber() const { return line; }

private:
    
    string file;
    int line;
};

// TableValue has to be declared here (as it's used below in KConfig)
// However, it is not strictly part of the public interface.
class TableValue : public Value {
public:
    TableValue(const string &file, int line) : Value(file, line) { }
    void add(const string &key, const Value *val) {
        if (val) tbl[key] = val;
    }
    virtual bool isTable() const { return true; }
    virtual const Value & getTable(const string &key) const; 
    virtual const Value * getTableOptional(const string &key) const;
    virtual string getTableNext(const string &prev_key) const;
    virtual string describe() const { return "<table>"; }
private:
    typedef map<string, const Value *> TableType;
    TableType tbl;
};

// Interface for KConfig
class KConfigSource {
public:
    virtual ~KConfigSource() { }
    virtual boost::shared_ptr<istream> openFile(const string &name) = 0;
};

// KConfigLoader -- loads the config file(s).
// A tree of Value objects is constructed (representing the AST
// of the config file) and these can be accessed through getRootTable().
// The Values are held in memory for the lifetime of the KConfig object.
// The RandomInts generated will be stored in the RandomIntContainer, hence
// these can live for longer than the KConfig if necessary. 
class KConfigLoader {
public:
    KConfigLoader(const string &filename, KConfigSource &file_loader, RandomIntContainer &ric);
    ~KConfigLoader();
    const Value & getRootTable() const { return root_table; }
    const Value & get(const string &s) const { return root_table.getTable(s); }
    const Value * getOptional(const string &s) const { return root_table.getTableOptional(s); }
    
private:
    KConfigLoader(const KConfigLoader &); // not defined
    void operator=(const KConfigLoader &) const;// not defined

private:
    void shift(int symbol_code, const string &token, RandomIntContainer &);
    void reduce(int reduction, RandomIntContainer &);
    void processFile(istream &, RandomIntContainer &);
    
private:
    set<string> loaded_files;
    set<string> waiting_files;
    stack<Value *> value_stack;
    vector<Value *> stored_values;
    TableValue root_table;
    GoldParser::GoldParser gp;
    string file; // file currently being processed.
};

}

#endif
