/*
 * gold_parser.hpp
 *
 * A simple implementation of the GOLD Parsing Engine in C++.
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef GOLD_PARSER_HPP
#define GOLD_PARSER_HPP

#include "boost/shared_ptr.hpp"
#include <exception>
#include <iosfwd>
#include <string>

namespace GoldParser {

	struct GPImpl; // private
	class DFAState; // private
	class LALRState; // private
	
	// ParseMessage:
	// These codes are returned by the GoldParser::parse function
	enum ParseMessage {
		REDUCTION,  // A reduction rule was applied ("Parse stack" should be updated appropriately)
		SHIFT,      // A shift was done (A token should now be added to the "parse stack")
		ACCEPT,     // The end of the input was reached
		LEXICAL_ERROR,  // A lexical error was encountered
		SYNTAX_ERROR,   // A syntax error was encountered
		COMMENT_ERROR,  // Unmatched start/end comment tokens
		INTERNAL_ERROR  // This should not happen :-)
	};

	// BadCGT: This exception gets thrown if the CGT could not be loaded correctly.
	class BadCGT : public std::exception {
	public:
		const char * what() const throw() { return "GOLD parser: bad CGT file"; }
	};

	// The main GoldParser class:
	class GoldParser {
	public:
		// Ctor takes a memory buffer containing the CGT file.
		// p is the first byte, q is the one-past-the-end byte.
		// Throws BadCGT if anything goes wrong.
		GoldParser(const char *p, const char *q);
		
		// Initialize for a new file. This must be called before parsing begins.
		// If called after parsing is underway, then it will reset everything and
		// restart at the beginning of the new file.
		void setup(std::istream &str);
		
		// Parse function.
		// This will return one of the ParseMessage codes from above. Further info
		// can be obtained by using the "get" functions below. 
		ParseMessage parse();
		
		// Use this function to see which rule was used in a REDUCTION:
		int getReduction() const;
		
		// Use these functions to see which token was read (after a SHIFT result):
		int getTokenCode() const;
		const std::string & getTokenData() const;

		// This returns the current line number (useful for error messages):
		int getLineNumber() const;
		
	private:
		// Implementation
		void readRecord(const char *&p, const char *q,
						const DFAState *&init_dfa, const LALRState *&init_lalr);		
		boost::shared_ptr<GPImpl> pimpl;
	};

}

#endif
