/*
 * knights_config.hpp
 *
 * Class representing the setup and configuration of a single game of
 * Knights.
 * 
 * This class knows how to generate the dungeon and perform other
 * initialization tasks. This is used by KnightsEngine.
 *
 * NOTE: Have to be a little careful with thread safety as
 * KnightsConfigs can be shared between multiple KnightsEngines. (In
 * particular note that things like the contained Graphics and Anims
 * are referred to throughout the whole game, not just during
 * initialization!) However we only require concurrent *read* access,
 * as KnightsEngine does not write to the config at all. Therefore, we
 * should be OK, as long as (a) we are careful not to use any
 * "mutable" variables, and (b) the standard library (and in
 * particular STL containers) are thread safe for multiple concurrent
 * reads to the same container.
 * 
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef KNIGHTS_CONFIG_HPP
#define KNIGHTS_CONFIG_HPP

#include "gfx/color.hpp"

#include "boost/shared_ptr.hpp"
#include <vector>

class Anim;
class ConfigMap;
class DungeonGenerator;
class DungeonMap;
class EventManager;
class GoreManager;
class Graphic;
class HomeManager;
class ItemType;
class KnightsConfigImpl;
class Menu;
class MenuConstraints;
class MenuSelections;
class MonsterManager;
class Overlay;
class Player;
class Quest;
class Sound;
class StuffManager;
class TaskManager;
class UserControl;

// KnightsConfig itself
class KnightsConfig {
public:

    //
    // Interface used by client code
    //

    // Constructor.
    // NOTE: The KnightsConfig should live at least as long as the
    // KnightsGame, as it owns certain objects (tiles, graphics etc)
    // that are used during the game.
    explicit KnightsConfig(const std::string &config_filename);

    // Get lists of all anims, graphics etc that will be used in the
    // game. This is used in network games to send lists of these
    // objects to the client computers, so that they can load all
    // necessary files before the game begins.
    void getAnims(std::vector<const Anim*> &anims) const;
    void getGraphics(std::vector<const Graphic*> &graphics) const;
    void getOverlays(std::vector<const Overlay*> &overlays) const;
    void getSounds(std::vector<const Sound*> &sounds) const;
    void getStandardControls(std::vector<const UserControl*> &standard_controls) const;
    void getOtherControls(std::vector<const UserControl*> &other_controls) const;
    const Menu & getMenu() const;
    const MenuConstraints & getMenuConstraints() const;
    int getApproachOffset() const;  // clients will need to know this for rendering purposes.
    void getHouseColours(std::vector<Coercri::Color> &cols) const;   // for use on the menu screen.
    std::string getQuestDescription(int quest_num) const;
    
    
    //
    // Interface used by KnightsEngine
    //

    // NOTE: this creates dependencies on to quite a lot of other
    // classes, it might be nice if we could reduce this somehow.
    void initializeGame(const MenuSelections &msel,
                        boost::shared_ptr<DungeonMap> &dungeon_map,
                        std::vector<boost::shared_ptr<Quest> > &quests,
                        HomeManager &home_manager,
                        std::vector<boost::shared_ptr<Player> > &players,
                        StuffManager &stuff_manager,
                        GoreManager &gore_manager,
                        MonsterManager &monster_manager,
                        EventManager &event_manager,
                        bool &premapped,
                        std::vector<std::pair<const ItemType *, std::vector<int> > > &starting_gears,
                        TaskManager &task_manager,
                        int house_col_0,
                        int house_col_1) const;
    boost::shared_ptr<const ConfigMap> getConfigMap() const;

private:
    boost::shared_ptr<KnightsConfigImpl> pimpl;
};

#endif
