/*
 * tile.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "action.hpp"
#include "creature.hpp"
#include "dungeon_map.hpp"
#include "item.hpp"
#include "item_type.hpp"
#include "mediator.hpp"
#include "sweep.hpp"
#include "tile.hpp"

//
// public functions
//

Tile::Tile()
    : graphic(0), depth(0), is_stair(false), items_mode(BLOCKED), 
      stair_direction(D_NORTH), hit_points(0), initial_hit_points(0), on_activate(0), 
      on_close(0), on_open(0), on_walk_over(0), on_approach(0), on_withdraw(0), on_hit(0),
      control(0)
{
    for (int i=0; i<=H_MISSILES; ++i) access[i] = A_CLEAR;
}

void Tile::construct(const Graphic *g, int d,
                     bool ita, bool dstry, int item_category_, 
                     const MapAccess acc[],
                     bool is, bool stop, MapDirection sdir, const RandomInt * ihp,
                     const Action *dto,
                     const Action *ac, const Action *c, const Action *o,
                     const Action *wo, const Action *ap, const Action *wi, const Action *hit,
                     const Control *ctrl)
{
    graphic = g; depth=d; item_category = item_category_;
    is_stair=is; stair_top=stop; stair_direction=sdir; 
    initial_hit_points=ihp; on_destroy = dto;
    on_activate = ac; on_close = c; on_open = o;
    on_walk_over = wo; on_approach = ap; on_withdraw = wi;
    on_hit = hit; control = ctrl;
    for (int i=0; i<=H_MISSILES; ++i) { access[i] = acc[i]; }

    items_mode = BLOCKED;
    if (ita) items_mode = ALLOWED;
    if (dstry) items_mode = DESTROY;
}

void Tile::construct(const Graphic *g, int dpth)
{
    graphic = g; depth=dpth;
    items_mode = ALLOWED;
    // the rest take default values as set in the ctor.
}

void Tile::construct(const Action *walkover, const Action *activate)
{
    on_walk_over = walkover;
    on_activate = activate;
    items_mode = ALLOWED;
}


shared_ptr<Tile> Tile::clone(bool force_copy)
{
    shared_ptr<Tile> new_tile = doClone(force_copy);
    new_tile->setHitPoints();
    return new_tile;
}

shared_ptr<Tile> Tile::cloneWithNewGraphic(const Graphic *new_graphic)
{
    shared_ptr<Tile> new_tile = clone(true);  // we must force a copy because this is the only way we can sensibly change the graphic.
    new_tile->graphic = new_graphic;
    return new_tile;
}

void Tile::setHitPoints()
{
    if (initial_hit_points) {
        hit_points = initial_hit_points->get();
    }
}

void Tile::damage(DungeonMap &dmap, const MapCoord &mc, int amount, shared_ptr<Creature> actor)
{
    // NB: hit_points <= 0 means that the tile is indestructible.
    if (hit_points > 0) {
        hit_points -= amount;
        if (hit_points <= 0) {
            // Tile has been destroyed
            shared_ptr<Tile> self(shared_from_this());
            dmap.rmTile(mc, self);
            self->onDestroy(dmap, mc, actor);
        }
    }
}

void Tile::onDestroy(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> actor)
{
    // Run on_destroy event
    if (on_destroy) {
        ActionData ad;
        ad.setActor(actor, false);
        ad.setTile(&dmap, mc, shared_ptr<Tile>());
        on_destroy->execute(ad);
    }
    
    // Destroy any fragile items present
    shared_ptr<Item> item = dmap.getItem(mc);
    if (item && item->getType().isFragile()) dmap.rmItem(mc);
}

bool Tile::targettable() const
{
    // Current definition of targettable is that it is not A_CLEAR at H_WALKING,
    // *OR* that it has an on_hit action.
    // This rules out "floor" tiles (except the special pentagram which has on_hit),
    // but pretty much everything else is included.
    return (getAccess(H_WALKING) != A_CLEAR || on_hit);
}

void Tile::onActivate(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> actor,
                      ActivateType act_type, bool success)
{
    // NB act_type isn't used here, but it is used in subclasses (in particular Lockable).
    ActionData ad;
    ad.setActor(actor, false);
    ad.setTile(&dmap, mc, shared_from_this());
    ad.setSuccess(success);
    if (on_activate) {
        on_activate->execute(ad);
    } else if (on_close && !on_open) {
        // As a special rule, if on_activate is NULL then it will be defaulted
        // to either on_open or on_close (if one but not both of these is given).
        on_close->execute(ad);
    } else if (on_open) {
        on_open->execute(ad);
    }
}

void Tile::onClose(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> actor)
{
    if (on_close) {
        ActionData ad;
        ad.setActor(actor, false);
        ad.setTile(&dmap, mc, shared_from_this());
        on_close->execute(ad);
    }
}

void Tile::onOpen(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> actor)
{
    if (on_open) {
        ActionData ad;
        ad.setActor(actor, false);
        ad.setTile(&dmap, mc, shared_from_this());
        on_open->execute(ad);
    }
}

void Tile::onWalkOver(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> actor)
{
    if (on_walk_over && actor && actor->getHeight() == H_WALKING) {
        ActionData ad;
        ad.setActor(actor, false);
        ad.setTile(&dmap, mc, shared_from_this());
        on_walk_over->execute(ad);
    }
}

void Tile::onApproach(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> actor)
{
    if (on_approach) {
        ActionData ad;
        ad.setActor(actor, false);
        ad.setTile(&dmap, mc, shared_from_this());
        on_approach->execute(ad);
    }
}

void Tile::onWithdraw(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> actor)
{
    if (on_withdraw) {
        ActionData ad;
        ad.setActor(actor, false);
        ad.setTile(&dmap, mc, shared_from_this());
        on_withdraw->execute(ad);
    }
}

void Tile::onHit(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> actor)
{
    if (on_hit) {
        ActionData ad;
        ad.setActor(actor, false);
        ad.setTile(&dmap, mc, shared_from_this());
        on_hit->execute(ad);
    }
}



//
// protected functions
//

shared_ptr<Tile> Tile::doClone(bool force_copy)
{
    ASSERT(typeid(*this)==typeid(Tile)); //doClone must be overridden in subclasses.

    // basic tiles are shared rather than cloned
    // however, if a tile has hit points, it must be copied. (and hit points must be rerolled, also.)
    if (initial_hit_points || force_copy) {
        return shared_ptr<Tile>(new Tile(*this));
    } else {
        return shared_from_this();
    }
}

void Tile::setGraphic(DungeonMap *dmap, const MapCoord &mc, const Graphic *g,
                      shared_ptr<const ColourChange> ccnew)
{
    graphic = g;
    cc = ccnew;
    if (dmap) {
        // Graphic-change needs to be propagated to clients
        Mediator::instance().onChangeTile(*dmap, mc, *this);
    }
}

void Tile::setItemsAllowed(DungeonMap *dmap, const MapCoord &mc, bool allow, bool destroy) 
{
    if (destroy) items_mode = DESTROY;
    else if (allow) items_mode = ALLOWED;
    else items_mode = BLOCKED;
    if (dmap && items_mode != ALLOWED) {
        // We don't need to tell Mediator about items-allowed changes.
        // However we do need to call SweepItems if items-allowed is now BLOCKED, or destroy
        // item if mode is DESTROY.
        if (items_mode == DESTROY) {
            dmap->rmItem(mc);
        } else {
            SweepItems(*dmap, mc);
        }
    }
}

void Tile::setAccess(DungeonMap *dmap, const MapCoord &mc, MapHeight height, MapAccess acc)
{
    access[height] = acc;
    if (dmap) {
        // We don't need to tell Mediator about access changes, but we do need 
        // to call SweepCreatures.
        SweepCreatures(*dmap, mc, true, height);
    }
}

void Tile::setAccess(DungeonMap *dmap, const MapCoord &mc, MapAccess acc)
{
    for (int i=0; i<=H_MISSILES; ++i) {
        access[i] = acc;
    }
    SweepCreatures(*dmap, mc, false, H_MISSILES);
}


MiniMapColour Tile::getColour() const
{
    // Current method of determining mini map colour is as follows: colour is
    // COL_WALL if access level is A_BLOCKED at all available heights; else colour is
    // COL_FLOOR.
    // This can be overridden by subclasses if required.
    for (int h=0; h<=H_MISSILES; ++h) {
        if (getAccess(MapHeight(h)) != A_BLOCKED) {
            return COL_FLOOR;
        }
    }
    return COL_WALL;
}
