/*
 * teleport.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "dungeon_map.hpp"
#include "knight.hpp"
#include "mediator.hpp"
#include "rng.hpp"
#include "player.hpp"
#include "room_map.hpp"
#include "teleport.hpp"

namespace {
    void DoTeleportToSquare(shared_ptr<Entity> from, DungeonMap &dmap, const MapCoord &mc,
                            MapDirection new_facing)
    {
        if (!from) return;
        from->rmFromMap();
        from->setFacing(new_facing);
        from->addToMap(&dmap, mc);
    }

    bool TrySquare(shared_ptr<Entity> from, DungeonMap &dmap, const MapCoord &mc)
    {
        if (dmap.getAccess(mc, H_WALKING) == A_CLEAR) {
            DoTeleportToSquare(from, dmap, mc, from->getFacing());
            return true;
        }
        return false;
    }
}

bool TeleportToSquare(shared_ptr<Entity> from, DungeonMap &dmap, const MapCoord &mc)
{
    if (!from) return false;
    if (from->getMap() != &dmap) return false;  // just checking
    const MapDirection facing = from->getFacing();
        
    // Try to find either this square or one of the four surrounding squares (for now).
    return (TrySquare(from, dmap, mc)
            || TrySquare(from, dmap, DisplaceCoord(mc, facing))
            || TrySquare(from, dmap, DisplaceCoord(mc, Opposite(facing)))
            || TrySquare(from, dmap, DisplaceCoord(mc, Clockwise(facing)))
            || TrySquare(from, dmap, DisplaceCoord(mc, Anticlockwise(facing))));
}

void TeleportToRoom(shared_ptr<Entity> from, shared_ptr<Entity> to)
{
    if (!from || !to) return;
    DungeonMap *dmap = from->getMap();
    if (!dmap || to->getMap() != dmap) return;

    const RoomMap *rmap = dmap->getRoomMap();
    int r1, r2;
    rmap->getRoomAtPos(to->getPos(), r1, r2);
    if (r2 != -1) {
        // Two rooms were returned. We just choose one at random.
        if (g_rng.getBool()) r1 = r2;
        r2 = -1;
    }

    // r1 now contains the room we want to teleport into.
    // Find the room dimensions
    MapCoord top_left;
    int width, height;
    rmap->getRoomLocation(r1, top_left, width, height);
        
    // We must now find an unoccupied space within the room.
    MapDirection new_facing = MapDirection(g_rng.getInt(0,4));
    MapCoord new_mc;
    for (int i=0; i<100; ++i) {
        // (try 100 times to find a suitable square)
        const int x = g_rng.getInt(0, width);
        const int y = g_rng.getInt(0, height);
        MapCoord mc2 = MapCoord(top_left.getX() + x,
                                top_left.getY() + y);
        if (dmap->getAccess(mc2, H_WALKING) == A_CLEAR) {
            new_mc = mc2;
            break;
        }
    }
    if (new_mc.isNull()) return;  // failed to find an empty square.

    // Now we just have to move "from" to the new square.
    DoTeleportToSquare(from, *dmap, new_mc, new_facing);
}

shared_ptr<Knight> FindNearestOtherKnight(const DungeonMap &dmap, const MapCoord &mypos)
{
    // Search through all players, looking for the closest.
    const vector<Player*> & players(Mediator::instance().getPlayers());
    int dist = 99999;
    shared_ptr<Knight> target;
    for (vector<Player*>::const_iterator it = players.begin(); it != players.end(); ++it) {
        shared_ptr<Knight> kt = (*it)->getKnight();
        if (!kt) continue;
        if (kt->getMap() != &dmap) continue;
        const MapCoord kpos = kt->getPos();
                
        // Measure distance by Manhattan distance:
        const int d = abs(kpos.getX() - mypos.getX()) + abs(kpos.getY() - mypos.getY());

        if ((d > 0 && d < dist) || (d == dist && g_rng.getBool())) {
            // Closer target found
            // (or if equal distance, then choose target randomly)
            dist = d;
            target = kt;
        }
    }

    return target;
}
