/*
 * stuff_bag.hpp
 *
 * Bags of stuff that get dropped when a knight dies
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef STUFF_BAG_HPP
#define STUFF_BAG_HPP

#include "action.hpp"
#include "item.hpp"

#include "boost/shared_ptr.hpp"
using namespace boost;

#include <map>
#include <vector>
using namespace std;

class DungeonMap;
class Knight;
class MapCoord;


// StuffContents: a thin wrapper over a vector of Items, used to hold
// the contents of a stuff bag.
class StuffContents {
public:
    // Get/Set the list of items
    void addItem(shared_ptr<Item> i);
    const vector<shared_ptr<Item> > & getItems() const { return contents; }

    // remove items from bag and give to kt (as far as poss); also run pickup events:
    void giveToKnight(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Knight> kt);

    // run drop events for the contents of the stuff bag:
    void runDropEvents(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Knight> kt) const;

    // put the contents into the "displaced items" list, also running drop events:
    void putIntoDisplacedItems(DungeonMap &dmap, shared_ptr<Knight> kt);

    // If stuff bag contains only one item, then the following returns that item, else
    // it returns null.
    shared_ptr<Item> getSingleItem() const;

    bool isEmpty() const { return contents.empty(); }
        
private:
    vector<shared_ptr<Item> > contents;
};


// Action classes for StuffManager:

class StuffPickup : public Action {
public:
    virtual void execute(const ActionData &) const;
};

class StuffDrop : public Action {
public:
    virtual void execute(const ActionData &) const;
};


// StuffManager:
// 
// Because ItemTypes are stateless, we can't directly store the
// contents of a stuff bag within the stuff bag's Item or ItemType
// objects. (The system just isn't set up for that -- it assumes all
// items of a given type are identical.)
//
// Therefore we set up a StuffManager class which maps MapCoords to
// StuffContents objects.

class StuffManager {
public:
    // initialization
    void setStuffBagGraphic(const Graphic *gfx);
    
    // Get the ItemType for a stuff bag: (this is unique)
    const ItemType & getStuffBagItemType() const { return stuff_bag_item_type; }
    
    // Assign a StuffContents to a given MapCoord
    // (it's assumed that a stuff bag item will be put down at that position)
    // This is used by Knight::dropAllItems. 
    void setStuffContents(const MapCoord &mc, const StuffContents &stuff);

    // Get the Items contained within a given stuff bag.
    // (returns NULL if no stuff bag could be found.)
    const vector<shared_ptr<Item> > * getItems(const MapCoord &mc) const;
    
    // Pickup and Drop functions for stuff bags. (Only defined for Knights, not general
    // Creatures)

    // doPickUp does two things:
    // (i) runs the onPickUp event for each item type in the stuff bag
    // (ii) adds the items to the knight's inventory.
    // If any items weren't picked up, then a stuff bag containing the remainder will be
    // put back down into the map.
    void doPickUp(DungeonMap &, const MapCoord &, shared_ptr<Knight> actor);

    // doDrop runs the onDrop event for each item type in the stuff bag.
    void doDrop(DungeonMap &, const MapCoord &, shared_ptr<Knight> actor);

    // This is used for counting how many lockpicks exist in stuff bags
    int getNumBagsContaining(const ItemType &itype) const;    

    
private:
    map<MapCoord, StuffContents> stuff_map;
    ItemType stuff_bag_item_type;
    static StuffPickup pickup_action;
    static StuffDrop drop_action;
};

#endif
