/*
 * segment.hpp
 *
 * "Segment" represents a rectangular dungeon section, used by the dungeon
 * generator.
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef SEGMENT_HPP
#define SEGMENT_HPP

#include "map_support.hpp"

#include "boost/shared_ptr.hpp"
using namespace boost;

#include <list>
#include <vector>
using namespace std;

class DungeonMap;
class ItemType;
class Tile;

struct HomeInfo {
    int x;
    int y; 
    MapDirection facing; // points inwards, towards the home
};


class Segment {
public:
    Segment(int w, int h) : data(w*h), width(w), height(h) { }

    // get width and height
    int getWidth() const { return width; }
    int getHeight() const { return height; }

    // add tile or item (can be called more than once per square)
    // x,y range from 0 to w-1, h-1, and increase rightwards and upwards as usual.
    void addTile(int x, int y, shared_ptr<Tile> t);
    void addItem(int x, int y, const ItemType * i);
        
    // set a room (coords are relative to bottom left of segment -- note that the
    // outer boundary wall is not counted as part of the segment).
    void addRoom(int blx, int bly, int w, int h);

    // get/set "bat placement tile" (used by the dungeon generator)
    void setBatPlacementTile(shared_ptr<Tile> t) { bat_placement_tile = t; }
    shared_ptr<Tile> getBatPlacementTile() const { return bat_placement_tile; }
        
    // copy the tiles and items into a dungeon map
    // The tiles replace any that were already there.
    // This routine also adds appropriate rooms to the RoomMap.
    void copyToMap(DungeonMap &dmap, const MapCoord &bottom_left) const;

    // access to homes
    const vector<HomeInfo> & getHomes() const { return homes; }

private:
    // (TODO) might be better for segments to store tile data in a standard pointer -- would
    // halve memory requirements. (ServerConfig could look after the pointers.) However there
    // might be complications with switch tiles?
    vector<list<shared_ptr<Tile> > > data;
    vector<HomeInfo> homes;
    int width;
    int height;
    struct RoomInfo {
        int tlx, tly, w, h;
    };
    vector<RoomInfo> rooms;
    struct ItemInfo {
        int x;
        int y;
        const ItemType * itype;
    };
    vector<ItemInfo> items;
    shared_ptr<Tile> bat_placement_tile;
};

#endif
