/*
 * segment.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "dungeon_map.hpp"
#include "item.hpp"
#include "room_map.hpp"
#include "segment.hpp"
#include "special_tiles.hpp"  // for Home

void Segment::addTile(int x, int y, shared_ptr<Tile> t)
{
    if (!t) return;
    if (x < 0 || x >= width || y < 0 || y >= height) return;
    data[y*width+x].push_front( t );

    shared_ptr<Home> h = dynamic_pointer_cast<Home>(t);
    if (h) {
        HomeInfo hi;
        hi.x = x;
        hi.y = y;
        hi.facing = h->getFacing();
        homes.push_back(hi);
    }
}

void Segment::addItem(int x, int y, const ItemType *itype)
{
    if (!itype) return;
    if (x < 0 || x >= width || y < 0 || y >= height) return;
    ItemInfo ii;
    ii.x = x;
    ii.y = y;
    ii.itype = itype;
    items.push_back(ii);
}

void Segment::addRoom(int tlx, int tly, int w, int h)
{
    RoomInfo ri = { tlx, tly, w, h };
    rooms.push_back(ri);
}

void Segment::copyToMap(DungeonMap &dmap, const MapCoord &top_left) const
{
    for (int i=0; i<width; ++i) {
        for (int j=0; j<height; ++j) {
            MapCoord mc(i+top_left.getX(), j+top_left.getY());
            dmap.clearTiles(mc);
            const list<shared_ptr<Tile> > &tiles( data[j*width+i] );
            for (list<shared_ptr<Tile> >::const_iterator it = tiles.begin();
                 it != tiles.end(); ++it) {
                shared_ptr<Tile> tile((*it)->clone(false));
                dmap.addTile(mc, tile);
            }
        }
    }

    for (vector<ItemInfo>::const_iterator it = items.begin(); it != items.end(); ++it) {
        shared_ptr<Item> item(new Item(*it->itype));
        MapCoord mc(it->x + top_left.getX(), it->y + top_left.getY());
        dmap.addItem(mc, item);
    }
        
    if (dmap.getRoomMap()) {
        for (vector<RoomInfo>::const_iterator it = rooms.begin(); it != rooms.end(); ++it) {
            dmap.getRoomMap()->addRoom(MapCoord(it->tlx + top_left.getX(),
                                                it->tly + top_left.getY()),
                                       it->w, it->h);
        }
    }
}
