/*
 * player_task.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "creature.hpp"
#include "dungeon_map.hpp"
#include "dungeon_view.hpp"
#include "knight.hpp"
#include "mediator.hpp"
#include "mini_map.hpp"
#include "player.hpp"
#include "player_task.hpp"
#include "room_map.hpp"
#include "task_manager.hpp"

void PlayerTask::execute(TaskManager &tm)
{
    // At the moment, all we do here is update the player's mini-map as necessary.

    // NB There might be a cleaner way of doing this. Eg we could trigger the check
    // if a creature is found to move within a player's current room. (Downside would be
    // that if there are a large number of creatures moving about, then there may be a
    // large number of redundant checks being done.)

    // Map current room if necessary:
    MiniMap &mini_map(pl.getMiniMap());
    shared_ptr<Knight> kt = pl.getKnight();
    const int rm = pl.getCurrentRoom();
    MapCoord top_left;     // top left of current room
    int w, h;              // width, height of current room
    if (rm != -1 && kt) {
        const DungeonMap *dmap = pl.getDungeonMap();
        const RoomMap *room_map = pl.getRoomMap();
        if (dmap && room_map) {
            room_map->getRoomLocation(rm, top_left, w, h);

            // Check that there are no creatures within the room
            // (TODO) This could be made faster, write a dungeonmap routine to find all
            // entities within a room. Try a cpu profiler first though.
            bool found_creature = false;
            vector<shared_ptr<Entity> > entities;
            for (int i=0; i<w; ++i) {
                for (int j=0; j<h; ++j) {
                    MapCoord mc(top_left.getX() + i, top_left.getY() + j);
                    dmap->getEntities(mc, entities);
                    for (vector<shared_ptr<Entity> >::iterator it = entities.begin();
                    it != entities.end(); ++it) {
                        if (dynamic_cast<Creature*>(it->get()) && (*it) != kt) {
                            found_creature = true;
                            break;
                        }
                    }
                    if (found_creature) break;
                }
                if (found_creature) break;
            }

            // If no creature was found, map the current room. Otherwise, clear
            // "top_left" so that the code below does not draw knights in the current
            // room.
            if (found_creature) {
                top_left = MapCoord();
            } else {
                pl.mapCurrentRoom(top_left);
            }
        }
    }

    // Look to see if we can see other knights:

    Mediator &med(Mediator::instance());

    if (kt && kt->getMap()) {
        bool have_sense_knight = kt->getSenseKnight() || kt->getCrystalBall();
        for (int i=0; i<med.getPlayers().size(); ++i) {
            const Player &p = *(med.getPlayers()[i]);
            if (&p == &(this->pl)) continue;  // don't do myself!
            shared_ptr<Knight> other_kt = p.getKnight();
            // Can see knight if I have SenseKnight or CrystalBall, or he has
            // RevealLocation
            bool can_see = false;
            if (other_kt) {
                can_see = have_sense_knight || other_kt->getRevealLocation();
                if (!can_see) {
                    // Can also see knight if he is in the same room as me.
                    if (!top_left.isNull()) {
                        const MapCoord &other_pos(other_kt->getPos());
                        can_see = (other_pos.getX() >= top_left.getX()
                                   && other_pos.getY() >= top_left.getY()
                                   && other_pos.getX() < top_left.getX() + w
                                   && other_pos.getY() < top_left.getY() + h);
                    }
                }
            }
            if (can_see && other_kt->getMap()==kt->getMap()) {
                mini_map.mapKnightLocation(p.getPlayerNum(), other_kt->getPos().getX(), other_kt->getPos().getY());
            } else {
                mini_map.mapKnightLocation(p.getPlayerNum(), -1, -1);
            }
        }
    }
        
    // Now reschedule the task
    shared_ptr<PlayerTask> pt(new PlayerTask(pl));
    tm.addTask(pt, TP_NORMAL, tm.getGVT() + med.cfgInt("player_task_interval"));
}
