/*
 * player.hpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef PLAYER_HPP
#define PLAYER_HPP

#include "map_support.hpp"
#include "status_display.hpp"

#include "boost/weak_ptr.hpp"
using namespace boost;

#include <deque>
#include <map>
#include <set>
#include <vector>
using namespace std;

class Anim;
class ColourChange;
class Control;
class DungeonMap;
class DungeonView;
class HealingTask;
class Item;
class ItemType;
class Knight;
class MiniMap;
class Quest;
class RespawnTask;
class RoomMap;
class StatusDisplay;
class TaskManager;
class Tile;

//
// Player
// Effectively this just manages respawning of a particular Knight.
// It also keeps track of where that knight's home is, and other such information.
//

class Player {
public:
    // Ctor. (Note that this does not automatically spawn the knight; you have to call respawn
    // explicitly to get the knight to appear for the first time.)
    // Home_facing means the direction YOU have to face to be approaching your home.
    // Control_set (NOT copied) is the set of controls available (for menus or "tapping"
    // fire). (Items and Tiles will be checked for additional controls, and the global
    // controls from control.hpp will also always be available.)
    Player(int plyr_num,
           DungeonMap &home_map, const MapCoord &home_pos, MapDirection home_facing,
           const Anim * anim, const ItemType * dflt_item, 
           const map<const ItemType *, int> * backpack_capacities_,
           const vector<const Control*> &control_set_,
           const vector<shared_ptr<Quest> > &quests_,
           shared_ptr<const ColourChange> secured_home_cc);

    // Add starting gear.
    void addStartingGear(const ItemType &itype, const vector<int> &nos);

    // Setup exit
    void setExit(const MapCoord &exit_loc, MapDirection facing)
    { exit_location = exit_loc; exit_facing = facing; exit_from_players_home = 0; }

    void setExitFromPlayersHome(const Player &p)
    { exit_from_players_home = &p; }
    
    // accessor functions   
    const MapCoord & getHomeLocation() const { return home_location; }  // square just outside the home,
                                                                        // or NULL if no escape.
    MapDirection getHomeFacing() const { return home_facing; }  // direction pointing towards the home.
    const MapCoord & getExitLocation() const;
    MapDirection getExitFacing() const;
    
    shared_ptr<Knight> getKnight() const { return knight.lock(); }
    MapCoord getKnightPos() const; // convenience. (Returns Null mapcoord if knight doesn't exist.)
    DungeonMap * getDungeonMap() const; // convenience
    RoomMap * getRoomMap() const; // convenience
    bool checkQuests(vector<string> &hints_out) const;
    bool isItemInteresting(const ItemType &itm) const; // check item against quest rqmts.
    std::string getQuestMessage() const;
    void getQuestIcons(std::vector<StatusDisplay::QuestIconInfo> &icons_out) const;
    shared_ptr<const ColourChange> getSecuredCC() const { return secured_home_cc; }

    // get player num
    int getPlayerNum() const { return player_num; }

    // Access to DungeonView, MiniMap, StatusDisplay.
    // NOTE: If changing the current room, or mapping/unmapping rooms, you MUST use the
    // special functions below. Otherwise can access dungeonview/minimap/statusdisplay directly.
    // This is so that local state (current room, list of mapped rooms) can be kept up to date.
    // (TODO: could probably come up with a better way of doing this? Maybe ViewManager should
    // keep track of the mapped squares, as it already has a cache of which tiles have been
    // sent to each player?)
    void setCurrentRoom(int room, int width, int height);
    int getCurrentRoom() const;
    void mapCurrentRoom(const MapCoord &top_left);
    void wipeMap();
    bool isRoomMapped(int room) const;
    DungeonView & getDungeonView() const;
    MiniMap & getMiniMap() const;
    StatusDisplay & getStatusDisplay() const;

    // access to controls. this is set by KnightsEngine when we
    // receive a control from the client, and read by KnightTask when
    // we need to process the controls.
    // NOTE: readControl will only return a non-cts control ONCE (for
    // each press). It will return cts controls multiple times (until
    // the user releases it).
    const Control * readControl();
    void setControl(const Control *ctrl);
    
    // reset home (called by HomeManager)
    void resetHome(const MapCoord &mc, const MapDirection &facing);
    
    // event handling functions
    void onDeath(); // Called by Knight dtor. Organizes respawning, and adds a skull.

    // Try to respawn. Returns true if successful.
    bool respawn();

    // Control functions (see .cpp file for more info)
    void getControlInfo(const Control *ctrl_in, const ItemType *& itype_out,
                        weak_ptr<Tile> &tile_out, MapCoord &tile_mc_out) const;
    void computeAvailableControls();

private:
    struct ControlInfo {
        weak_ptr<Tile> tile;
        MapCoord tile_mc;
        const ItemType *item_type; // The item is always assumed to be in the creature's inventory...
        bool primary;
        
        ControlInfo() : item_type(0), primary(true) { }
        bool operator==(const ControlInfo &rhs) const {
            return !(tile < rhs.tile) && !(rhs.tile < tile) 
                && tile_mc == rhs.tile_mc
                && item_type == rhs.item_type
                && primary == rhs.primary;
        }
    };

private:
    static void addTileControls(DungeonMap *dmap, const MapCoord &mc,
                                map<const Control *, ControlInfo> &cmap, bool ahead,
                                bool approaching, MapHeight ht, shared_ptr<Creature>);
    void addItemControls(const ItemType &itype,
                         map<const Control *, ControlInfo> &cmap,
                         shared_ptr<Creature>);
    static void giveStartingGear(shared_ptr<Knight> knight, const vector<pair<const ItemType*, int> > &items);

private:
    friend class RespawnTask;
    friend class HealingTask;

    int player_num;
    const Control *control;

    int current_room;
    int current_room_width, current_room_height;
    std::set<int> mapped_rooms;
    
    DungeonMap &home_dmap;
    const Player * exit_from_players_home;   // If set, this overrides exit_location/exit_facing.
    MapCoord home_location, exit_location;
    MapDirection home_facing, exit_facing;
    const Anim * anim;
    const ItemType * default_item;
    const map<const ItemType *, int> * backpack_capacities;
    const vector<const Control *> & control_set;

    map<const Control *, ControlInfo> current_controls;
    
    weak_ptr<Knight> knight;
    shared_ptr<RespawnTask> respawn_task;
    shared_ptr<HealingTask> healing_task;

    vector<shared_ptr<Quest> > quests;

    shared_ptr<const ColourChange> secured_home_cc;

    deque<vector<pair<const ItemType *, int> > > gears;
};

#endif
