/*
 * player.cpp
 *
 * Player class, and Respawning.
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "action.hpp"
#include "control.hpp"
#include "dungeon_map.hpp"
#include "dungeon_view.hpp"
#include "item_type.hpp"
#include "knight.hpp"
#include "knight_task.hpp"
#include "knights_callbacks.hpp"
#include "mediator.hpp"
#include "mini_map.hpp"
#include "player.hpp"
#include "quest.hpp"
#include "status_display.hpp"
#include "task.hpp"
#include "task_manager.hpp"
#include "tile.hpp"

//
// RespawnTask
//

class RespawnTask : public Task {
public:
    RespawnTask(Player &p) : player(p) { }
    virtual void execute(TaskManager &);
private:
    Player &player;
};

void RespawnTask::execute(TaskManager &tm)
{
    // The player->respawn_task thing is to make sure that any manual calls to respawn() do
    // not conflict with the auto-respawn system.
    if (this != player.respawn_task.get()) return;
    bool success = player.respawn();
    if (!success) {
        tm.addTask(shared_from_this(), TP_NORMAL, 
            tm.getGVT() + Mediator::instance().cfgInt("respawn_delay")); // try again later
    }
}



//
// Player
//

Player::Player(int plyr_num,
               DungeonMap &home_map, const MapCoord &home_pos, MapDirection hf,
               const Anim *a, const ItemType * di, const map<const ItemType*, int> * bc,
               const vector<const Control*> &cs,
               const vector<shared_ptr<Quest> > &qs,
               shared_ptr<const ColourChange> sec_home_cc)
    : player_num(plyr_num), control(0),
      current_room(-1), current_room_width(0), current_room_height(0),
      home_dmap(home_map), exit_from_players_home(0), home_location(home_pos),
      home_facing(hf), anim(a),
      default_item(di), backpack_capacities(bc), control_set(cs), quests(qs),
      secured_home_cc(sec_home_cc)
{
}


void Player::addStartingGear(const ItemType &itype, const vector<int> &numbers)
{
    if (gears.size() < numbers.size()) gears.resize(numbers.size());

    for (int i=0; i<numbers.size(); ++i) {
        gears[i].push_back(make_pair(&itype, numbers[i]));
    }
}


//
// exit location
//

const MapCoord& Player::getExitLocation() const
{
    if (exit_from_players_home) {
        return exit_from_players_home->getHomeLocation();
    } else {
        return exit_location;
    }
}

MapDirection Player::getExitFacing() const
{
    if (exit_from_players_home) {
        return exit_from_players_home->getHomeFacing();
    } else {
        return exit_facing;
    }
}


//
// convenience functions
//

MapCoord Player::getKnightPos() const
{
    shared_ptr<Knight> kt(getKnight());
    if (kt) return kt->getPos();
    else return MapCoord();
}

DungeonMap * Player::getDungeonMap() const
{
    shared_ptr<Knight> kt = getKnight();
    if (kt) return kt->getMap();
    else return 0;
}

RoomMap * Player::getRoomMap() const
{
    DungeonMap * dmap = getDungeonMap();
    if (dmap) return dmap->getRoomMap();
    else return 0;
}

bool Player::checkQuests(vector<string> &hints_out) const
{
    hints_out.clear();
    bool result = true;
    
    shared_ptr<Knight> kt = getKnight();
    if (!kt) return false;
    
    for (vector<shared_ptr<Quest> >::const_iterator it = quests.begin();
    it != quests.end(); ++it) {
        if (!(*it)->check(*kt)) {
            result = false;
            hints_out.push_back((*it)->getHint());
        }
    }
    return result;
}

bool Player::isItemInteresting(const ItemType &itype) const
{
    for (vector<shared_ptr<Quest> >::const_iterator it = quests.begin();
    it != quests.end(); ++it) {
        if ((*it)->isItemInteresting(itype)) {
            return true;
        }
    }
    return false;
}

std::string Player::getQuestMessage() const
{
    std::string msg;
    for (vector<shared_ptr<Quest> >::const_iterator it = quests.begin();
    msg.empty() && it != quests.end(); ++it) {
        msg = (*it)->getQuestMessage();
    }
    return msg;
}

void Player::getQuestIcons(vector<StatusDisplay::QuestIconInfo> &quests_out) const
{
    quests_out.clear();
    shared_ptr<Knight> kt = getKnight();
    
    for (vector<shared_ptr<Quest> >::const_iterator it = quests.begin();
    it != quests.end(); ++it) {
        (*it)->appendQuestIcon(kt.get(), quests_out);
    }
}

//
// mapping functions
//

void Player::setCurrentRoom(int room, int width, int height)
{
    getDungeonView().setCurrentRoom(room, width, height);
    current_room = room;
    current_room_width = width;
    current_room_height = height;
}

int Player::getCurrentRoom() const
{
    return current_room;
}

void Player::mapCurrentRoom(const MapCoord &top_left)
{
    if (current_room == -1) return;
    if (isRoomMapped(current_room)) return;

    MiniMap & mini_map = getMiniMap();

    const int ox = top_left.getX();
    const int oy = top_left.getY();

    const DungeonMap *dmap = getDungeonMap();
    if (!dmap) return;

    std::vector<boost::shared_ptr<Tile> > tiles;
    
    for (int j = 0; j < current_room_height; ++j) {
        for (int i = 0; i < current_room_width; ++i) {

            MiniMapColour col = COL_UNMAPPED;
            dmap->getTiles(MapCoord(i+ox, j+oy), tiles);
            for (std::vector<boost::shared_ptr<Tile> >::const_iterator it = tiles.begin(); it != tiles.end(); ++it) {
                const MiniMapColour c = (*it)->getColour();
                if (c < col) col = c;
            }
            
            mini_map.setColour(i + ox, j + oy, col);
        }
    }
    
    mapped_rooms.insert(current_room);
}

void Player::wipeMap()
{
    Mediator::instance().getCallbacks().getMiniMap(player_num).wipeMap();
    mapped_rooms.clear();
}

bool Player::isRoomMapped(int room) const
{
    return room != -1 && mapped_rooms.find(room) != mapped_rooms.end();
}

DungeonView & Player::getDungeonView() const
{
    return Mediator::instance().getCallbacks().getDungeonView(player_num);
}

MiniMap & Player::getMiniMap() const
{
    return Mediator::instance().getCallbacks().getMiniMap(player_num);
}

StatusDisplay & Player::getStatusDisplay() const
{
    return Mediator::instance().getCallbacks().getStatusDisplay(player_num);
}

void Player::setControl(const Control *ctrl)
{
    // Discontinuous controls are never replaced by a new control (the
    // second control just gets lost!). Other controls are replaced
    // immediately.
    if (control == 0 || control->isContinuous()) {
        control = ctrl;
    }
}

const Control * Player::readControl()
{
    const Control * result = control;
    
    // If the control is discts it will only be returned once
    if (control && !control->isContinuous()) {
        control = 0;
    }

    return result;
}


//
// event handling functions
//

void Player::onDeath()
{
    // This is called by the knight's dtor. It schedules the
    // respawning of a new knight, after a short delay.
    Mediator &mediator = Mediator::instance();
    shared_ptr<RespawnTask> rt(new RespawnTask(*this));
    respawn_task = rt;
    TaskManager &tm(mediator.getTaskManager());
    tm.addTask(respawn_task, TP_NORMAL, tm.getGVT() + mediator.cfgInt("respawn_delay"));
    // Also add a skull
    getStatusDisplay().addSkull();
}



//
// respawn
//

void Player::giveStartingGear(shared_ptr<Knight> knight, const vector<pair<const ItemType *, int> > &items)
{
    for (vector<pair<const ItemType *, int> >::const_iterator it = items.begin(); it != items.end(); ++it) {
        knight->addToBackpack(*it->first, it->second);
    }
}
        
bool Player::respawn()
{
    Mediator &mediator = Mediator::instance();

    // If knight is still alive, then respawn will fail.
    if (knight.lock()) return false;

    // If we do not have a home, then it's game over
    if (getHomeLocation().isNull()) {
        mediator.loseGame(*this);
        return false;
    }
    
    // Check whether the home is free.
    if (home_dmap.getAccess(getHomeLocation(), H_WALKING) < A_CLEAR) {
        return false;
    }
    
    // Respawn successful -- Create a new knight.
    shared_ptr<Knight> my_knight(new Knight(*this, backpack_capacities,
        mediator.cfgInt("knight_hitpoints"), H_WALKING, default_item, anim, 100));
    knight = my_knight;

    // Move the knight into the map.
    // NB This is done AFTER setting "knight" because we want getKnight to respond with 
    // the current knight, even before that knight is added to the map. (Otherwise Game
    // won't correctly process the onAddEntity event.)
    my_knight->addToMap(&home_dmap, home_location);
    my_knight->setFacing(Opposite(home_facing));

    // Add starting gear
    if (!gears.empty()) {
        giveStartingGear(my_knight, gears.front());
        gears.pop_front();
    }

    // Cancel any respawn task that might still be in progress.
    respawn_task = shared_ptr<RespawnTask>();

    // Create a KnightTask so that the knight can be controlled.
    shared_ptr<KnightTask> ktsk(new KnightTask(*this));
    TaskManager &tm(mediator.getTaskManager());
    tm.addTask(ktsk, TP_NORMAL, tm.getGVT() + 1);
    
    return true;
}


//
// control functions
//

// This returns the item and tile that caused a control to be generated. This information can
// be used to set up an ActionData. Note that the corresponding dmap can be taken from the
// Knight itself.
void Player::getControlInfo(const Control *ctrl_in, const ItemType *& itype_out,
                            weak_ptr<Tile> &tile_out, MapCoord &tile_mc_out) const
{
    map<const Control *, ControlInfo>::const_iterator it;
    it = current_controls.find(ctrl_in);
    if (it != current_controls.end()) {
        // Note: Item controls always refer to an item in the knight's
        // inventory, not an item in the map. Therefore we have only
        // itype_out, and not item_dmap_out / item_mc_out.
        itype_out = it->second.item_type;
        tile_out = it->second.tile;
        tile_mc_out = it->second.tile_mc;
    }
}


// Work out the possible controls available to the player. This
// includes "standard" controls plus any controls available from items
// or nearby tiles. The available controls are then sent to the
// DungeonView (which is responsible for selecting a control based on
// the joystick/controller input).
// Player::computeAvailableControls is called by KnightTask::execute.
//
void Player::computeAvailableControls()
{
    shared_ptr<Knight> kt = knight.lock();

    map<const Control *, ControlInfo> new_controls;
    
    if (kt && kt->getMap()) {
        // "Generic" controls
        vector<const Control *>::const_iterator it;
        for (it = control_set.begin(); it != control_set.end(); ++it) {
            if (*it) {
                // check if the control is currently possible
                ActionData ad;
                ad.setActor(kt, true);
                if ((*it)->getAction() && (*it)->getAction()->possible(ad)) {
                    new_controls.insert(make_pair(*it, ControlInfo()));
                }
            }
        }

        // Item-based controls
        for (int i=0; i<kt->getBackpackCount(); ++i) {
            addItemControls(kt->getBackpackItem(i), new_controls, kt);
        }
        if (kt->getItemInHand()) {
            addItemControls(*kt->getItemInHand(), new_controls, kt);
        }

        // Tile-based controls
        MapCoord mc = kt->getPos();
        const bool app = kt->isApproaching();
        const MapHeight ht = kt->getHeight();
        addTileControls(kt->getMap(), mc, new_controls, false, app, ht, kt);
        mc = DisplaceCoord(mc, kt->getFacing());
        addTileControls(kt->getMap(), mc, new_controls, true, app, ht, kt);
    }

    if (new_controls != current_controls) {
        current_controls.swap(new_controls);
        vector<pair<const UserControl *, bool> > ctrl_vec;
        for (map<const Control *, ControlInfo>::const_iterator it = current_controls.begin();
        it != current_controls.end(); ++it) {
            ctrl_vec.push_back(make_pair(it->first, it->second.primary));
        }
        Mediator::instance().getCallbacks().setAvailableControls(getPlayerNum(), ctrl_vec);
    }

}


void Player::addTileControls(DungeonMap *dmap, const MapCoord &mc, 
                             map<const Control *, ControlInfo> &cmap, bool ahead, 
                             bool approaching, MapHeight ht, shared_ptr<Creature> cr)
{
    if (!dmap) return;
    if (!ahead && approaching) return;  // can't do current square if approaching the square ahead.

    vector<shared_ptr<Tile> > tiles;
    dmap->getTiles(mc, tiles);

    for (vector<shared_ptr<Tile> >::const_iterator it = tiles.begin(); it != tiles.end();
    ++it) {
        const Control *ctrl = (*it)->getControl();
        if (ctrl) {
            bool ok;
            bool primary;
            if (ahead) {
                // square-ahead is ok if there are no entities there.
                // (This prevents us from shutting doors on other knights, or monsters.)
                vector<shared_ptr<Entity> > entities;
                dmap->getEntities(mc, entities);
                ok = (entities.empty());

                // square-ahead is usually secondary. However, if we are approaching it, or
                // it cannot be approached, then it is primary.
                primary = (approaching || (*it)->getAccess(ht) != A_APPROACH);
            } else {
                // current square is ok if canActivateHere() is true. It is always primary.
                ok = (*it)->canActivateHere();
                primary = true;
            }
            if (ok) {
                ActionData ad;
                ad.setActor(cr, true);
                ad.setTile(dmap, mc, *it);
                if (ctrl->getAction() && ctrl->getAction()->possible(ad)) {
                    ControlInfo ci;
                    ci.tile = *it;
                    ci.tile_mc = mc;
                    ci.primary = primary;
                    cmap.insert(make_pair(ctrl, ci));
                }
            }
        }
    }
}

void Player::addItemControls(const ItemType &itype, map<const Control *, ControlInfo> &cmap,
                             shared_ptr<Creature> cr)
{
    if (!cr) return;
    const Control *ctrl = itype.getControl();
    if (ctrl) {
        ActionData ad;
        ad.setActor(cr, true);
        ad.setItem(0, MapCoord(), &itype);
        if (ctrl->getAction() && ctrl->getAction()->possible(ad)) {
            ControlInfo ci;
            ci.item_type = &itype;
            cmap.insert(make_pair(ctrl, ci));
        }
    }
}


void Player::resetHome(const MapCoord &mc, const MapDirection &facing)
{
    // NOTE: We assume the player is not currently approaching his own home.
    // (This is OK at the moment because homes can only be secured when no-one is
    // approaching them...)
    home_location = mc;
    home_facing = facing;
}
