/*
 * missile.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "creature.hpp"
#include "dungeon_map.hpp"
#include "item.hpp"
#include "item_type.hpp"
#include "mediator.hpp"
#include "missile.hpp"
#include "rng.hpp"
#include "task_manager.hpp"
#include "tile.hpp"

class MissileTask : public Task {
public:
    MissileTask(weak_ptr<Missile> m) : missile(m) { }
    virtual void execute(TaskManager &);
private:
    weak_ptr<Missile> missile;
};

void MissileTask::execute(TaskManager &tm)
{
    // Check missile is valid
    shared_ptr<Missile> m(missile.lock());
    if (!m) return;
    if (!m->getMap()) return;

    bool delete_from_map = false;

    // Check to see if we have hit something
    shared_ptr<Creature> cr = m->getMap()->getTargetCreature(*m, false);
    if (cr) {
        const int hit_chance = m->range_left * m->itype.getMissileHitMultiplier() + 1;
        if (g_rng.getBool(1.0f - 1.0f / hit_chance)) {
            const RandomInt *st = m->itype.getMissileStunTime();
            const RandomInt *dam = m->itype.getMissileDamage();
            const int stun_time = st? st->get() : 0;
            const int damage = dam? dam->get() : 0;
            if (damage > 0) {
                cr->damage(damage, stun_time + tm.getGVT());
            }
            delete_from_map = true;
        }
    }

    // Have we arrived at the next square yet?
    if (!m->isMoving()) {

        // We have
        bool do_hook = false;

        // Decrease range (taking stairs into account)
        MapCoord mc_ahead = DisplaceCoord(m->getPos(), m->getFacing());
        --(m->range_left);
        vector<shared_ptr<Tile> > tiles;
        m->getMap()->getTiles(mc_ahead, tiles);
        for (vector<shared_ptr<Tile> >::iterator it = tiles.begin(); it != tiles.end(); ++it) {
            if ((*it)->isStair()) {
                MapDirection dn = (*it)->stairsDownDirection();
                if (dn == m->getFacing()) {
                    // going down. add to range
                    if ((mc_ahead.getX() + mc_ahead.getY()) % 2 == 0) {
                        ++(m->range_left);
                    }
                } else if (dn == Opposite(m->getFacing())) {
                    // going up. subtract from range
                    --(m->range_left);
                }
                break;
            }
        }
        if (m->range_left <= 0) {
            delete_from_map = true;
            do_hook = true;
        }
    
        // Check access ahead (and at current sq).
        if (!delete_from_map) {
            const MapAccess ma = m->getMap()->getAccess(mc_ahead,
                                                        MapHeight(H_MISSILES + m->getFacing()));
            if (ma == A_APPROACH) {
                // 'partial' missile access
                int chance = m->itype.getMissileAccessChance();
                if (g_rng.getBool((100-chance) / 100.0f)) {
                    delete_from_map = true;
                    do_hook = true;
                }
            } else if (ma == A_BLOCKED) {
                delete_from_map = true;
                do_hook = true;
            }
        }

        if (do_hook) {
            Mediator::instance().runHook("HOOK_MISSILE_MISS", m->getMap(), m->getPos());
        }
    }
    
    if (delete_from_map) {
        // Remove the missile from the map. Drop item if necessary.
        // NOTE: we don't check if the drop was successful; if there was nowhere to
        // place the missile then tough, the item is lost. (Although we do set "allow_nonlocal"
        // so as to give the best chance of finding a place for the item.)
        if (m->drop_after) {
            shared_ptr<Item> item(new Item(m->itype));
            DropItem(item, *m->getMap(), m->getPos(), true, m->getFacing(), shared_ptr<Creature>());
        }
        m->rmFromMap();
    } else {

        // Move on if necessary
        if (!m->isMoving()) {
            m->move(MT_MOVE);
        }

        // Reschedule. We check for collisions every "missile_check_interval" but make sure
        // to run the missile task at arrival_time+1 if that is sooner.
        const int new_time = min(tm.getGVT() + Mediator::instance().cfgInt("missile_check_interval"),
                                 m->getArrivalTime() + 1);
        tm.addTask(shared_from_this(), TP_NORMAL, new_time);
    }
}

bool CreateMissile(DungeonMap &dmap, const MapCoord &mc, MapDirection dir,
                   const ItemType &itype, bool drop_after, bool with_strength)
{
    // Check if we can place a missile here
    if (!dmap.canPlaceMissile(mc, dir)) return false;

    // Check access ahead. If A_APPROACH ("partial missile access") then check random chance that missile cant be placed
    const MapAccess acc = dmap.getAccess(DisplaceCoord(mc, dir), MapHeight(H_MISSILES + dir));
    if (acc == A_APPROACH) {
        if (g_rng.getBool((100 - itype.getMissileAccessChance()) / 100.0f)) {
            return false;
        }
    }
    
    Mediator &mediator = Mediator::instance();

    // Create the missile
    shared_ptr<Missile> m(new Missile(itype, drop_after));
    if (with_strength) m->doubleRange();
    m->setFacing(dir);
    m->addToMap(&dmap, mc);
    m->move(MT_MOVE, true);  // start it forward half a square
    shared_ptr<MissileTask> mt(new MissileTask(m));
    const int new_time = min(mediator.getGVT() + mediator.cfgInt("missile_check_interval"),
                              m->getArrivalTime() + 1);
    mediator.getTaskManager().addTask(mt, TP_NORMAL, new_time);
    return true;
}

Missile::Missile(const ItemType &it, bool da)
    : itype(it), range_left(it.getMissileRange()), drop_after(da)
{
    setAnim(it.getMissileAnim());
    setSpeed(it.getMissileSpeed());
}

MapHeight Missile::getHeight() const
{
    return MapHeight(H_MISSILES + getFacing());
}

