/*
 * menu_constraints.cpp
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "kfile.hpp"
#include "knights_config_impl.hpp"
#include "menu_constraints.hpp"
#include "menu_int.hpp"
#include "menu_selections.hpp"

//
// MenuConstraint base class
//

class MenuConstraint {
public:
    MenuConstraint(const std::string &k, int v) : key(k), val(v) { }
    bool applicable(const MenuSelections &msel) const { return msel.getValue(key) == val; }
    bool isQuest() const { return key == "quest"; }  // see below
    
    virtual ~MenuConstraint() { }
    virtual bool apply(MenuSelections &msel) const = 0;
    virtual bool canSetTo(const MenuSelections &msel, const std::string &key, int val) const = 0;

private:
    // The constraint is only active if the given key has the given value
    std::string key;
    int val;
};

//
// Constrain
//

class Constrain : public MenuConstraint {
public:
    Constrain(const std::string &ak, int av, const std::string &ck, const MenuInt * cv) 
        : MenuConstraint(ak,av), key(ck), val(cv) { }
    virtual bool apply(MenuSelections &msel) const;
    virtual bool canSetTo(const MenuSelections &msel, const string &k, int v) const;
private:
    std::string key;
    const MenuInt *val;
};

bool Constrain::apply(MenuSelections &msel) const
{
    if (!val) return false;
    const int new_val = val->getValue(msel);
    if (msel.getValue(key) == new_val) return false;
    msel.setValue(key, new_val);
    return true;
}

bool Constrain::canSetTo(const MenuSelections &msel, const string &k, int v) const
{
    if (k != key) return true;
    const int new_val = val->getValue(msel);
    return v == new_val;
}

//
// ConstrainMin
//

class ConstrainMin : public MenuConstraint {
public:
    ConstrainMin(const string &ak, int av, const string &ck, const MenuInt * cv) : MenuConstraint(ak,av), key(ck), val(cv) { }
    virtual bool apply(MenuSelections &msel) const;
    virtual bool canSetTo(const MenuSelections &msel, const string &k, int v) const;
private:
    std::string key;
    const MenuInt *val;
};

bool ConstrainMin::apply(MenuSelections &msel) const
{
    if (!val) return false;
    const int mn = val->getValue(msel);
    const int cur = msel.getValue(key);
    if (cur < mn) {
        msel.setValue(key, mn);
        return true;
    } else {
        return false;
    }
}

bool ConstrainMin::canSetTo(const MenuSelections &msel, const string &k, int v) const
{
    if (k != key) return true;
    const int mn = val->getValue(msel);
    return v >= mn;
}


//
// ConstrainMax
//

class ConstrainMax : public MenuConstraint {
public:
    ConstrainMax(const string &ak, int av, const string &ck, const MenuInt * cv) : MenuConstraint(ak,av), key(ck), val(cv) { }
    virtual bool apply(MenuSelections &msel) const;
    virtual bool canSetTo(const MenuSelections &msel, const std::string &k, int v) const;
private:
    std::string key;
    const MenuInt *val;
};

bool ConstrainMax::apply(MenuSelections &msel) const
{
    if (!val) return false;
    const int mx = val->getValue(msel);
    const int cur = msel.getValue(key);
    if (cur > mx) {
        msel.setValue(key, mx);
        return true;
    } else {
        return false;
    }
}

bool ConstrainMax::canSetTo(const MenuSelections &msel, const string &k, int v) const
{
    if (k != key) return true;
    const int mx = val->getValue(msel);
    return v <= mx;
}


//
// Create function
//

boost::shared_ptr<MenuConstraint> CreateMenuConstraint(const std::string &key, int val,
                                                       const std::string &name, KnightsConfigImpl &kc)
{
    boost::shared_ptr<MenuConstraint> result;
    
    KConfig::KFile::List lst(*kc.getKFile(), "", 2);
    lst.push(0);
    std::string con_key = kc.getKFile()->popString();
    lst.push(1);
    const MenuInt * con_mi = kc.popMenuInt();
    
    if (name == "Constrain") {
        result.reset(new Constrain(key, val, con_key, con_mi));
    } else if (name == "ConstrainMin") {
        result.reset(new ConstrainMin(key, val, con_key, con_mi));
    } else if (name == "ConstrainMax") {
        result.reset(new ConstrainMax(key, val, con_key, con_mi));
    } else {
        kc.getKFile()->errExpected("MenuDirective");
    }
    
    return result;
}


//
// implementation of MenuConstraints methods
//

void MenuConstraints::apply(const Menu &menu, MenuSelections &msel) const
{
    // First update the current values to a legal configuration.
    // (Limit of 100 iterations to prevent infinite loops.)
    for (int count = 0; count < 100; ++count) {
        bool changed = false;
        
        for (int i = 0; i < menu.getNumItems(); ++i) {
            const MenuItem &item = menu.getItem(i);
            const std::string &key = item.getKey();
            const int min_value = item.getMinValue();
            const int max_value = min_value + item.getNumValues() - 1;

            const int cur_val = msel.getValue(key);

            if (cur_val < min_value || cur_val > max_value) {
                // set it back to its lower bound
                msel.setValue(key, min_value);
                changed = true;
            } else {
                // apply all constraints to this value.
                for (std::vector<boost::shared_ptr<MenuConstraint> >::const_iterator con = constraints.begin();
                con != constraints.end(); ++con) {
                    if ((*con)->applicable(msel) && (*con)->apply(msel)) changed = true;
                }
            }
        }

        // exit if nothing changed since previous iteration
        if (!changed) break;
    }

    // Now set all 'allowed values'
    for (int i = 0; i < menu.getNumItems(); ++i) {
        const MenuItem &item = menu.getItem(i);
        const std::string &key = item.getKey();

        std::vector<int> allowed_values;
        
        for (int x = 0; x < item.getNumValues(); ++x) {
            const int proposed_value = item.getMinValue() + x;
            bool can_set = true;
            for (std::vector<boost::shared_ptr<MenuConstraint> >::const_iterator con = constraints.begin();
            con != constraints.end(); ++con) {
                if (!(*con)->isQuest()) { // allow them to set up a custom quest without having to deselect current quest first
                    if ((*con)->applicable(msel) && !(*con)->canSetTo(msel, key, proposed_value)) {
                        can_set = false;
                        break;
                    }
                }
            }
            if (can_set) allowed_values.push_back(proposed_value);
        }
        
        msel.setAllowedValues(key, allowed_values);
    }
}

void MenuConstraints::addConstraintFromKFile(const std::string &key, int val, const std::string &dname, KnightsConfigImpl &kc)
{
    constraints.push_back(CreateMenuConstraint(key, val, dname, kc));
}
