/*
 * mediator.hpp
 *
 * The Mediator holds references to the key objects in the system
 * (mainly the DungeonMap and the various "Manager" objects). These
 * objects can easily be accessed via the Mediator::instance()
 * function.
 *
 * Note that there are various "onEvent" routines that get called when
 * certain game events occur. (These are always called AFTER the
 * change is made.) These "onEvent" routines call the Managers (e.g.
 * the ViewManager and the EventManager) to tell them about the event.
 * Note that the Managers themselves cannot be accessed directly; they
 * are only accessed through the Mediator. This both reduces
 * dependencies and also increases abstraction (since clients don't
 * need to know the details of which managers get called in response
 * to which events).
 *
 * The GVT (global virtual time) can also be accessed from this class,
 * as can other things like the player list, the dungeon map, the task
 * manager, etc.
 * 
 * Mediator::instance() is programmed in a thread-local way, i.e.
 * there is one Mediator per server thread. This allows several games
 * to run concurrently, provided they run in separate threads.
 * 
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef MEDIATOR_HPP
#define MEDIATOR_HPP

#include "exception_base.hpp"
#include "map_support.hpp"

#include "boost/shared_ptr.hpp"
using namespace boost;

#include <vector>
using namespace std;

class ConfigMap;
class Creature;
class DungeonMap;
class Entity;
class EventManager;
class GoreManager;
class Graphic;
class HomeManager;
class Item;
class KnightsCallbacks;
class MapCoord;
class MonsterManager;
class MonsterType;
class Player;
class Sound;
class StuffManager;
class TaskManager;
class Tile;
class ViewManager;

// Exception classes:-

class MediatorCreatedTwice : public ExceptionBase {
public:
    MediatorCreatedTwice() : ExceptionBase("Mediator created twice!") { }
};

class MediatorUnavailable : public ExceptionBase {
public:
    // NB if this is thrown it probably indicates that either (a) the mediator was accessed
    // before the game started, or (b) the mediator was accessed from outside of a game thread.
    MediatorUnavailable() : ExceptionBase("No Mediator instance available!") { }
};

class CallbacksUnavailable : public ExceptionBase {
public:
    CallbacksUnavailable() : ExceptionBase("KnightsCallbacks not found") { }
};


// The mediator class itself:-

class Mediator {
public: 
    // Access to instance. (This is like the Singleton pattern except
    // (a) there is one Mediator per thread rather than one across the
    // whole program, and (b) the instance has to be created
    // explicitly by createInstance.)
    static Mediator & instance();

    //
    // Tell us which KnightsCallbacks to use when events of interest
    // to the client occur.
    //
    // NOTE: KnightsEngine has to be quite careful to ensure that the
    // KnightsCallbacks pointer is always valid when running anything
    // that might call the Mediator (i.e. KnightsEngine::update).
    //
    void setCallbacks(KnightsCallbacks *cb) { callbacks = cb; }
    KnightsCallbacks & getCallbacks() const;   // throws if cb==0.

    //
    // Access general configuration settings.
    //
    int cfgInt(const std::string &key) const;
    const std::string &cfgString(const std::string &key) const;
    
    
    //
    // "Event" routines:-
    //

    // Entities
    void onAddEntity(shared_ptr<Entity>);
    void onRmEntity(shared_ptr<Entity>);
    void onRepositionEntity(shared_ptr<Entity>);
    void onChangeEntityMotion(shared_ptr<Entity>, bool missile_mode);
    void onFlipEntityMotion(shared_ptr<Entity>);
    void onChangeEntityAnim(shared_ptr<Entity>);
    void onChangeEntityFacing(shared_ptr<Entity>);
    void onChangeEntityVisible(shared_ptr<Entity>);
    
    // Icons
    void placeIcon(DungeonMap &, const MapCoord &, const Graphic *, int duration);

    // Items
    void onAddItem(const DungeonMap &, const MapCoord &, const Item &);
    void onRmItem(const DungeonMap &, const MapCoord &, const Item &);
    void onChangeItemGraphic(const DungeonMap &, const MapCoord &, const Item &);

    // Tiles
    void onAddTile(DungeonMap &, const MapCoord &, Tile &);
    void onRmTile(DungeonMap &, const MapCoord &, Tile &);
    void onChangeTile(const DungeonMap &, const MapCoord &, const Tile &);

    // Generic Event Hooks (Usually used for sound effects.)
    void runHook(const string &hook_name, shared_ptr<Creature> cr);
    void runHook(const string &hook_name, DungeonMap *dmap, const MapCoord &mc);
    
    // Gore effects.
    // (These just call into GoreManager at present.)
    // Creature is responsible for calling these at appropriate times.
    void placeBlood(DungeonMap &, const MapCoord &);
    void placeKnightCorpse(DungeonMap &, const MapCoord &, const Player &, bool blood);
    void placeMonsterCorpse(DungeonMap &, const MapCoord &, const MonsterType &);

    // Misc view effects
    void flashScreen(shared_ptr<Entity> ent, int delay);
    void playSound(DungeonMap &, const MapCoord &, const Sound &sound, int frequency, bool all);
    
    // Home manager (wand of securing)
    void secureHome(const Player &pl, DungeonMap &dmap, const MapCoord &pos,
                    MapDirection facing, shared_ptr<Tile> secured_wall_tile);

    // End-of-Game handling
    void winGame(const Player &);  // ends the game
    void loseGame(const Player &); // Also ends the game (at present) because only 2 players
    const Player * getWinningPlayer() const  // returns 0 if game still in progress
        { return winning_player; }
    
    // Various accessor functions.
    shared_ptr<DungeonMap> getMap() const { return dmap; }
    const vector<Player*> &getPlayers() const { return players; }
    MonsterManager & getMonsterManager() const { return monster_manager; }
    StuffManager & getStuffManager() const { return stuff_manager; }
    TaskManager & getTaskManager() const { return task_manager; }
    int getGVT() const;
    
    
    //
    // Initialization stuff -- The following routines are called by
    // KnightsEngine (which is responsible for the creation and
    // destruction of the Mediator).
    //
    
    static void createInstance(EventManager &em, GoreManager &gm, HomeManager &hm,
                               MonsterManager &mm, StuffManager &sm, TaskManager &tm,
                               ViewManager &vm, boost::shared_ptr<const ConfigMap> cmap);
    static void destroyInstance();  // must be called before the game thread exits, otherwise will leak memory
    void setMap(shared_ptr<DungeonMap> dm) { dmap = dm; }
    void addPlayer(Player &);

private:
    // Note -- Mediator is not responsible for creating all of these
    // objects. That is done by the higher-ups (ie KnightsEngine).
    // Here, we mostly only have to keep references to things...
    Mediator(EventManager &em, GoreManager &gm, HomeManager &hm, MonsterManager &mm,
             StuffManager &sm, TaskManager &tm, ViewManager &vm, boost::shared_ptr<const ConfigMap> cmap)
        : config_map(cmap), event_manager(em), gore_manager(gm), home_manager(hm), monster_manager(mm),
          stuff_manager(sm), task_manager(tm), view_manager(vm), winning_player(0), callbacks(0) { }
    void operator=(const Mediator &) const;  // not defined
    Mediator(const Mediator &);              // not defined
    
private:

    // Config Map
    boost::shared_ptr<const ConfigMap> config_map;
    
    // References to the Manager objects for this game.
    EventManager &event_manager;
    GoreManager &gore_manager;
    HomeManager &home_manager;
    MonsterManager &monster_manager;
    StuffManager &stuff_manager;
    TaskManager &task_manager;
    ViewManager &view_manager;  

    // Other variables
    shared_ptr<DungeonMap> dmap;
    vector<Player*> players;    
    const Player *winning_player;

    // KnightsCallbacks.
    KnightsCallbacks *callbacks;
};

#endif
