/*
 * mediator.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "config_map.hpp"
#include "creature.hpp"
#include "dungeon_view.hpp"
#include "entity.hpp"
#include "event_manager.hpp"
#include "gore_manager.hpp"
#include "home_manager.hpp"
#include "knights_callbacks.hpp"
#include "mediator.hpp"
#include "player.hpp"
#include "player_task.hpp"
#include "task_manager.hpp"
#include "view_manager.hpp"

#include "boost/thread/tss.hpp"

namespace {
    // Thread-local storage for the mediator.
    boost::thread_specific_ptr<Mediator> g_mediator_ptr;
}


//
// initialization stuff
//

void Mediator::createInstance(EventManager &em, GoreManager &gm, HomeManager &hm,
                              MonsterManager &mm, StuffManager &sm, TaskManager &tm,
                              ViewManager &vm, boost::shared_ptr<const ConfigMap> cmap)
{
    if (g_mediator_ptr.get()) throw MediatorCreatedTwice();
    g_mediator_ptr.reset(new Mediator(em, gm, hm, mm, sm, tm, vm, cmap));
}

void Mediator::destroyInstance()
{
    if (!g_mediator_ptr.get()) throw MediatorUnavailable();
    g_mediator_ptr.reset();
}

void Mediator::addPlayer(Player &player)
{
    players.push_back(&player);
    view_manager.onAddPlayer(player);
    shared_ptr<PlayerTask> pt(new PlayerTask(player)); 
    task_manager.addTask(pt, TP_NORMAL, 1);  // initial exec time does not really matter
}


//
// instance
//

Mediator & Mediator::instance()
{
    Mediator *m = g_mediator_ptr.get();
    if (!m) throw MediatorUnavailable();
    else return *m;
}


//
// configmap
//

int Mediator::cfgInt(const std::string &key) const
{
    return config_map->getInt(key);
}

const std::string &Mediator::cfgString(const std::string &key) const
{
    return config_map->getString(key);
}


//
// entities
//

void Mediator::onAddEntity(shared_ptr<Entity> entity)
{
    view_manager.onAddEntity(entity);
    Creature *cr = dynamic_cast<Creature*>(entity.get());
    if (cr) event_manager.onAddCreature(*cr);
}

void Mediator::onRmEntity(shared_ptr<Entity> entity)
{
    view_manager.onRmEntity(entity);
    Creature *cr = dynamic_cast<Creature*>(entity.get());
    if (cr) event_manager.onRmCreature(*cr);
}

void Mediator::onRepositionEntity(shared_ptr<Entity> entity)
{
    view_manager.onRepositionEntity(entity);
    Creature *cr = dynamic_cast<Creature*>(entity.get());
    if (cr) event_manager.postRepositionCreature(*cr);
}

void Mediator::onChangeEntityMotion(shared_ptr<Entity> entity, bool missile_mode)
{
    view_manager.onChangeEntityMotion(entity, missile_mode);
    Creature *cr = dynamic_cast<Creature*>(entity.get());
    if (cr) event_manager.onChangeEntityMotion(*cr);
}

void Mediator::onFlipEntityMotion(shared_ptr<Entity> entity)
{
    view_manager.onFlipEntityMotion(entity);
}

void Mediator::onChangeEntityAnim(shared_ptr<Entity> entity)
{
    view_manager.onChangeEntityAnim(entity);
}

void Mediator::onChangeEntityFacing(shared_ptr<Entity> entity)
{
    view_manager.onChangeEntityFacing(entity);
}

void Mediator::onChangeEntityVisible(shared_ptr<Entity> entity)
{
    view_manager.onChangeEntityVisible(entity);
}


//
// icons
//

void Mediator::placeIcon(DungeonMap &dmap, const MapCoord &mc, const Graphic *g, int duration)
{
    view_manager.placeIcon(dmap, mc, g, duration);
}


//
// items
//

void Mediator::onAddItem(const DungeonMap &dmap, const MapCoord &mc, const Item &item)
{
    view_manager.onAddItem(dmap, mc, item);
}

void Mediator::onRmItem(const DungeonMap &dmap, const MapCoord &mc, const Item &item)
{
    view_manager.onRmItem(dmap, mc, item);
}

void Mediator::onChangeItemGraphic(const DungeonMap &dmap, const MapCoord &mc, const Item &item)
{
    view_manager.onChangeItemGraphic(dmap, mc, item);
}


//
// tiles
//

void Mediator::onAddTile(DungeonMap &dmap, const MapCoord &mc, Tile &tile)
{
    view_manager.onAddTile(dmap, mc, tile);
    event_manager.onAddTile(dmap, mc, tile);
}

void Mediator::onRmTile(DungeonMap &dmap, const MapCoord &mc, Tile &tile)
{
    view_manager.onRmTile(dmap, mc, tile);
    event_manager.onRmTile(dmap, mc, tile);
}

void Mediator::onChangeTile(const DungeonMap &dmap, const MapCoord &mc, const Tile &tile)
{
    view_manager.onChangeTile(dmap, mc, tile);
}


//
// generic event hooks
//

void Mediator::runHook(const string &hook_name, shared_ptr<Creature> cr)
{
    event_manager.runHook(hook_name, cr);
}

void Mediator::runHook(const string &hook_name, DungeonMap *dmap, const MapCoord &mc)
{
    event_manager.runHook(hook_name, dmap, mc);
}


//
// blood/gore effects
//

void Mediator::placeBlood(DungeonMap &dmap, const MapCoord &mc)
{
    gore_manager.placeBlood(dmap, mc);
}

void Mediator::placeKnightCorpse(DungeonMap &dmap, const MapCoord &mc, const Player &p, bool b)
{
    gore_manager.placeKnightCorpse(dmap, mc, p, b);
}

void Mediator::placeMonsterCorpse(DungeonMap &dmap, const MapCoord &mc, const MonsterType &m)
{
    gore_manager.placeMonsterCorpse(dmap, mc, m);
}


//
// Misc view effects
//

void Mediator::flashScreen(shared_ptr<Entity> ent, int delay)
{
    view_manager.flashScreen(ent, delay);
}

void Mediator::playSound(DungeonMap &dm, const MapCoord &mc, const Sound &sound, int frequency, bool all)
{
    view_manager.playSound(dm, mc, sound, frequency, all);
}


//
// secureHome
//

void Mediator::secureHome(const Player &pl, DungeonMap &dmap, const MapCoord &pos,
                          MapDirection facing, shared_ptr<Tile> wall)
{
    home_manager.secureHome(pl, dmap, pos, facing, wall);
}


//
// ending the game
//

void Mediator::winGame(const Player &pl)
{
    winning_player = &pl;

    // tell clients that the game has ended
    for (std::vector<Player*>::const_iterator it = players.begin(); it != players.end(); ++it) {
        if (*it == &pl) {
            Mediator::getCallbacks().winGame((*it)->getPlayerNum());
        } else {
            Mediator::getCallbacks().loseGame((*it)->getPlayerNum());
        }
    }

    // kill all tasks, this prevents anything further from happening in-game.
    task_manager.rmAllTasks();
}

void Mediator::loseGame(const Player &pl)
{
    // We are really assuming 2 players here ...
    for (int i=0; i<players.size(); ++i) {
        if (players[i] != &pl) {
            winGame(*players[i]);
            break;
        }
    }
}

int Mediator::getGVT() const
{
    return task_manager.getGVT();
}

KnightsCallbacks & Mediator::getCallbacks() const
{
    if (!callbacks) throw UnexpectedError("Mediator: KnightsCallbacks not set");
    return *callbacks;
}
