/*
 * magic_actions.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "dispel_magic.hpp"
#include "dungeon_view.hpp"
#include "knight.hpp"
#include "lockable.hpp"
#include "magic_actions.hpp"
#include "magic_map.hpp"
#include "mediator.hpp"
#include "monster_manager.hpp"
#include "player.hpp"
#include "potion_magic.hpp"
#include "task_manager.hpp"
#include "teleport.hpp"

#include "random_int.hpp"
using namespace KConfig;

#ifdef _MSC_VER
    // fix "bug" with MSVC static libraries and global constructors
    extern "C" void InitMagicActions()
    {
    }
#endif


//
// SetPotion, FlashMessage
//

namespace {
    void SetPotion(int gvt, shared_ptr<Knight> kt, PotionMagic pm, const RandomInt *dur)
    {
        if (!kt) return;
        int stop_time = gvt;
        if (dur) stop_time += dur->get();
        kt->setPotionMagic(pm, stop_time);
    }

    void FlashMessage(shared_ptr<Knight> kt, const string &msg)
    {
        if (!kt) return;
        kt->getPlayer().getDungeonView().flashMessage(msg, 4);
    }

    void FlashMessage(const ActionData &ad, const string &msg)
    {
        FlashMessage(dynamic_pointer_cast<Knight>(ad.getActor()), msg);
    }
}


//
// A_Attractor
//

void A_Attractor::execute(const ActionData &ad) const
{
    shared_ptr<Creature> cr(ad.getActor());
    if (cr && cr->getMap()) {
        shared_ptr<Knight> nearest = FindNearestOtherKnight(*cr->getMap(), cr->getPos());
        TeleportToRoom(nearest, cr);
    }
}

A_Attractor::Maker A_Attractor::Maker::register_me;

Action * A_Attractor::Maker::make(ActionPars &pars) const
{
    pars.require(0);
    return new A_Attractor;
}


//
// A_DispelMagic
//

void A_DispelMagic::execute(const ActionData &ad) const
{
    FlashMessage(ad, msg);
    DispelMagic(Mediator::instance().getPlayers());
}

A_DispelMagic::Maker A_DispelMagic::Maker::register_me;

Action * A_DispelMagic::Maker::make(ActionPars &pars) const
{
    pars.require(1);
    return new A_DispelMagic(pars.getString(0));
}


//
// A_Healing
//

void A_Healing::execute(const ActionData &ad) const
{
    if (ad.getActor() && ad.getActor()->getHealth() < ad.getActor()->getMaxHealth()) {
        // reset his health to maximum
        FlashMessage(ad, msg);
        ad.getActor()->addToHealth( ad.getActor()->getMaxHealth() - ad.getActor()->getHealth() );
    }
}

A_Healing::Maker A_Healing::Maker::register_me;

Action * A_Healing::Maker::make(ActionPars &pars) const
{
    pars.require(1);
    return new A_Healing(pars.getString(0));
}


//
// A_Invisibility
//

void A_Invisibility::execute(const ActionData &ad) const
{
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(ad.getActor());
    FlashMessage(kt, msg);
    SetPotion(Mediator::instance().getGVT(), kt, INVISIBILITY, dur);
}

A_Invisibility::Maker A_Invisibility::Maker::register_me;

Action * A_Invisibility::Maker::make(ActionPars &pars) const
{
    pars.require(2);
    return new A_Invisibility(pars.getRandomInt(0), pars.getString(1));
}

//
// A_Invulnerability
//

void A_Invulnerability::execute(const ActionData &ad) const
{
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(ad.getActor());
    if (kt) {
        FlashMessage(kt, msg);
        int stop_time = Mediator::instance().getGVT();
        if (dur) stop_time += dur->get();
        kt->setInvulnerability(true, stop_time);
    }
}

A_Invulnerability::Maker A_Invulnerability::Maker::register_me;

Action * A_Invulnerability::Maker::make(ActionPars &pars) const
{
    pars.require(2);
    return new A_Invulnerability(pars.getRandomInt(0), pars.getString(1));
}


//
// A_MagicMapping
//

void A_MagicMapping::execute(const ActionData &ad) const
{
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(ad.getActor());
    if (kt) MagicMapping(kt);
}

A_MagicMapping::Maker A_MagicMapping::Maker::register_me;

Action * A_MagicMapping::Maker::make(ActionPars &pars) const
{
    pars.require(0);
    return new A_MagicMapping;
}


//
// A_OpenWays
//

void A_OpenWays::execute(const ActionData &ad) const
{
    DungeonMap *dm;
    MapCoord mc;
    shared_ptr<Tile> t;
    ad.getTile(dm, mc, t);
    if (!dm || !t) return;
    shared_ptr<Creature> actor = ad.getActor();
    
    // The current rule is that "open ways" only applies to Lockables that can
    // be trapped. This includes doors/chests but excludes gates (which is what we want).
    shared_ptr<Lockable> lock = dynamic_pointer_cast<Lockable>(t);
    if (lock && lock->canTrap()) {
        lock->onOpen(*dm, mc, actor);
    } else if (actor->hasStrength()) {
        // We turn off allow_strength for wand of open ways, so the normal melee code
        // will not attempt to damage the tile if the knight has strength. This is good
        // for doors/chests but for barrels, etc we DO want the normal smashing to take
        // place. So we do it manually here:
        t->damage(*dm, mc, 9999, actor);
    }
}

A_OpenWays::Maker A_OpenWays::Maker::register_me;

Action * A_OpenWays::Maker::make(ActionPars &pars) const
{
    pars.require(0);
    return new A_OpenWays;
}


//
// A_Paralyzation
//

void A_Paralyzation::execute(const ActionData &ad) const
{
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(ad.getActor());
    SetPotion(Mediator::instance().getGVT(), kt, PARALYZATION, dur);
}

A_Paralyzation::Maker A_Paralyzation::Maker::register_me;

Action * A_Paralyzation::Maker::make(ActionPars &pars) const
{
    pars.require(1);
    return new A_Paralyzation(pars.getRandomInt(0));
}


//
// A_Poison
//

void A_Poison::execute(const ActionData &ad) const
{
    if (ad.getActor()) {
        FlashMessage(ad, msg);
        ad.getActor()->poison();
    }
}

A_Poison::Maker A_Poison::Maker::register_me;

Action * A_Poison::Maker::make(ActionPars &pars) const
{
    pars.require(1);
    return new A_Poison(pars.getString(0));
}


//
// A_PoisonImmunity
//

void A_PoisonImmunity::execute(const ActionData &ad) const
{
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(ad.getActor());
    if (kt) {
        FlashMessage(kt, msg);
        const int stop_time = Mediator::instance().getGVT() + (dur ? dur->get() : 0);
        kt->setPoisonImmunity(true, stop_time);
    }
}

A_PoisonImmunity::Maker A_PoisonImmunity::Maker::register_me;

Action * A_PoisonImmunity::Maker::make(ActionPars &pars) const
{
    pars.require(2);
    return new A_PoisonImmunity(pars.getRandomInt(0), pars.getString(1));
}


//
// A_Quickness
//

void A_Quickness::execute(const ActionData &ad) const
{
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(ad.getActor());
    FlashMessage(kt, msg);
    SetPotion(Mediator::instance().getGVT(), kt, QUICKNESS, dur);
}

A_Quickness::Maker A_Quickness::Maker::register_me;

Action * A_Quickness::Maker::make(ActionPars &pars) const
{
    pars.require(2);
    return new A_Quickness(pars.getRandomInt(0), pars.getString(1));
}


//
// A_Regeneration
//

void A_Regeneration::execute(const ActionData &ad) const
{
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(ad.getActor());
    FlashMessage(kt, msg);
    SetPotion(Mediator::instance().getGVT(), kt, REGENERATION, dur);
}

A_Regeneration::Maker A_Regeneration::Maker::register_me;

Action * A_Regeneration::Maker::make(ActionPars &pars) const
{
    pars.require(2);
    return new A_Regeneration(pars.getRandomInt(0), pars.getString(1));
}


//
// A_RevealLocation
//

void A_RevealLocation::execute(const ActionData &ad) const
{
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(ad.getActor());
    if (kt) {
        const int stop_time = Mediator::instance().getGVT() + (dur ? dur->get() : 0);
        kt->setRevealLocation(true, stop_time);
    }
}

A_RevealLocation::Maker A_RevealLocation::Maker::register_me;

Action * A_RevealLocation::Maker::make(ActionPars &pars) const
{
    pars.require(1);
    return new A_RevealLocation(pars.getRandomInt(0));
}


//
// A_SenseItems
//

void A_SenseItems::execute(const ActionData &ad) const
{
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(ad.getActor());
    if (kt) {
        const int stop_time = Mediator::instance().getGVT() + (dur ? dur->get() : 0);
        SenseItems(kt, stop_time);
    }
}

A_SenseItems::Maker A_SenseItems::Maker::register_me;

Action * A_SenseItems::Maker::make(ActionPars &pars) const
{
    pars.require(1);
    return new A_SenseItems(pars.getRandomInt(0));
}


//
// A_SenseKnight
//

void A_SenseKnight::execute(const ActionData &ad) const
{
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(ad.getActor());
    if (kt) {
        const int stop_time = Mediator::instance().getGVT() + (dur ? dur->get() : 0);
        kt->setSenseKnight(true, stop_time);
    }
}

A_SenseKnight::Maker A_SenseKnight::Maker::register_me;

Action * A_SenseKnight::Maker::make(ActionPars &pars) const
{
    pars.require(1);
    return new A_SenseKnight(pars.getRandomInt(0));
}


//
// A_Strength
//

void A_Strength::execute(const ActionData &ad) const
{
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(ad.getActor());
    FlashMessage(kt, msg);
    SetPotion(Mediator::instance().getGVT(), kt, STRENGTH, dur);
}

A_Strength::Maker A_Strength::Maker::register_me;

Action * A_Strength::Maker::make(ActionPars &pars) const
{
    pars.require(2);
    return new A_Strength(pars.getRandomInt(0), pars.getString(1));
}


//
// A_Super
//

void A_Super::execute(const ActionData &ad) const
{
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(ad.getActor());
    FlashMessage(kt, msg);
    SetPotion(Mediator::instance().getGVT(), kt, SUPER, dur);
}

A_Super::Maker A_Super::Maker::register_me;

Action * A_Super::Maker::make(ActionPars &pars) const
{
    pars.require(2);
    return new A_Super(pars.getRandomInt(0), pars.getString(1));
}


//
// A_Teleport
//

void A_Teleport::execute(const ActionData &ad) const
{
    // Only Knights can teleport, because we don't want zombies to be teleported about
    // by pentagrams....
    shared_ptr<Creature> cr(ad.getActor());
    if (cr && cr->getMap() && dynamic_cast<Knight*>(cr.get())) {
        shared_ptr<Knight> nearest = FindNearestOtherKnight(*cr->getMap(), cr->getPos());
        TeleportToRoom(cr, nearest);
    }
}

A_Teleport::Maker A_Teleport::Maker::register_me;

Action * A_Teleport::Maker::make(ActionPars &pars) const
{
    pars.require(0);
    return new A_Teleport;
}


//
// A_WipeMap
//

void A_WipeMap::execute(const ActionData &ad) const
{
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(ad.getActor());
    if (kt) WipeMap(kt);
}

A_WipeMap::Maker A_WipeMap::Maker::register_me;

Action * A_WipeMap::Maker::make(ActionPars &pars) const
{
    pars.require(0);
    return new A_WipeMap;
}


//
// A_ZombifyActor, A_ZombifyTarget
//

namespace {
    void ZombifyCreature(shared_ptr<Creature> cr)
    {
        // Only Knights can be zombified. This prevents zombification
        // of vampire bats or other weird things like that. Of course,
        // if new monster types are added (and they are to be
        // zombifiable) then this rule might need to change ... (maybe
        // only zombify things which are not of class Zombie and are
        // at height H_WALKING?)
        shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(cr);        
        if (kt && kt->getMap()) {
            DungeonMap *dmap = kt->getMap();
            MapCoord mc = kt->getPos();
            MapDirection facing = kt->getFacing();
            kt->onDeath(Creature::ZOMBIE_MODE);       // this drops items, etc.
            kt->rmFromMap();
            Mediator::instance().getMonsterManager().placeZombie(*dmap, mc, facing);
        }
    }
}

void A_ZombifyActor::execute(const ActionData &ad) const
{
    ZombifyCreature(ad.getActor());
}

A_ZombifyActor::Maker A_ZombifyActor::Maker::register_me;

Action * A_ZombifyActor::Maker::make(ActionPars &pars) const
{
    pars.require(0);
    return new A_ZombifyActor;
}

bool A_ZombifyTarget::possible(const ActionData &ad) const
{
    // Only knights can be zombified
    return dynamic_pointer_cast<Knight>(ad.getVictim());
}

void A_ZombifyTarget::execute(const ActionData &ad) const
{
    ZombifyCreature(ad.getVictim());
}

A_ZombifyTarget::Maker A_ZombifyTarget::Maker::register_me;

Action * A_ZombifyTarget::Maker::make(ActionPars &pars) const
{
    pars.require(0);
    return new A_ZombifyTarget;
}
