/*
 * lockable.hpp
 *
 * Lockable: Base class for openable/trappable/lockable tiles (ie doors and chests).
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef LOCKABLE_HPP
#define LOCKABLE_HPP

#include "tile.hpp"

#include "boost/shared_ptr.hpp"
using namespace boost;

class Creature;
class DungeonMap;
class MapCoord;
class Trap;

class Lockable : public Tile {
    enum { PICK_ONLY_LOCK_NUM=99998, SPECIAL_LOCK_NUM = 99999 };
    
public:
    Lockable() : closed(true), can_trap(true), lock(-1), lock_chance(0), pick_only_chance(0), keymax(1) { }

    void setOpenInitially() { closed = false; }
    void setLockChance(int lock_percent, int pick_only_percent, int kmax)
        { lock_chance = lock_percent; pick_only_chance = pick_only_percent; keymax = kmax; }
    void setCannotTrap() { can_trap = false; }
    void setSpecialLock() { lock = SPECIAL_LOCK_NUM; }

    // Override canActivateHere, onActivate, onOpen and onClose to deal with opening and
    // closing
    virtual bool canActivateHere() const { return false; }
    virtual void onActivate(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature>,
                            ActivateType, bool success = true);
    virtual void onClose(DungeonMap &, const MapCoord &, shared_ptr<Creature>);
    virtual void onOpen(DungeonMap &, const MapCoord &, shared_ptr<Creature>);
    
    bool isClosed() const { return closed; }
    
    // Locks
    bool isLocked() const { return lock > 0; }
    bool isSpecialLocked() const { return lock == SPECIAL_LOCK_NUM; }
    void generateLock(int nkeys);  // If "lock" is negative then this generates a lock.

    // Traps
    // canTrap: true if this Lockable can have traps applied to it
    // setTrap: Set a trap which will be triggered when the door/chest is opened. If a trap
    // is already present, then the existing trap will be set off.
    // generateTrap: like setTrap but called during initial "pretrapped chests" generation.
    // (returns true if a trap was generated.)
    // onHit: Override this to deal with setting off traps when a door/chest is hit.
    bool canTrap() const { return can_trap; }
    void setTrap(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> cr,
                 shared_ptr<Trap> newtrap);
    virtual bool generateTrap(DungeonMap &, const MapCoord &) { return false; }
    virtual void onHit(DungeonMap &, const MapCoord &, shared_ptr<Creature>);

protected:
    // "open", "close" are implemented by subclasses, and handle the actual opening
    // and closing.
    virtual void open(DungeonMap &, const MapCoord &) = 0;
    virtual void close(DungeonMap &, const MapCoord &) = 0; 

private:
    bool doOpen(DungeonMap &, const MapCoord &, shared_ptr<Creature>, ActivateType);
    bool doClose(DungeonMap &, const MapCoord &, shared_ptr<Creature>, ActivateType);
    bool checkUnlock(shared_ptr<Creature>) const;
    void activateTraps(DungeonMap &, const MapCoord &, shared_ptr<Creature>);
    void disarmTraps(DungeonMap &, const MapCoord &, shared_ptr<Creature>);
    
private:
    bool closed;
    bool can_trap;
    int lock; // key number, or zero for unlocked
    // NOTE: a negative value for lock means that lock has not been generated yet.
    // See lock_chance and pick_only_chance for generation parameters.
    // ALSO: if lock is set to SPECIAL_LOCK_NUM then the door cannot be unlocked by keys
    // or lock picks; only switches will work.

    int lock_chance, pick_only_chance, keymax;
    shared_ptr<Trap> trap;
};

#endif
