/*
 * lockable.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "dungeon_map.hpp"
#include "dungeon_view.hpp"
#include "item.hpp"
#include "item_type.hpp"
#include "knight.hpp"
#include "lockable.hpp"
#include "rng.hpp"
#include "trap.hpp"

//
// opening/closing: private functions
//

// returns true if the door was opened, or false if it was locked & we couldn't unlock
bool Lockable::doOpen(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> cr,
                      ActivateType act_type)
{
    if (isLocked() && act_type != ACT_UNLOCK_ALL) {
        bool ok = checkUnlock(cr);
        if (!ok) return false;
    }
    open(dmap, mc);
    if (lock != SPECIAL_LOCK_NUM) lock = 0;  // special lock is always left; but other locks always unlock when opened.
    closed = false;
    if (act_type == ACT_NORMAL) activateTraps(dmap, mc, cr);
    else disarmTraps(dmap, mc, cr);
    return true;
}

bool Lockable::doClose(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> cr,
                       ActivateType act_type)
{
    // Doors can be closed UNLESS they are "special-locked". The latter can
    // only be closed by switches or traps (which we identify by the act_type,
    // which should be ACT_UNLOCK_ALL for switches/traps).
    
    if (lock == SPECIAL_LOCK_NUM && act_type != ACT_UNLOCK_ALL) return false;
    
    close(dmap, mc);
    closed = true;
    return true;
}

bool Lockable::checkUnlock(shared_ptr<Creature> cr) const
{
    if (!isLocked()) return true;
    
    // Only Knights have the ability to unlock doors
    shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(cr);
    if (kt) {
        for (int i=0; i<kt->getBackpackCount(); ++i) {
            int key = kt->getBackpackItem(i).getKey();
            if (key == this->lock) return true;
        }
    }
    return false;
}

//
// opening/closing: public functions
//

void Lockable::onActivate(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> cr,
                          ActivateType act_type, bool success_dummy)
{
    // An "activate" only succeeds if the relevant doOpen/doClose routine says it does
    bool success;
    if (closed) {
        success = doOpen(dmap, mc, cr, act_type);
    } else {
        success = doClose(dmap, mc, cr, act_type);
    }
    Tile::onActivate(dmap, mc, cr, act_type, success);
}

void Lockable::onClose(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> cr)
{
    // An explicit "close" always succeeds
    if (!closed) {
        close(dmap, mc);
        closed = true;
        Tile::onClose(dmap, mc, cr);
    }
}

void Lockable::onOpen(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> cr)
{
    // An explicit "open" always succeeds
    if (closed) {
        doOpen(dmap, mc, cr, ACT_UNLOCK_ALL);
        Tile::onOpen(dmap, mc, cr);
    }
}


//
// lock- and trap-related stuff
//

void Lockable::generateLock(int nkeys)
{
    if (lock < 0) {
        if (nkeys <= 0) {
            lock = 0;    // unlocked
        } else {
            if (g_rng.getBool(float(lock_chance)/100.0f)) {
                if (g_rng.getBool(float(pick_only_chance)/100.0f)) {
                    lock = PICK_ONLY_LOCK_NUM;    // locked, and can be opened by lockpicks only
                } else {
                    lock = g_rng.getInt(1, max(keymax,nkeys)+1); // normal lock
                    if (lock > nkeys) lock = 0; // unlocked (keymax has come into effect)
                }
            } else {
                lock = 0;   // unlocked
            }
        }
    }
}

void Lockable::disarmTraps(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> cr)
{
    if (trap) {
        const ItemType *itype = trap->getTrapItem();
        if (itype) {
            shared_ptr<Item> dummy;
            const bool can_drop = CheckDropSquare(dmap, mc, *itype, dummy);
            if (can_drop) {
                dummy.reset(new Item(*trap->getTrapItem()));
                dmap.addItem(mc, dummy);
            }
        }
        trap = shared_ptr<Trap>();
    }
}

void Lockable::activateTraps(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> cr)
{
    if (trap) {
        trap->spring(dmap, mc, cr);
        trap = shared_ptr<Trap>();
    }
}

void Lockable::setTrap(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> cr,
                       shared_ptr<Trap> newtrap)
{
    if (!canTrap()) return;
    if (trap) {
        activateTraps(dmap, mc, cr);
    }
    trap = newtrap;
}

void Lockable::onHit(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Creature> cr)
{
    if (trap && trap->activateOnHit()) {
        activateTraps(dmap, mc, cr);
    }
    Tile::onHit(dmap, mc, cr);
}
